// REQUIRES: target-is-powerpc64le
// RUN: %clang_builtins %s %librt -o %t && %run %t
#include <stdio.h>
#include "DD.h"

long double __gcc_qadd(long double x, long double y);

double testAccuracy();
int testEdgeCases();

int main(int argc, char *argv[]) {
	if (testEdgeCases())
		return 1;
	
	if (testAccuracy() > 1.0)
		return 1;
	
	return 0;
}

struct testVector {
	double xhi; double xlo;
	double yhi; double ylo;
	double rhi; double rlo;
};

#define INFINITY __builtin_inf()
#define HUGE 0x1.fffffffffffffp1023
#define QNAN __builtin_nan("")

const struct testVector edgeCases[] = {
{	0.0,		0.0,		0.0,		0.0,		0.0,		0.0			},
{  -0.0,		0.0,		0.0,		0.0,		0.0,		0.0			},
{   0.0,		0.0,	   -0.0,		0.0,		0.0,		0.0			},
{  -0.0,		0.0,	   -0.0,		0.0,	   -0.0,		0.0			},

{   INFINITY,	0.0,		0.0,		0.0,		INFINITY,	0.0			},
{  -INFINITY,	0.0,		0.0,		0.0,	   -INFINITY,	0.0			},
{   INFINITY,	0.0,	   -0.0,		0.0,		INFINITY,	0.0			},
{  -INFINITY,	0.0,	   -0.0,		0.0,	   -INFINITY,	0.0			},
{   INFINITY,	0.0,	    1.0,		0.0,		INFINITY,	0.0			},
{  -INFINITY,	0.0,	    1.0,		0.0,	   -INFINITY,	0.0			},
{   INFINITY,	0.0,	   -1.0,		0.0,		INFINITY,	0.0			},
{  -INFINITY,	0.0,	   -1.0,		0.0,	   -INFINITY,	0.0			},
{   INFINITY,	0.0,	    HUGE,		0.0,		INFINITY,	0.0			},
{  -INFINITY,	0.0,	    HUGE,		0.0,	   -INFINITY,	0.0			},
{   INFINITY,	0.0,	   -HUGE,		0.0,	    INFINITY,	0.0			},
{  -INFINITY,	0.0,	   -HUGE,		0.0,	   -INFINITY,	0.0			},
{   INFINITY,	0.0,	    INFINITY,	0.0,		INFINITY,	0.0			},
{  -INFINITY,	0.0,	    INFINITY,	0.0,		QNAN,		0.0			},
{   INFINITY,	0.0,	   -INFINITY,	0.0,		QNAN,		0.0			},
{  -INFINITY,	0.0,	   -INFINITY,	0.0,	   -INFINITY,	0.0			},
{   0.0,		0.0,		INFINITY,	0.0,		INFINITY,	0.0			},
{   0.0,		0.0,	   -INFINITY,	0.0,	   -INFINITY,	0.0			},
{  -0.0,		0.0,		INFINITY,	0.0,		INFINITY,	0.0			},
{  -0.0,		0.0,	   -INFINITY,	0.0,	   -INFINITY,	0.0			},
{   1.0,		0.0,		INFINITY,	0.0,		INFINITY,	0.0			},
{   1.0,		0.0,	   -INFINITY,	0.0,	   -INFINITY,	0.0			},
{  -1.0,		0.0,		INFINITY,	0.0,		INFINITY,	0.0			},
{  -1.0,		0.0,	   -INFINITY,	0.0,	   -INFINITY,	0.0			},
{   HUGE,		0.0,		INFINITY,	0.0,		INFINITY,	0.0			},
{   HUGE,		0.0,	   -INFINITY,	0.0,	   -INFINITY,	0.0			},
{  -HUGE,		0.0,		INFINITY,	0.0,		INFINITY,	0.0			},
{  -HUGE,		0.0,	   -INFINITY,	0.0,	   -INFINITY,	0.0			},

{   QNAN,		0.0,		0.0,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    0.0,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -0.0,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -0.0,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,		1.0,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    1.0,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -1.0,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -1.0,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,		HUGE,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    HUGE,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -HUGE,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -HUGE,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,		INFINITY,	0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    INFINITY,	0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -INFINITY,	0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -INFINITY,	0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	    QNAN,		0.0,	    QNAN,		0.0			},
{   QNAN,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -QNAN,		0.0,	   -QNAN,		0.0,	    QNAN,		0.0			},
{	0.0,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{	0.0,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -0.0,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -0.0,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{	1.0,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{	1.0,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -1.0,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -1.0,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{	HUGE,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{	HUGE,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -HUGE,		0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -HUGE,		0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{	INFINITY,	0.0,		QNAN,		0.0,		QNAN,		0.0			},
{	INFINITY,	0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
{  -INFINITY,	0.0,		QNAN,		0.0,		QNAN,		0.0			},
{  -INFINITY,	0.0,	   -QNAN,		0.0,		QNAN,		0.0			},
};

const int numEdgeCases = sizeof(edgeCases) / sizeof(struct testVector);


int testEdgeCases() {
	int i;
	DD a, b, c, r;
	for (i=0; i<numEdgeCases; ++i) {
		a.hi = edgeCases[i].xhi;
		a.lo = edgeCases[i].xlo;
		b.hi = edgeCases[i].yhi;
		b.lo = edgeCases[i].ylo;
		r.hi = edgeCases[i].rhi;
		r.lo = edgeCases[i].rlo;
		
		int error;
		
		DD c = { .ld = __gcc_qadd(a.ld, b.ld) };
		
		if (r.hi != r.hi) {
			if (c.hi == c.hi)
				error = 1;
			else if (c.lo != 0.0)
				error = 1;
			else
				error = 0;
		}

		else if (r.hi == 0.0)
			error = memcmp(&c, &r, sizeof(DD));
		
		else
			error = ((c.hi != r.hi) || (c.lo != r.lo));
		
		if (error) {
			printf("Error on edge case %a + %a: expected (%a, %a), got (%a, %a).\n", a.hi, b.hi, r.hi, r.lo, c.hi, c.lo);
			return 1;
		}
	}
	
	return 0;
}


/*

 Code for generating the test cases, requires the mpfr package to run.
 
 #include <stdio.h>
 #include <stdlib.h>
 #include <mpfr.h>
 
 #ifdef __x86_64__
 #define randlength 2
 #else
 #define randlength 4
 #endif
 
 
 int main(int argc, char *argv[]) {
 
 MPFR_DECL_INIT(a, 106);
 MPFR_DECL_INIT(b, 106);
 MPFR_DECL_INIT(c, 106);
 
 MPFR_DECL_INIT(tmp, 53);
 
 int exponent_range = atoi(argv[1]);
 
 int i;
 for (i=0; i<128; ++i) {
 mpfr_random2(a, randlength, exponent_range);
 mpfr_random2(b, randlength, exponent_range);
 mpfr_add(c, a, b, GMP_RNDN);
 
 double ahi = mpfr_get_d(a, GMP_RNDN);
 mpfr_set_d(tmp, ahi, GMP_RNDN);
 mpfr_sub(tmp, a, tmp, GMP_RNDN);
 double alo = mpfr_get_d(tmp, GMP_RNDN);
 printf("{%0.13a, %0.13a, ", ahi, alo);
 
 double bhi = mpfr_get_d(b, GMP_RNDN);
 mpfr_set_d(tmp, bhi, GMP_RNDN);
 mpfr_sub(tmp, b, tmp, GMP_RNDN);
 double blo = mpfr_get_d(tmp, GMP_RNDN);
 printf("%0.13a, %0.13a, ", bhi, blo);
 
 double chi = mpfr_get_d(c, GMP_RNDN);
 mpfr_set_d(tmp, chi, GMP_RNDN);
 mpfr_sub(tmp, c, tmp, GMP_RNDN);
 double clo = mpfr_get_d(tmp, GMP_RNDN);
 printf("%0.13a, %0.13a},\n", chi, clo);
 
 mpfr_neg(b, b, GMP_RNDN);
 mpfr_add(c, a, b, GMP_RNDN);
 
 ahi = mpfr_get_d(a, GMP_RNDN);
 mpfr_set_d(tmp, ahi, GMP_RNDN);
 mpfr_sub(tmp, a, tmp, GMP_RNDN);
 alo = mpfr_get_d(tmp, GMP_RNDN);
 printf("{%0.13a, %0.13a, ", ahi, alo);
 
 bhi = mpfr_get_d(b, GMP_RNDN);
 mpfr_set_d(tmp, bhi, GMP_RNDN);
 mpfr_sub(tmp, b, tmp, GMP_RNDN);
 blo = mpfr_get_d(tmp, GMP_RNDN);
 printf("%0.13a, %0.13a, ", bhi, blo);
 
 chi = mpfr_get_d(c, GMP_RNDN);
 mpfr_set_d(tmp, chi, GMP_RNDN);
 mpfr_sub(tmp, c, tmp, GMP_RNDN);
 clo = mpfr_get_d(tmp, GMP_RNDN);
 printf("%0.13a, %0.13a},\n", chi, clo);
 
 mpfr_neg(a, a, GMP_RNDN);
 mpfr_neg(b, b, GMP_RNDN);
 mpfr_add(c, a, b, GMP_RNDN);
 
 ahi = mpfr_get_d(a, GMP_RNDN);
 mpfr_set_d(tmp, ahi, GMP_RNDN);
 mpfr_sub(tmp, a, tmp, GMP_RNDN);
 alo = mpfr_get_d(tmp, GMP_RNDN);
 printf("{%0.13a, %0.13a, ", ahi, alo);
 
 bhi = mpfr_get_d(b, GMP_RNDN);
 mpfr_set_d(tmp, bhi, GMP_RNDN);
 mpfr_sub(tmp, b, tmp, GMP_RNDN);
 blo = mpfr_get_d(tmp, GMP_RNDN);
 printf("%0.13a, %0.13a, ", bhi, blo);
 
 chi = mpfr_get_d(c, GMP_RNDN);
 mpfr_set_d(tmp, chi, GMP_RNDN);
 mpfr_sub(tmp, c, tmp, GMP_RNDN);
 clo = mpfr_get_d(tmp, GMP_RNDN);
 printf("%0.13a, %0.13a},\n", chi, clo);
 
 mpfr_neg(b, b, GMP_RNDN);
 mpfr_add(c, a, b, GMP_RNDN);
 
 ahi = mpfr_get_d(a, GMP_RNDN);
 mpfr_set_d(tmp, ahi, GMP_RNDN);
 mpfr_sub(tmp, a, tmp, GMP_RNDN);
 alo = mpfr_get_d(tmp, GMP_RNDN);
 printf("{%0.13a, %0.13a, ", ahi, alo);
 
 bhi = mpfr_get_d(b, GMP_RNDN);
 mpfr_set_d(tmp, bhi, GMP_RNDN);
 mpfr_sub(tmp, b, tmp, GMP_RNDN);
 blo = mpfr_get_d(tmp, GMP_RNDN);
 printf("%0.13a, %0.13a, ", bhi, blo);
 
 chi = mpfr_get_d(c, GMP_RNDN);
 mpfr_set_d(tmp, chi, GMP_RNDN);
 mpfr_sub(tmp, c, tmp, GMP_RNDN);
 clo = mpfr_get_d(tmp, GMP_RNDN);
 printf("%0.13a, %0.13a},\n", chi, clo);
 }
 
 return 0;
 }


 */

const struct testVector accuracyTests[] = {
// Some test cases that cover the whole exponent range:
{0x1.0000000000002p+366, -0x1.ffc0ffff90000p+302, 0x1.0003ffffffffcp+140, 0x1.ffffc00000000p+53, 0x1.0000000000002p+366, -0x1.ffc0ffff90000p+302},
{0x1.0000000000002p+366, -0x1.ffc0ffff90000p+302, -0x1.0003ffffffffcp+140, -0x1.ffffc00000000p+53, 0x1.0000000000002p+366, -0x1.ffc0ffff90000p+302},
{-0x1.0000000000002p+366, 0x1.ffc0ffff90000p+302, 0x1.0003ffffffffcp+140, 0x1.ffffc00000000p+53, -0x1.0000000000002p+366, 0x1.ffc0ffff90000p+302},
{-0x1.0000000000002p+366, 0x1.ffc0ffff90000p+302, -0x1.0003ffffffffcp+140, -0x1.ffffc00000000p+53, -0x1.0000000000002p+366, 0x1.ffc0ffff90000p+302},
{0x1.0000100000000p-746, -0x1.0000000000000p-808, 0x1.0000000004000p-636, -0x1.ffffe00fc0200p-692, 0x1.0000000004000p-636, -0x1.ffffe00fc0200p-692},
{0x1.0000100000000p-746, -0x1.0000000000000p-808, -0x1.0000000004000p-636, 0x1.ffffe00fc0200p-692, -0x1.0000000004000p-636, 0x1.ffffe00fc0200p-692},
{-0x1.0000100000000p-746, 0x1.0000000000000p-808, 0x1.0000000004000p-636, -0x1.ffffe00fc0200p-692, 0x1.0000000004000p-636, -0x1.ffffe00fc0200p-692},
{-0x1.0000100000000p-746, 0x1.0000000000000p-808, -0x1.0000000004000p-636, 0x1.ffffe00fc0200p-692, -0x1.0000000004000p-636, 0x1.ffffe00fc0200p-692},
{0x1.0000000800000p+426, -0x1.0000000000000p+354, 0x1.000003fffc002p-730, -0x1.0000000000000p-835, 0x1.0000000800000p+426, -0x1.0000000000000p+354},
{0x1.0000000800000p+426, -0x1.0000000000000p+354, -0x1.000003fffc002p-730, 0x1.0000000000000p-835, 0x1.0000000800000p+426, -0x1.0000000000000p+354},
{-0x1.0000000800000p+426, 0x1.0000000000000p+354, 0x1.000003fffc002p-730, -0x1.0000000000000p-835, -0x1.0000000800000p+426, 0x1.0000000000000p+354},
{-0x1.0000000800000p+426, 0x1.0000000000000p+354, -0x1.000003fffc002p-730, 0x1.0000000000000p-835, -0x1.0000000800000p+426, 0x1.0000000000000p+354},
{0x1.0000000000000p-262, 0x1.ffc001ffffe00p-317, 0x1.000003fff8000p-258, 0x1.fffc000fc0000p-329, 0x1.100003fff8000p-258, 0x1.ffe001c000dc0p-317},
{0x1.0000000000000p-262, 0x1.ffc001ffffe00p-317, -0x1.000003fff8000p-258, -0x1.fffc000fc0000p-329, -0x1.e00007fff0000p-259, 0x1.ffa0023ffee40p-317},
{-0x1.0000000000000p-262, -0x1.ffc001ffffe00p-317, 0x1.000003fff8000p-258, 0x1.fffc000fc0000p-329, 0x1.e00007fff0000p-259, -0x1.ffa0023ffee40p-317},
{-0x1.0000000000000p-262, -0x1.ffc001ffffe00p-317, -0x1.000003fff8000p-258, -0x1.fffc000fc0000p-329, -0x1.100003fff8000p-258, -0x1.ffe001c000dc0p-317},
{0x1.0000000000003p+980, 0x1.fc00001fff800p+916, 0x1.0000000000020p+626, -0x1.fc01ffe002000p+562, 0x1.0000000000003p+980, 0x1.fc00001fff800p+916},
{0x1.0000000000003p+980, 0x1.fc00001fff800p+916, -0x1.0000000000020p+626, 0x1.fc01ffe002000p+562, 0x1.0000000000003p+980, 0x1.fc00001fff800p+916},
{-0x1.0000000000003p+980, -0x1.fc00001fff800p+916, 0x1.0000000000020p+626, -0x1.fc01ffe002000p+562, -0x1.0000000000003p+980, -0x1.fc00001fff800p+916},
{-0x1.0000000000003p+980, -0x1.fc00001fff800p+916, -0x1.0000000000020p+626, 0x1.fc01ffe002000p+562, -0x1.0000000000003p+980, -0x1.fc00001fff800p+916},
{0x1.0000000000000p-800, 0x1.fffff00001ffep-854, 0x1.0000000000000p+470, 0x1.ff00000ffff80p+410, 0x1.0000000000000p+470, 0x1.ff00000ffff80p+410},
{0x1.0000000000000p-800, 0x1.fffff00001ffep-854, -0x1.0000000000000p+470, -0x1.ff00000ffff80p+410, -0x1.0000000000000p+470, -0x1.ff00000ffff80p+410},
{-0x1.0000000000000p-800, -0x1.fffff00001ffep-854, 0x1.0000000000000p+470, 0x1.ff00000ffff80p+410, 0x1.0000000000000p+470, 0x1.ff00000ffff80p+410},
{-0x1.0000000000000p-800, -0x1.fffff00001ffep-854, -0x1.0000000000000p+470, -0x1.ff00000ffff80p+410, -0x1.0000000000000p+470, -0x1.ff00000ffff80p+410},
{0x1.0000000000000p+925, 0x1.ffffffffffc00p+862, 0x1.0000200000000p-454, -0x1.ffc0000080000p-526, 0x1.0000000000000p+925, 0x1.ffffffffffc00p+862},
{0x1.0000000000000p+925, 0x1.ffffffffffc00p+862, -0x1.0000200000000p-454, 0x1.ffc0000080000p-526, 0x1.0000000000000p+925, 0x1.ffffffffffc00p+862},
{-0x1.0000000000000p+925, -0x1.ffffffffffc00p+862, 0x1.0000200000000p-454, -0x1.ffc0000080000p-526, -0x1.0000000000000p+925, -0x1.ffffffffffc00p+862},
{-0x1.0000000000000p+925, -0x1.ffffffffffc00p+862, -0x1.0000200000000p-454, 0x1.ffc0000080000p-526, -0x1.0000000000000p+925, -0x1.ffffffffffc00p+862},
{0x1.0000000000800p+265, -0x1.fffc000000010p+208, 0x1.0000000000008p-204, -0x1.ffe03fffff010p-261, 0x1.0000000000800p+265, -0x1.fffc000000010p+208},
{0x1.0000000000800p+265, -0x1.fffc000000010p+208, -0x1.0000000000008p-204, 0x1.ffe03fffff010p-261, 0x1.0000000000800p+265, -0x1.fffc000000010p+208},
{-0x1.0000000000800p+265, 0x1.fffc000000010p+208, 0x1.0000000000008p-204, -0x1.ffe03fffff010p-261, -0x1.0000000000800p+265, 0x1.fffc000000010p+208},
{-0x1.0000000000800p+265, 0x1.fffc000000010p+208, -0x1.0000000000008p-204, 0x1.ffe03fffff010p-261, -0x1.0000000000800p+265, 0x1.fffc000000010p+208},
{0x1.0000000000000p-690, 0x1.fffef0ffff000p-753, 0x1.0000100000000p+763, -0x1.fff8000000800p+699, 0x1.0000100000000p+763, -0x1.fff8000000800p+699},
{0x1.0000000000000p-690, 0x1.fffef0ffff000p-753, -0x1.0000100000000p+763, 0x1.fff8000000800p+699, -0x1.0000100000000p+763, 0x1.fff8000000800p+699},
{-0x1.0000000000000p-690, -0x1.fffef0ffff000p-753, 0x1.0000100000000p+763, -0x1.fff8000000800p+699, 0x1.0000100000000p+763, -0x1.fff8000000800p+699},
{-0x1.0000000000000p-690, -0x1.fffef0ffff000p-753, -0x1.0000100000000p+763, 0x1.fff8000000800p+699, -0x1.0000100000000p+763, 0x1.fff8000000800p+699},
{0x1.07c00007ffff0p-506, 0x1.fffc00f800000p-582, 0x1.0000007ffffe0p-302, 0x1.f000000000000p-373, 0x1.0000007ffffe0p-302, 0x1.f000000000000p-373},
{0x1.07c00007ffff0p-506, 0x1.fffc00f800000p-582, -0x1.0000007ffffe0p-302, -0x1.f000000000000p-373, -0x1.0000007ffffe0p-302, -0x1.f000000000000p-373},
{-0x1.07c00007ffff0p-506, -0x1.fffc00f800000p-582, 0x1.0000007ffffe0p-302, 0x1.f000000000000p-373, 0x1.0000007ffffe0p-302, 0x1.f000000000000p-373},
{-0x1.07c00007ffff0p-506, -0x1.fffc00f800000p-582, -0x1.0000007ffffe0p-302, -0x1.f000000000000p-373, -0x1.0000007ffffe0p-302, -0x1.f000000000000p-373},
{0x1.0000000000000p-958, 0x1.ff800fffff800p-1014, 0x1.0000000000000p+373, 0x1.fffc0000ffff8p+317, 0x1.0000000000000p+373, 0x1.fffc0000ffff8p+317},
{0x1.0000000000000p-958, 0x1.ff800fffff800p-1014, -0x1.0000000000000p+373, -0x1.fffc0000ffff8p+317, -0x1.0000000000000p+373, -0x1.fffc0000ffff8p+317},
{-0x1.0000000000000p-958, -0x1.ff800fffff800p-1014, 0x1.0000000000000p+373, 0x1.fffc0000ffff8p+317, 0x1.0000000000000p+373, 0x1.fffc0000ffff8p+317},
{-0x1.0000000000000p-958, -0x1.ff800fffff800p-1014, -0x1.0000000000000p+373, -0x1.fffc0000ffff8p+317, -0x1.0000000000000p+373, -0x1.fffc0000ffff8p+317},
{0x1.0000000008000p+114, -0x1.ffff807ffe002p+60, 0x1.0000000800000p-783, -0x1.ffffff8000008p-839, 0x1.0000000008000p+114, -0x1.ffff807ffe002p+60},
{0x1.0000000008000p+114, -0x1.ffff807ffe002p+60, -0x1.0000000800000p-783, 0x1.ffffff8000008p-839, 0x1.0000000008000p+114, -0x1.ffff807ffe002p+60},
{-0x1.0000000008000p+114, 0x1.ffff807ffe002p+60, 0x1.0000000800000p-783, -0x1.ffffff8000008p-839, -0x1.0000000008000p+114, 0x1.ffff807ffe002p+60},
{-0x1.0000000008000p+114, 0x1.ffff807ffe002p+60, -0x1.0000000800000p-783, 0x1.ffffff8000008p-839, -0x1.0000000008000p+114, 0x1.ffff807ffe002p+60},
{0x1.0000000000020p+340, -0x1.0000000000000p+257, 0x1.0000000000000p-442, 0x1.fffff000003c0p-501, 0x1.0000000000020p+340, -0x1.0000000000000p+257},
{0x1.0000000000020p+340, -0x1.0000000000000p+257, -0x1.0000000000000p-442, -0x1.fffff000003c0p-501, 0x1.0000000000020p+340, -0x1.0000000000000p+257},
{-0x1.0000000000020p+340, 0x1.0000000000000p+257, 0x1.0000000000000p-442, 0x1.fffff000003c0p-501, -0x1.0000000000020p+340, 0x1.0000000000000p+257},
{-0x1.0000000000020p+340, 0x1.0000000000000p+257, -0x1.0000000000000p-442, -0x1.fffff000003c0p-501, -0x1.0000000000020p+340, 0x1.0000000000000p+257},
{0x1.0000001ffff80p+385, 0x1.fffffdfffff8ep+331, 0x1.0000000200000p+69, -0x1.0000000000000p+0, 0x1.0000001ffff80p+385, 0x1.fffffdfffff8ep+331},
{0x1.0000001ffff80p+385, 0x1.fffffdfffff8ep+331, -0x1.0000000200000p+69, 0x1.0000000000000p+0, 0x1.0000001ffff80p+385, 0x1.fffffdfffff8ep+331},
{-0x1.0000001ffff80p+385, -0x1.fffffdfffff8ep+331, 0x1.0000000200000p+69, -0x1.0000000000000p+0, -0x1.0000001ffff80p+385, -0x1.fffffdfffff8ep+331},
{-0x1.0000001ffff80p+385, -0x1.fffffdfffff8ep+331, -0x1.0000000200000p+69, 0x1.0000000000000p+0, -0x1.0000001ffff80p+385, -0x1.fffffdfffff8ep+331},
{0x1.01ffffff80400p-140, -0x1.c7ffffc000100p-198, 0x1.0000000000000p-934, 0x1.8fff01fffff00p-994, 0x1.01ffffff80400p-140, -0x1.c7ffffc000100p-198},
{0x1.01ffffff80400p-140, -0x1.c7ffffc000100p-198, -0x1.0000000000000p-934, -0x1.8fff01fffff00p-994, 0x1.01ffffff80400p-140, -0x1.c7ffffc000100p-198},
{-0x1.01ffffff80400p-140, 0x1.c7ffffc000100p-198, 0x1.0000000000000p-934, 0x1.8fff01fffff00p-994, -0x1.01ffffff80400p-140, 0x1.c7ffffc000100p-198},
{-0x1.01ffffff80400p-140, 0x1.c7ffffc000100p-198, -0x1.0000000000000p-934, -0x1.8fff01fffff00p-994, -0x1.01ffffff80400p-140, 0x1.c7ffffc000100p-198},
{0x1.0000000000008p+975, -0x1.ffffc00800000p+902, 0x1.00003ffe00002p-979, -0x1.ffffffc180000p-1039, 0x1.0000000000008p+975, -0x1.ffffc00800000p+902},
{0x1.0000000000008p+975, -0x1.ffffc00800000p+902, -0x1.00003ffe00002p-979, 0x1.ffffffc180000p-1039, 0x1.0000000000008p+975, -0x1.ffffc00800000p+902},
{-0x1.0000000000008p+975, 0x1.ffffc00800000p+902, 0x1.00003ffe00002p-979, -0x1.ffffffc180000p-1039, -0x1.0000000000008p+975, 0x1.ffffc00800000p+902},
{-0x1.0000000000008p+975, 0x1.ffffc00800000p+902, -0x1.00003ffe00002p-979, 0x1.ffffffc180000p-1039, -0x1.0000000000008p+975, 0x1.ffffc00800000p+902},
{0x1.000000ffffff8p+56, 0x1.ffffffc000000p-1, 0x1.0000000000000p-40, 0x1.ffffffe000000p-98, 0x1.000000ffffff8p+56, 0x1.ffffffc002000p-1},
{0x1.000000ffffff8p+56, 0x1.ffffffc000000p-1, -0x1.0000000000000p-40, -0x1.ffffffe000000p-98, 0x1.000000ffffff8p+56, 0x1.ffffffbffe000p-1},
{-0x1.000000ffffff8p+56, -0x1.ffffffc000000p-1, 0x1.0000000000000p-40, 0x1.ffffffe000000p-98, -0x1.000000ffffff8p+56, -0x1.ffffffbffe000p-1},
{-0x1.000000ffffff8p+56, -0x1.ffffffc000000p-1, -0x1.0000000000000p-40, -0x1.ffffffe000000p-98, -0x1.000000ffffff8p+56, -0x1.ffffffc002000p-1},
{0x1.0000000020000p-764, -0x1.0000000000000p-869, 0x1.0000000000400p-403, -0x1.ff1ffffff4000p-470, 0x1.0000000000400p-403, -0x1.ff1ffffff4000p-470},
{0x1.0000000020000p-764, -0x1.0000000000000p-869, -0x1.0000000000400p-403, 0x1.ff1ffffff4000p-470, -0x1.0000000000400p-403, 0x1.ff1ffffff4000p-470},
{-0x1.0000000020000p-764, 0x1.0000000000000p-869, 0x1.0000000000400p-403, -0x1.ff1ffffff4000p-470, 0x1.0000000000400p-403, -0x1.ff1ffffff4000p-470},
{-0x1.0000000020000p-764, 0x1.0000000000000p-869, -0x1.0000000000400p-403, 0x1.ff1ffffff4000p-470, -0x1.0000000000400p-403, 0x1.ff1ffffff4000p-470},
{0x1.0000ff8020000p-469, -0x1.fffffa0000000p-531, 0x1.0000000001ff0p-14, 0x1.f700000ffe000p-80, 0x1.0000000001ff0p-14, 0x1.f700000ffe000p-80},
{0x1.0000ff8020000p-469, -0x1.fffffa0000000p-531, -0x1.0000000001ff0p-14, -0x1.f700000ffe000p-80, -0x1.0000000001ff0p-14, -0x1.f700000ffe000p-80},
{-0x1.0000ff8020000p-469, 0x1.fffffa0000000p-531, 0x1.0000000001ff0p-14, 0x1.f700000ffe000p-80, 0x1.0000000001ff0p-14, 0x1.f700000ffe000p-80},
{-0x1.0000ff8020000p-469, 0x1.fffffa0000000p-531, -0x1.0000000001ff0p-14, -0x1.f700000ffe000p-80, -0x1.0000000001ff0p-14, -0x1.f700000ffe000p-80},
{0x1.0000020000000p+518, -0x1.0000000000000p+454, 0x1.003e00001f800p+772, 0x1.ffde000001e00p+710, 0x1.003e00001f800p+772, 0x1.ffde000001e00p+710},
{0x1.0000020000000p+518, -0x1.0000000000000p+454, -0x1.003e00001f800p+772, -0x1.ffde000001e00p+710, -0x1.003e00001f800p+772, -0x1.ffde000001e00p+710},
{-0x1.0000020000000p+518, 0x1.0000000000000p+454, 0x1.003e00001f800p+772, 0x1.ffde000001e00p+710, 0x1.003e00001f800p+772, 0x1.ffde000001e00p+710},
{-0x1.0000020000000p+518, 0x1.0000000000000p+454, -0x1.003e00001f800p+772, -0x1.ffde000001e00p+710, -0x1.003e00001f800p+772, -0x1.ffde000001e00p+710},
{0x1.0007e3ffe0000p-546, 0x1.fe0ffffffe000p-610, 0x1.0000000000000p+567, 0x1.fffffe07fff00p+506, 0x1.0000000000000p+567, 0x1.fffffe07fff00p+506},
{0x1.0007e3ffe0000p-546, 0x1.fe0ffffffe000p-610, -0x1.0000000000000p+567, -0x1.fffffe07fff00p+506, -0x1.0000000000000p+567, -0x1.fffffe07fff00p+506},
{-0x1.0007e3ffe0000p-546, -0x1.fe0ffffffe000p-610, 0x1.0000000000000p+567, 0x1.fffffe07fff00p+506, 0x1.0000000000000p+567, 0x1.fffffe07fff00p+506},
{-0x1.0007e3ffe0000p-546, -0x1.fe0ffffffe000p-610, -0x1.0000000000000p+567, -0x1.fffffe07fff00p+506, -0x1.0000000000000p+567, -0x1.fffffe07fff00p+506},
{0x1.00000000fe000p-853, 0x1.ff00004000000p-920, 0x1.000003fffff80p+382, 0x1.fff8400000000p+322, 0x1.000003fffff80p+382, 0x1.fff8400000000p+322},
{0x1.00000000fe000p-853, 0x1.ff00004000000p-920, -0x1.000003fffff80p+382, -0x1.fff8400000000p+322, -0x1.000003fffff80p+382, -0x1.fff8400000000p+322},
{-0x1.00000000fe000p-853, -0x1.ff00004000000p-920, 0x1.000003fffff80p+382, 0x1.fff8400000000p+322, 0x1.000003fffff80p+382, 0x1.fff8400000000p+322},
{-0x1.00000000fe000p-853, -0x1.ff00004000000p-920, -0x1.000003fffff80p+382, -0x1.fff8400000000p+322, -0x1.000003fffff80p+382, -0x1.fff8400000000p+322},
{0x1.0000800000000p-529, -0x1.0000000000000p-585, 0x1.0000000000000p-716, 0x1.fffffffc1fc00p-779, 0x1.0000800000000p-529, -0x1.0000000000000p-585},
{0x1.0000800000000p-529, -0x1.0000000000000p-585, -0x1.0000000000000p-716, -0x1.fffffffc1fc00p-779, 0x1.0000800000000p-529, -0x1.0000000000000p-585},
{-0x1.0000800000000p-529, 0x1.0000000000000p-585, 0x1.0000000000000p-716, 0x1.fffffffc1fc00p-779, -0x1.0000800000000p-529, 0x1.0000000000000p-585},
{-0x1.0000800000000p-529, 0x1.0000000000000p-585, -0x1.0000000000000p-716, -0x1.fffffffc1fc00p-779, -0x1.0000800000000p-529, 0x1.0000000000000p-585},
{0x1.0000000000000p+545, 0x1.ff000ff3ffe00p+491, 0x1.0000000000000p-565, 0x1.fe0003fffc000p-624, 0x1.0000000000000p+545, 0x1.ff000ff3ffe00p+491},
{0x1.0000000000000p+545, 0x1.ff000ff3ffe00p+491, -0x1.0000000000000p-565, -0x1.fe0003fffc000p-624, 0x1.0000000000000p+545, 0x1.ff000ff3ffe00p+491},
{-0x1.0000000000000p+545, -0x1.ff000ff3ffe00p+491, 0x1.0000000000000p-565, 0x1.fe0003fffc000p-624, -0x1.0000000000000p+545, -0x1.ff000ff3ffe00p+491},
{-0x1.0000000000000p+545, -0x1.ff000ff3ffe00p+491, -0x1.0000000000000p-565, -0x1.fe0003fffc000p-624, -0x1.0000000000000p+545, -0x1.ff000ff3ffe00p+491},
{0x1.000000e000010p-739, -0x1.ffffffc020000p-809, 0x1.00000001f0000p+66, 0x1.8000000000000p-38, 0x1.00000001f0000p+66, 0x1.8000000000000p-38},
{0x1.000000e000010p-739, -0x1.ffffffc020000p-809, -0x1.00000001f0000p+66, -0x1.8000000000000p-38, -0x1.00000001f0000p+66, -0x1.8000000000000p-38},
{-0x1.000000e000010p-739, 0x1.ffffffc020000p-809, 0x1.00000001f0000p+66, 0x1.8000000000000p-38, 0x1.00000001f0000p+66, 0x1.8000000000000p-38},
{-0x1.000000e000010p-739, 0x1.ffffffc020000p-809, -0x1.00000001f0000p+66, -0x1.8000000000000p-38, -0x1.00000001f0000p+66, -0x1.8000000000000p-38},
{0x1.000003ff00000p-980, 0x1.8000000000000p-1057, 0x1.0001ffffe0000p-313, 0x1.f800000100000p-386, 0x1.0001ffffe0000p-313, 0x1.f800000100000p-386},
{0x1.000003ff00000p-980, 0x1.8000000000000p-1057, -0x1.0001ffffe0000p-313, -0x1.f800000100000p-386, -0x1.0001ffffe0000p-313, -0x1.f800000100000p-386},
{-0x1.000003ff00000p-980, -0x1.8000000000000p-1057, 0x1.0001ffffe0000p-313, 0x1.f800000100000p-386, 0x1.0001ffffe0000p-313, 0x1.f800000100000p-386},
{-0x1.000003ff00000p-980, -0x1.8000000000000p-1057, -0x1.0001ffffe0000p-313, -0x1.f800000100000p-386, -0x1.0001ffffe0000p-313, -0x1.f800000100000p-386},
{0x1.0000000000200p+292, -0x1.ffffffffffc08p+236, 0x1.0000000000000p-660, 0x1.ffffffffffffep-714, 0x1.0000000000200p+292, -0x1.ffffffffffc08p+236},
{0x1.0000000000200p+292, -0x1.ffffffffffc08p+236, -0x1.0000000000000p-660, -0x1.ffffffffffffep-714, 0x1.0000000000200p+292, -0x1.ffffffffffc08p+236},
{-0x1.0000000000200p+292, 0x1.ffffffffffc08p+236, 0x1.0000000000000p-660, 0x1.ffffffffffffep-714, -0x1.0000000000200p+292, 0x1.ffffffffffc08p+236},
{-0x1.0000000000200p+292, 0x1.ffffffffffc08p+236, -0x1.0000000000000p-660, -0x1.ffffffffffffep-714, -0x1.0000000000200p+292, 0x1.ffffffffffc08p+236},
{0x1.0000000040000p-475, -0x1.fffff80002000p-541, 0x1.0000000000000p-741, 0x1.ffc01ffff3c00p-804, 0x1.0000000040000p-475, -0x1.fffff80002000p-541},
{0x1.0000000040000p-475, -0x1.fffff80002000p-541, -0x1.0000000000000p-741, -0x1.ffc01ffff3c00p-804, 0x1.0000000040000p-475, -0x1.fffff80002000p-541},
{-0x1.0000000040000p-475, 0x1.fffff80002000p-541, 0x1.0000000000000p-741, 0x1.ffc01ffff3c00p-804, -0x1.0000000040000p-475, 0x1.fffff80002000p-541},
{-0x1.0000000040000p-475, 0x1.fffff80002000p-541, -0x1.0000000000000p-741, -0x1.ffc01ffff3c00p-804, -0x1.0000000040000p-475, 0x1.fffff80002000p-541},
{0x1.0000800000000p+1022, -0x1.0000000000000p+917, 0x1.0000003800001p-290, 0x1.fffffe000003ep-344, 0x1.0000800000000p+1022, -0x1.0000000000000p+917},
{0x1.0000800000000p+1022, -0x1.0000000000000p+917, -0x1.0000003800001p-290, -0x1.fffffe000003ep-344, 0x1.0000800000000p+1022, -0x1.0000000000000p+917},
{-0x1.0000800000000p+1022, 0x1.0000000000000p+917, 0x1.0000003800001p-290, 0x1.fffffe000003ep-344, -0x1.0000800000000p+1022, 0x1.0000000000000p+917},
{-0x1.0000800000000p+1022, 0x1.0000000000000p+917, -0x1.0000003800001p-290, -0x1.fffffe000003ep-344, -0x1.0000800000000p+1022, 0x1.0000000000000p+917},
{0x1.0f00000020000p-425, -0x1.fffe000000100p-479, 0x1.0000000400000p-527, -0x1.ffffffc004000p-594, 0x1.0f00000020000p-425, -0x1.fffe0000000f0p-479},
{0x1.0f00000020000p-425, -0x1.fffe000000100p-479, -0x1.0000000400000p-527, 0x1.ffffffc004000p-594, 0x1.0f00000020000p-425, -0x1.fffe000000110p-479},
{-0x1.0f00000020000p-425, 0x1.fffe000000100p-479, 0x1.0000000400000p-527, -0x1.ffffffc004000p-594, -0x1.0f00000020000p-425, 0x1.fffe000000110p-479},
{-0x1.0f00000020000p-425, 0x1.fffe000000100p-479, -0x1.0000000400000p-527, 0x1.ffffffc004000p-594, -0x1.0f00000020000p-425, 0x1.fffe0000000f0p-479},
{0x1.003fe00020000p+720, -0x1.ffffffffe8000p+652, 0x1.0000000000400p+142, -0x1.ffff800007ffcp+87, 0x1.003fe00020000p+720, -0x1.ffffffffe8000p+652},
{0x1.003fe00020000p+720, -0x1.ffffffffe8000p+652, -0x1.0000000000400p+142, 0x1.ffff800007ffcp+87, 0x1.003fe00020000p+720, -0x1.ffffffffe8000p+652},
{-0x1.003fe00020000p+720, 0x1.ffffffffe8000p+652, 0x1.0000000000400p+142, -0x1.ffff800007ffcp+87, -0x1.003fe00020000p+720, 0x1.ffffffffe8000p+652},
{-0x1.003fe00020000p+720, 0x1.ffffffffe8000p+652, -0x1.0000000000400p+142, 0x1.ffff800007ffcp+87, -0x1.003fe00020000p+720, 0x1.ffffffffe8000p+652},
{0x1.ffff00fffffc4p+71, -0x1.8080000000000p-11, 0x1.0000020000000p-926, -0x1.0000000000000p-1002, 0x1.ffff00fffffc4p+71, -0x1.8080000000000p-11},
{0x1.ffff00fffffc4p+71, -0x1.8080000000000p-11, -0x1.0000020000000p-926, 0x1.0000000000000p-1002, 0x1.ffff00fffffc4p+71, -0x1.8080000000000p-11},
{-0x1.ffff00fffffc4p+71, 0x1.8080000000000p-11, 0x1.0000020000000p-926, -0x1.0000000000000p-1002, -0x1.ffff00fffffc4p+71, 0x1.8080000000000p-11},
{-0x1.ffff00fffffc4p+71, 0x1.8080000000000p-11, -0x1.0000020000000p-926, 0x1.0000000000000p-1002, -0x1.ffff00fffffc4p+71, 0x1.8080000000000p-11},
{0x1.0000000000040p-915, -0x1.8000000800000p-990, 0x1.0000000002000p-454, -0x1.0000000000000p-558, 0x1.0000000002000p-454, -0x1.0000000000000p-558},
{0x1.0000000000040p-915, -0x1.8000000800000p-990, -0x1.0000000002000p-454, 0x1.0000000000000p-558, -0x1.0000000002000p-454, 0x1.0000000000000p-558},
{-0x1.0000000000040p-915, 0x1.8000000800000p-990, 0x1.0000000002000p-454, -0x1.0000000000000p-558, 0x1.0000000002000p-454, -0x1.0000000000000p-558},
{-0x1.0000000000040p-915, 0x1.8000000800000p-990, -0x1.0000000002000p-454, 0x1.0000000000000p-558, -0x1.0000000002000p-454, 0x1.0000000000000p-558},
{0x1.0000000800000p-123, -0x1.ffffc02000000p-201, 0x1.7fffe00800000p+593, -0x1.0000000000000p+488, 0x1.7fffe00800000p+593, -0x1.0000000000000p+488},
{0x1.0000000800000p-123, -0x1.ffffc02000000p-201, -0x1.7fffe00800000p+593, 0x1.0000000000000p+488, -0x1.7fffe00800000p+593, 0x1.0000000000000p+488},
{-0x1.0000000800000p-123, 0x1.ffffc02000000p-201, 0x1.7fffe00800000p+593, -0x1.0000000000000p+488, 0x1.7fffe00800000p+593, -0x1.0000000000000p+488},
{-0x1.0000000800000p-123, 0x1.ffffc02000000p-201, -0x1.7fffe00800000p+593, 0x1.0000000000000p+488, -0x1.7fffe00800000p+593, 0x1.0000000000000p+488},
{0x1.0000400000000p+352, -0x1.0000000000000p+261, 0x1.4000000000000p-216, -0x1.ffffffff00020p-274, 0x1.0000400000000p+352, -0x1.0000000000000p+261},
{0x1.0000400000000p+352, -0x1.0000000000000p+261, -0x1.4000000000000p-216, 0x1.ffffffff00020p-274, 0x1.0000400000000p+352, -0x1.0000000000000p+261},
{-0x1.0000400000000p+352, 0x1.0000000000000p+261, 0x1.4000000000000p-216, -0x1.ffffffff00020p-274, -0x1.0000400000000p+352, 0x1.0000000000000p+261},
{-0x1.0000400000000p+352, 0x1.0000000000000p+261, -0x1.4000000000000p-216, 0x1.ffffffff00020p-274, -0x1.0000400000000p+352, 0x1.0000000000000p+261},
{0x1.0000000000000p-714, 0x1.03eff7ff3fff0p-771, 0x1.00001fffc0000p-889, 0x1.f80001f000f80p-949, 0x1.0000000000000p-714, 0x1.03eff7ff3fff0p-771},
{0x1.0000000000000p-714, 0x1.03eff7ff3fff0p-771, -0x1.00001fffc0000p-889, -0x1.f80001f000f80p-949, 0x1.0000000000000p-714, 0x1.03eff7ff3fff0p-771},
{-0x1.0000000000000p-714, -0x1.03eff7ff3fff0p-771, 0x1.00001fffc0000p-889, 0x1.f80001f000f80p-949, -0x1.0000000000000p-714, -0x1.03eff7ff3fff0p-771},
{-0x1.0000000000000p-714, -0x1.03eff7ff3fff0p-771, -0x1.00001fffc0000p-889, -0x1.f80001f000f80p-949, -0x1.0000000000000p-714, -0x1.03eff7ff3fff0p-771},
{0x1.03ffffffffffep-517, 0x1.8000000000000p-621, 0x1.0000040000000p-478, -0x1.0000000000000p-583, 0x1.0000040002080p-478, -0x1.0002000000000p-568},
{0x1.03ffffffffffep-517, 0x1.8000000000000p-621, -0x1.0000040000000p-478, 0x1.0000000000000p-583, -0x1.000003fffdf80p-478, -0x1.fffc000000000p-569},
{-0x1.03ffffffffffep-517, -0x1.8000000000000p-621, 0x1.0000040000000p-478, -0x1.0000000000000p-583, 0x1.000003fffdf80p-478, 0x1.fffc000000000p-569},
{-0x1.03ffffffffffep-517, -0x1.8000000000000p-621, -0x1.0000040000000p-478, 0x1.0000000000000p-583, -0x1.0000040002080p-478, 0x1.0002000000000p-568},
{0x1.0007ffffcfffcp-566, 0x1.ff01fffe00000p-640, 0x1.01fff08000000p-12, -0x1.fffffe0000020p-66, 0x1.01fff08000000p-12, -0x1.fffffe0000020p-66},
{0x1.0007ffffcfffcp-566, 0x1.ff01fffe00000p-640, -0x1.01fff08000000p-12, 0x1.fffffe0000020p-66, -0x1.01fff08000000p-12, 0x1.fffffe0000020p-66},
{-0x1.0007ffffcfffcp-566, -0x1.ff01fffe00000p-640, 0x1.01fff08000000p-12, -0x1.fffffe0000020p-66, 0x1.01fff08000000p-12, -0x1.fffffe0000020p-66},
{-0x1.0007ffffcfffcp-566, -0x1.ff01fffe00000p-640, -0x1.01fff08000000p-12, 0x1.fffffe0000020p-66, -0x1.01fff08000000p-12, 0x1.fffffe0000020p-66},
{0x1.00000000007fep-907, 0x1.ff00003ffff00p-968, 0x1.004000e000008p+829, -0x1.ffffe02000000p+751, 0x1.004000e000008p+829, -0x1.ffffe02000000p+751},
{0x1.00000000007fep-907, 0x1.ff00003ffff00p-968, -0x1.004000e000008p+829, 0x1.ffffe02000000p+751, -0x1.004000e000008p+829, 0x1.ffffe02000000p+751},
{-0x1.00000000007fep-907, -0x1.ff00003ffff00p-968, 0x1.004000e000008p+829, -0x1.ffffe02000000p+751, 0x1.004000e000008p+829, -0x1.ffffe02000000p+751},
{-0x1.00000000007fep-907, -0x1.ff00003ffff00p-968, -0x1.004000e000008p+829, 0x1.ffffe02000000p+751, -0x1.004000e000008p+829, 0x1.ffffe02000000p+751},
{0x1.00000003ff000p-540, 0x1.ffff000000000p-629, 0x1.0000fffff8000p-864, 0x1.f003ffe0003e0p-922, 0x1.00000003ff000p-540, 0x1.ffff000000000p-629},
{0x1.00000003ff000p-540, 0x1.ffff000000000p-629, -0x1.0000fffff8000p-864, -0x1.f003ffe0003e0p-922, 0x1.00000003ff000p-540, 0x1.ffff000000000p-629},
{-0x1.00000003ff000p-540, -0x1.ffff000000000p-629, 0x1.0000fffff8000p-864, 0x1.f003ffe0003e0p-922, -0x1.00000003ff000p-540, -0x1.ffff000000000p-629},
{-0x1.00000003ff000p-540, -0x1.ffff000000000p-629, -0x1.0000fffff8000p-864, -0x1.f003ffe0003e0p-922, -0x1.00000003ff000p-540, -0x1.ffff000000000p-629},
{0x1.0000000000000p+14, 0x1.ffe001c0003c0p-45, 0x1.01fffffe00000p-954, 0x1.ffffffffffc00p-1012, 0x1.0000000000000p+14, 0x1.ffe001c0003c0p-45},
{0x1.0000000000000p+14, 0x1.ffe001c0003c0p-45, -0x1.01fffffe00000p-954, -0x1.ffffffffffc00p-1012, 0x1.0000000000000p+14, 0x1.ffe001c0003c0p-45},
{-0x1.0000000000000p+14, -0x1.ffe001c0003c0p-45, 0x1.01fffffe00000p-954, 0x1.ffffffffffc00p-1012, -0x1.0000000000000p+14, -0x1.ffe001c0003c0p-45},
{-0x1.0000000000000p+14, -0x1.ffe001c0003c0p-45, -0x1.01fffffe00000p-954, -0x1.ffffffffffc00p-1012, -0x1.0000000000000p+14, -0x1.ffe001c0003c0p-45},
{0x1.00003ff808000p-919, -0x1.ffe0000800000p-980, 0x1.000001ffff000p-62, 0x1.ff80000000000p-136, 0x1.000001ffff000p-62, 0x1.ff80000000000p-136},
{0x1.00003ff808000p-919, -0x1.ffe0000800000p-980, -0x1.000001ffff000p-62, -0x1.ff80000000000p-136, -0x1.000001ffff000p-62, -0x1.ff80000000000p-136},
{-0x1.00003ff808000p-919, 0x1.ffe0000800000p-980, 0x1.000001ffff000p-62, 0x1.ff80000000000p-136, 0x1.000001ffff000p-62, 0x1.ff80000000000p-136},
{-0x1.00003ff808000p-919, 0x1.ffe0000800000p-980, -0x1.000001ffff000p-62, -0x1.ff80000000000p-136, -0x1.000001ffff000p-62, -0x1.ff80000000000p-136},
{0x1.0000000000000p-665, 0x1.fc007f0ff0000p-727, 0x1.0000000007f80p-960, 0x1.ff0007f802000p-1024, 0x1.0000000000000p-665, 0x1.fc007f0ff0000p-727},
{0x1.0000000000000p-665, 0x1.fc007f0ff0000p-727, -0x1.0000000007f80p-960, -0x1.ff0007f802000p-1024, 0x1.0000000000000p-665, 0x1.fc007f0ff0000p-727},
{-0x1.0000000000000p-665, -0x1.fc007f0ff0000p-727, 0x1.0000000007f80p-960, 0x1.ff0007f802000p-1024, -0x1.0000000000000p-665, -0x1.fc007f0ff0000p-727},
{-0x1.0000000000000p-665, -0x1.fc007f0ff0000p-727, -0x1.0000000007f80p-960, -0x1.ff0007f802000p-1024, -0x1.0000000000000p-665, -0x1.fc007f0ff0000p-727},
{0x1.0020000000000p+107, -0x1.0000000000000p+30, 0x1.000001ffff802p-529, -0x1.ffffffc000040p-588, 0x1.0020000000000p+107, -0x1.0000000000000p+30},
{0x1.0020000000000p+107, -0x1.0000000000000p+30, -0x1.000001ffff802p-529, 0x1.ffffffc000040p-588, 0x1.0020000000000p+107, -0x1.0000000000000p+30},
{-0x1.0020000000000p+107, 0x1.0000000000000p+30, 0x1.000001ffff802p-529, -0x1.ffffffc000040p-588, -0x1.0020000000000p+107, 0x1.0000000000000p+30},
{-0x1.0020000000000p+107, 0x1.0000000000000p+30, -0x1.000001ffff802p-529, 0x1.ffffffc000040p-588, -0x1.0020000000000p+107, 0x1.0000000000000p+30},
{0x1.0000000000800p+985, -0x1.ffe0000400000p+923, 0x1.0000000000038p+697, 0x1.e100000000000p+623, 0x1.0000000000800p+985, -0x1.ffe0000400000p+923},
{0x1.0000000000800p+985, -0x1.ffe0000400000p+923, -0x1.0000000000038p+697, -0x1.e100000000000p+623, 0x1.0000000000800p+985, -0x1.ffe0000400000p+923},
{-0x1.0000000000800p+985, 0x1.ffe0000400000p+923, 0x1.0000000000038p+697, 0x1.e100000000000p+623, -0x1.0000000000800p+985, 0x1.ffe0000400000p+923},
{-0x1.0000000000800p+985, 0x1.ffe0000400000p+923, -0x1.0000000000038p+697, -0x1.e100000000000p+623, -0x1.0000000000800p+985, 0x1.ffe0000400000p+923},
{0x1.0000000000020p+26, -0x1.8007ffff80c04p-29, 0x1.0000000000000p+576, 0x1.ffffe00000c00p+513, 0x1.0000000000000p+576, 0x1.ffffe00000c00p+513},
{0x1.0000000000020p+26, -0x1.8007ffff80c04p-29, -0x1.0000000000000p+576, -0x1.ffffe00000c00p+513, -0x1.0000000000000p+576, -0x1.ffffe00000c00p+513},
{-0x1.0000000000020p+26, 0x1.8007ffff80c04p-29, 0x1.0000000000000p+576, 0x1.ffffe00000c00p+513, 0x1.0000000000000p+576, 0x1.ffffe00000c00p+513},
{-0x1.0000000000020p+26, 0x1.8007ffff80c04p-29, -0x1.0000000000000p+576, -0x1.ffffe00000c00p+513, -0x1.0000000000000p+576, -0x1.ffffe00000c00p+513},
{0x1.7fffffff80000p-290, 0x1.f000000000000p-367, 0x1.000007ffffffep-351, 0x1.ffffff8000000p-427, 0x1.7fffffff80000p-290, 0x1.0001f80000000p-351},
{0x1.7fffffff80000p-290, 0x1.f000000000000p-367, -0x1.000007ffffffep-351, -0x1.ffffff8000000p-427, 0x1.7fffffff80000p-290, -0x1.fffc300000000p-352},
{-0x1.7fffffff80000p-290, -0x1.f000000000000p-367, 0x1.000007ffffffep-351, 0x1.ffffff8000000p-427, -0x1.7fffffff80000p-290, 0x1.fffc300000000p-352},
{-0x1.7fffffff80000p-290, -0x1.f000000000000p-367, -0x1.000007ffffffep-351, -0x1.ffffff8000000p-427, -0x1.7fffffff80000p-290, -0x1.0001f80000000p-351},
{0x1.0000000010000p-869, -0x1.0000000000000p-955, 0x1.0000000000000p-250, 0x1.c000003ffc3fep-304, 0x1.0000000000000p-250, 0x1.c000003ffc3fep-304},
{0x1.0000000010000p-869, -0x1.0000000000000p-955, -0x1.0000000000000p-250, -0x1.c000003ffc3fep-304, -0x1.0000000000000p-250, -0x1.c000003ffc3fep-304},
{-0x1.0000000010000p-869, 0x1.0000000000000p-955, 0x1.0000000000000p-250, 0x1.c000003ffc3fep-304, 0x1.0000000000000p-250, 0x1.c000003ffc3fep-304},
{-0x1.0000000010000p-869, 0x1.0000000000000p-955, -0x1.0000000000000p-250, -0x1.c000003ffc3fep-304, -0x1.0000000000000p-250, -0x1.c000003ffc3fep-304},
{0x1.00000ffff0800p-239, -0x1.fff80000ffc20p-297, 0x1.0000000000002p-749, -0x1.0000000000000p-854, 0x1.00000ffff0800p-239, -0x1.fff80000ffc20p-297},
{0x1.00000ffff0800p-239, -0x1.fff80000ffc20p-297, -0x1.0000000000002p-749, 0x1.0000000000000p-854, 0x1.00000ffff0800p-239, -0x1.fff80000ffc20p-297},
{-0x1.00000ffff0800p-239, 0x1.fff80000ffc20p-297, 0x1.0000000000002p-749, -0x1.0000000000000p-854, -0x1.00000ffff0800p-239, 0x1.fff80000ffc20p-297},
{-0x1.00000ffff0800p-239, 0x1.fff80000ffc20p-297, -0x1.0000000000002p-749, 0x1.0000000000000p-854, -0x1.00000ffff0800p-239, 0x1.fff80000ffc20p-297},
{0x1.000ffffc00000p+483, 0x1.fffff00000000p+414, 0x1.0000000000040p+210, -0x1.0000000000000p+105, 0x1.000ffffc00000p+483, 0x1.fffff00000000p+414},
{0x1.000ffffc00000p+483, 0x1.fffff00000000p+414, -0x1.0000000000040p+210, 0x1.0000000000000p+105, 0x1.000ffffc00000p+483, 0x1.fffff00000000p+414},
{-0x1.000ffffc00000p+483, -0x1.fffff00000000p+414, 0x1.0000000000040p+210, -0x1.0000000000000p+105, -0x1.000ffffc00000p+483, -0x1.fffff00000000p+414},
{-0x1.000ffffc00000p+483, -0x1.fffff00000000p+414, -0x1.0000000000040p+210, 0x1.0000000000000p+105, -0x1.000ffffc00000p+483, -0x1.fffff00000000p+414},
{0x1.0004000000000p-94, -0x1.0000000000000p-162, 0x1.0000000200000p-175, -0x1.ffbffffff8020p-233, 0x1.0004000000000p-94, -0x1.fff0000000000p-163},
{0x1.0004000000000p-94, -0x1.0000000000000p-162, -0x1.0000000200000p-175, 0x1.ffbffffff8020p-233, 0x1.0004000000000p-94, -0x1.0008000000000p-162},
{-0x1.0004000000000p-94, 0x1.0000000000000p-162, 0x1.0000000200000p-175, -0x1.ffbffffff8020p-233, -0x1.0004000000000p-94, 0x1.0008000000000p-162},
{-0x1.0004000000000p-94, 0x1.0000000000000p-162, -0x1.0000000200000p-175, 0x1.ffbffffff8020p-233, -0x1.0004000000000p-94, 0x1.fff0000000000p-163},
{0x1.0000000000000p-421, 0x1.e0001e7fbff00p-482, 0x1.0000001fffffep-395, 0x0.0000000000000p+0, 0x1.0000005fffffep-395, 0x1.e000000000000p-482},
{0x1.0000000000000p-421, 0x1.e0001e7fbff00p-482, -0x1.0000001fffffep-395, 0x0.0000000000000p+0, -0x1.ffffffbfffffcp-396, 0x1.e000200000000p-482},
{-0x1.0000000000000p-421, -0x1.e0001e7fbff00p-482, 0x1.0000001fffffep-395, 0x0.0000000000000p+0, 0x1.ffffffbfffffcp-396, -0x1.e000200000000p-482},
{-0x1.0000000000000p-421, -0x1.e0001e7fbff00p-482, -0x1.0000001fffffep-395, 0x0.0000000000000p+0, -0x1.0000005fffffep-395, -0x1.e000000000000p-482},
{0x1.0000000000fffp-362, 0x1.fffff3e000000p-435, 0x1.001ffe0000000p+872, 0x1.fffffffffffc0p+813, 0x1.001ffe0000000p+872, 0x1.fffffffffffc0p+813},
{0x1.0000000000fffp-362, 0x1.fffff3e000000p-435, -0x1.001ffe0000000p+872, -0x1.fffffffffffc0p+813, -0x1.001ffe0000000p+872, -0x1.fffffffffffc0p+813},
{-0x1.0000000000fffp-362, -0x1.fffff3e000000p-435, 0x1.001ffe0000000p+872, 0x1.fffffffffffc0p+813, 0x1.001ffe0000000p+872, 0x1.fffffffffffc0p+813},
{-0x1.0000000000fffp-362, -0x1.fffff3e000000p-435, -0x1.001ffe0000000p+872, -0x1.fffffffffffc0p+813, -0x1.001ffe0000000p+872, -0x1.fffffffffffc0p+813},
{0x1.0000080000000p+509, -0x1.0000000000000p+404, 0x1.0000000080000p-689, -0x1.0000000000000p-744, 0x1.0000080000000p+509, -0x1.0000000000000p+404},
{0x1.0000080000000p+509, -0x1.0000000000000p+404, -0x1.0000000080000p-689, 0x1.0000000000000p-744, 0x1.0000080000000p+509, -0x1.0000000000000p+404},
{-0x1.0000080000000p+509, 0x1.0000000000000p+404, 0x1.0000000080000p-689, -0x1.0000000000000p-744, -0x1.0000080000000p+509, 0x1.0000000000000p+404},
{-0x1.0000080000000p+509, 0x1.0000000000000p+404, -0x1.0000000080000p-689, 0x1.0000000000000p-744, -0x1.0000080000000p+509, 0x1.0000000000000p+404},
{0x1.003ffffffffe0p+511, 0x1.ffffe03800000p+435, 0x1.0000000002000p+705, -0x1.0000000000000p+652, 0x1.0000000002000p+705, -0x1.0000000000000p+652},
{0x1.003ffffffffe0p+511, 0x1.ffffe03800000p+435, -0x1.0000000002000p+705, 0x1.0000000000000p+652, -0x1.0000000002000p+705, 0x1.0000000000000p+652},
{-0x1.003ffffffffe0p+511, -0x1.ffffe03800000p+435, 0x1.0000000002000p+705, -0x1.0000000000000p+652, 0x1.0000000002000p+705, -0x1.0000000000000p+652},
{-0x1.003ffffffffe0p+511, -0x1.ffffe03800000p+435, -0x1.0000000002000p+705, 0x1.0000000000000p+652, -0x1.0000000002000p+705, 0x1.0000000000000p+652},
{0x1.0000000000008p-749, -0x1.ff000007ffc00p-809, 0x1.0000000060000p+866, 0x1.ffffff8000000p+786, 0x1.0000000060000p+866, 0x1.ffffff8000000p+786},
{0x1.0000000000008p-749, -0x1.ff000007ffc00p-809, -0x1.0000000060000p+866, -0x1.ffffff8000000p+786, -0x1.0000000060000p+866, -0x1.ffffff8000000p+786},
{-0x1.0000000000008p-749, 0x1.ff000007ffc00p-809, 0x1.0000000060000p+866, 0x1.ffffff8000000p+786, 0x1.0000000060000p+866, 0x1.ffffff8000000p+786},
{-0x1.0000000000008p-749, 0x1.ff000007ffc00p-809, -0x1.0000000060000p+866, -0x1.ffffff8000000p+786, -0x1.0000000060000p+866, -0x1.ffffff8000000p+786},
{0x1.0000000000080p+921, -0x1.0000000000000p+816, 0x1.01ffffff80001p+85, -0x1.ff90000000000p-8, 0x1.0000000000080p+921, -0x1.0000000000000p+816},
{0x1.0000000000080p+921, -0x1.0000000000000p+816, -0x1.01ffffff80001p+85, 0x1.ff90000000000p-8, 0x1.0000000000080p+921, -0x1.0000000000000p+816},
{-0x1.0000000000080p+921, 0x1.0000000000000p+816, 0x1.01ffffff80001p+85, -0x1.ff90000000000p-8, -0x1.0000000000080p+921, 0x1.0000000000000p+816},
{-0x1.0000000000080p+921, 0x1.0000000000000p+816, -0x1.01ffffff80001p+85, 0x1.ff90000000000p-8, -0x1.0000000000080p+921, 0x1.0000000000000p+816},
{0x1.0000000000800p-666, -0x1.fffff80002000p-732, 0x1.0008000000000p-639, -0x1.0000000000000p-718, 0x1.0008002000000p-639, 0x1.ffbffe0000000p-708},
{0x1.0000000000800p-666, -0x1.fffff80002000p-732, -0x1.0008000000000p-639, 0x1.0000000000000p-718, -0x1.0007ffe000000p-639, 0x1.001fff0000000p-707},
{-0x1.0000000000800p-666, 0x1.fffff80002000p-732, 0x1.0008000000000p-639, -0x1.0000000000000p-718, 0x1.0007ffe000000p-639, -0x1.001fff0000000p-707},
{-0x1.0000000000800p-666, 0x1.fffff80002000p-732, -0x1.0008000000000p-639, 0x1.0000000000000p-718, -0x1.0008002000000p-639, -0x1.ffbffe0000000p-708},
{0x1.0000000000002p+14, -0x1.ffffffe200000p-60, 0x1.0003fffffffc0p+133, 0x1.ffffe00000400p+70, 0x1.0003fffffffc0p+133, 0x1.ffffe00000400p+70},
{0x1.0000000000002p+14, -0x1.ffffffe200000p-60, -0x1.0003fffffffc0p+133, -0x1.ffffe00000400p+70, -0x1.0003fffffffc0p+133, -0x1.ffffe00000400p+70},
{-0x1.0000000000002p+14, 0x1.ffffffe200000p-60, 0x1.0003fffffffc0p+133, 0x1.ffffe00000400p+70, 0x1.0003fffffffc0p+133, 0x1.ffffe00000400p+70},
{-0x1.0000000000002p+14, 0x1.ffffffe200000p-60, -0x1.0003fffffffc0p+133, -0x1.ffffe00000400p+70, -0x1.0003fffffffc0p+133, -0x1.ffffe00000400p+70},
{0x1.000003f000000p+120, 0x1.fffffffe00000p+59, 0x1.0000000000ff0p+144, 0x1.c03fff0e00000p+70, 0x1.000001000102fp+144, 0x1.c07fff0e00000p+70},
{0x1.000003f000000p+120, 0x1.fffffffe00000p+59, -0x1.0000000000ff0p+144, -0x1.c03fff0e00000p+70, -0x1.fffffe0001f62p+143, -0x1.bfffff0e00000p+70},
{-0x1.000003f000000p+120, -0x1.fffffffe00000p+59, 0x1.0000000000ff0p+144, 0x1.c03fff0e00000p+70, 0x1.fffffe0001f62p+143, 0x1.bfffff0e00000p+70},
{-0x1.000003f000000p+120, -0x1.fffffffe00000p+59, -0x1.0000000000ff0p+144, -0x1.c03fff0e00000p+70, -0x1.000001000102fp+144, -0x1.c07fff0e00000p+70},
{0x1.0000000018000p+68, 0x1.fffffe0000000p-14, 0x1.0000100000000p+766, -0x1.0000000000000p+661, 0x1.0000100000000p+766, -0x1.0000000000000p+661},
{0x1.0000000018000p+68, 0x1.fffffe0000000p-14, -0x1.0000100000000p+766, 0x1.0000000000000p+661, -0x1.0000100000000p+766, 0x1.0000000000000p+661},
{-0x1.0000000018000p+68, -0x1.fffffe0000000p-14, 0x1.0000100000000p+766, -0x1.0000000000000p+661, 0x1.0000100000000p+766, -0x1.0000000000000p+661},
{-0x1.0000000018000p+68, -0x1.fffffe0000000p-14, -0x1.0000100000000p+766, 0x1.0000000000000p+661, -0x1.0000100000000p+766, 0x1.0000000000000p+661},
{0x1.0000000000001p-191, -0x1.fffffc1000000p-258, 0x1.1ffffff01f000p+311, 0x1.ffffc000060fcp+256, 0x1.1ffffff01f000p+311, 0x1.ffffc000060fcp+256},
{0x1.0000000000001p-191, -0x1.fffffc1000000p-258, -0x1.1ffffff01f000p+311, -0x1.ffffc000060fcp+256, -0x1.1ffffff01f000p+311, -0x1.ffffc000060fcp+256},
{-0x1.0000000000001p-191, 0x1.fffffc1000000p-258, 0x1.1ffffff01f000p+311, 0x1.ffffc000060fcp+256, 0x1.1ffffff01f000p+311, 0x1.ffffc000060fcp+256},
{-0x1.0000000000001p-191, 0x1.fffffc1000000p-258, -0x1.1ffffff01f000p+311, -0x1.ffffc000060fcp+256, -0x1.1ffffff01f000p+311, -0x1.ffffc000060fcp+256},
{0x1.0000003800000p-790, 0x1.ffe000001fffcp-845, 0x1.0000007fffff0p-815, 0x1.fe1ffffc00000p-877, 0x1.000000b800004p-790, 0x1.ffdfc0021e1fcp-845},
{0x1.0000003800000p-790, 0x1.ffe000001fffcp-845, -0x1.0000007fffff0p-815, -0x1.fe1ffffc00000p-877, 0x1.ffffff6fffff8p-791, 0x1.ffe03ffe21dfcp-845},
{-0x1.0000003800000p-790, -0x1.ffe000001fffcp-845, 0x1.0000007fffff0p-815, 0x1.fe1ffffc00000p-877, -0x1.ffffff6fffff8p-791, -0x1.ffe03ffe21dfcp-845},
{-0x1.0000003800000p-790, -0x1.ffe000001fffcp-845, -0x1.0000007fffff0p-815, -0x1.fe1ffffc00000p-877, -0x1.000000b800004p-790, -0x1.ffdfc0021e1fcp-845},
{0x1.0000001fffc00p-770, 0x1.f00007ffe0000p-834, 0x1.0000000000180p+996, 0x1.fffffe0000800p+932, 0x1.0000000000180p+996, 0x1.fffffe0000800p+932},
{0x1.0000001fffc00p-770, 0x1.f00007ffe0000p-834, -0x1.0000000000180p+996, -0x1.fffffe0000800p+932, -0x1.0000000000180p+996, -0x1.fffffe0000800p+932},
{-0x1.0000001fffc00p-770, -0x1.f00007ffe0000p-834, 0x1.0000000000180p+996, 0x1.fffffe0000800p+932, 0x1.0000000000180p+996, 0x1.fffffe0000800p+932},
{-0x1.0000001fffc00p-770, -0x1.f00007ffe0000p-834, -0x1.0000000000180p+996, -0x1.fffffe0000800p+932, -0x1.0000000000180p+996, -0x1.fffffe0000800p+932},
{0x1.0000000008000p+440, -0x1.c0000007ffe04p+385, 0x1.00000fffffffep-521, 0x1.fc00000000000p-620, 0x1.0000000008000p+440, -0x1.c0000007ffe04p+385},
{0x1.0000000008000p+440, -0x1.c0000007ffe04p+385, -0x1.00000fffffffep-521, -0x1.fc00000000000p-620, 0x1.0000000008000p+440, -0x1.c0000007ffe04p+385},
{-0x1.0000000008000p+440, 0x1.c0000007ffe04p+385, 0x1.00000fffffffep-521, 0x1.fc00000000000p-620, -0x1.0000000008000p+440, 0x1.c0000007ffe04p+385},
{-0x1.0000000008000p+440, 0x1.c0000007ffe04p+385, -0x1.00000fffffffep-521, -0x1.fc00000000000p-620, -0x1.0000000008000p+440, 0x1.c0000007ffe04p+385},
{0x1.0000003e00004p-731, -0x1.fe00040000000p-801, 0x1.00007c0001e00p+483, 0x1.fffe13ffe0000p+419, 0x1.00007c0001e00p+483, 0x1.fffe13ffe0000p+419},
{0x1.0000003e00004p-731, -0x1.fe00040000000p-801, -0x1.00007c0001e00p+483, -0x1.fffe13ffe0000p+419, -0x1.00007c0001e00p+483, -0x1.fffe13ffe0000p+419},
{-0x1.0000003e00004p-731, 0x1.fe00040000000p-801, 0x1.00007c0001e00p+483, 0x1.fffe13ffe0000p+419, 0x1.00007c0001e00p+483, 0x1.fffe13ffe0000p+419},
{-0x1.0000003e00004p-731, 0x1.fe00040000000p-801, -0x1.00007c0001e00p+483, -0x1.fffe13ffe0000p+419, -0x1.00007c0001e00p+483, -0x1.fffe13ffe0000p+419},
{0x1.03fffffffc000p-192, 0x1.ffffffc000000p-267, 0x1.0003ffffff000p+649, 0x1.ffc0000000000p+554, 0x1.0003ffffff000p+649, 0x1.ffc0000000000p+554},
{0x1.03fffffffc000p-192, 0x1.ffffffc000000p-267, -0x1.0003ffffff000p+649, -0x1.ffc0000000000p+554, -0x1.0003ffffff000p+649, -0x1.ffc0000000000p+554},
{-0x1.03fffffffc000p-192, -0x1.ffffffc000000p-267, 0x1.0003ffffff000p+649, 0x1.ffc0000000000p+554, 0x1.0003ffffff000p+649, 0x1.ffc0000000000p+554},
{-0x1.03fffffffc000p-192, -0x1.ffffffc000000p-267, -0x1.0003ffffff000p+649, -0x1.ffc0000000000p+554, -0x1.0003ffffff000p+649, -0x1.ffc0000000000p+554},
{0x1.07ffffff80000p+301, 0x1.8007ffffc0000p+230, 0x1.0000000000004p-464, -0x1.fffc000200000p-533, 0x1.07ffffff80000p+301, 0x1.8007ffffc0000p+230},
{0x1.07ffffff80000p+301, 0x1.8007ffffc0000p+230, -0x1.0000000000004p-464, 0x1.fffc000200000p-533, 0x1.07ffffff80000p+301, 0x1.8007ffffc0000p+230},
{-0x1.07ffffff80000p+301, -0x1.8007ffffc0000p+230, 0x1.0000000000004p-464, -0x1.fffc000200000p-533, -0x1.07ffffff80000p+301, -0x1.8007ffffc0000p+230},
{-0x1.07ffffff80000p+301, -0x1.8007ffffc0000p+230, -0x1.0000000000004p-464, 0x1.fffc000200000p-533, -0x1.07ffffff80000p+301, -0x1.8007ffffc0000p+230},
{0x1.00003ffffff80p-1007, 0x1.0000000000000p-1068, 0x1.00003fffffff8p+0, 0x0.0000000000000p+0, 0x1.00003fffffff8p+0, 0x0.0000000000000p+0},
{0x1.00003ffffff80p-1007, 0x1.0000000000000p-1068, -0x1.00003fffffff8p+0, 0x0.0000000000000p+0, -0x1.00003fffffff8p+0, 0x0.0000000000000p+0},
{-0x1.00003ffffff80p-1007, -0x1.0000000000000p-1068, 0x1.00003fffffff8p+0, 0x0.0000000000000p+0, 0x1.00003fffffff8p+0, 0x0.0000000000000p+0},
{-0x1.00003ffffff80p-1007, -0x1.0000000000000p-1068, -0x1.00003fffffff8p+0, 0x0.0000000000000p+0, -0x1.00003fffffff8p+0, 0x0.0000000000000p+0},
{0x1.0000010000000p-912, -0x1.0000000000000p-1017, 0x1.00f81ffffff81p+122, 0x1.fffff03c00000p+60, 0x1.00f81ffffff81p+122, 0x1.fffff03c00000p+60},
{0x1.0000010000000p-912, -0x1.0000000000000p-1017, -0x1.00f81ffffff81p+122, -0x1.fffff03c00000p+60, -0x1.00f81ffffff81p+122, -0x1.fffff03c00000p+60},
{-0x1.0000010000000p-912, 0x1.0000000000000p-1017, 0x1.00f81ffffff81p+122, 0x1.fffff03c00000p+60, 0x1.00f81ffffff81p+122, 0x1.fffff03c00000p+60},
{-0x1.0000010000000p-912, 0x1.0000000000000p-1017, -0x1.00f81ffffff81p+122, -0x1.fffff03c00000p+60, -0x1.00f81ffffff81p+122, -0x1.fffff03c00000p+60},
{0x1.0000000000004p+652, -0x1.fff0008000000p+588, 0x1.0000000100000p+402, -0x1.fffffe0004000p+335, 0x1.0000000000004p+652, -0x1.fff0008000000p+588},
{0x1.0000000000004p+652, -0x1.fff0008000000p+588, -0x1.0000000100000p+402, 0x1.fffffe0004000p+335, 0x1.0000000000004p+652, -0x1.fff0008000000p+588},
{-0x1.0000000000004p+652, 0x1.fff0008000000p+588, 0x1.0000000100000p+402, -0x1.fffffe0004000p+335, -0x1.0000000000004p+652, 0x1.fff0008000000p+588},
{-0x1.0000000000004p+652, 0x1.fff0008000000p+588, -0x1.0000000100000p+402, 0x1.fffffe0004000p+335, -0x1.0000000000004p+652, 0x1.fff0008000000p+588},
{0x1.0000000000800p-11, -0x1.0000000000000p-89, 0x1.0000010000000p+68, -0x1.0000000000000p-35, 0x1.0000010000000p+68, 0x1.fffffe0000000p-12},
{0x1.0000000000800p-11, -0x1.0000000000000p-89, -0x1.0000010000000p+68, 0x1.0000000000000p-35, -0x1.0000010000000p+68, 0x1.0000010000000p-11},
{-0x1.0000000000800p-11, 0x1.0000000000000p-89, 0x1.0000010000000p+68, -0x1.0000000000000p-35, 0x1.0000010000000p+68, -0x1.0000010000000p-11},
{-0x1.0000000000800p-11, 0x1.0000000000000p-89, -0x1.0000010000000p+68, 0x1.0000000000000p-35, -0x1.0000010000000p+68, -0x1.fffffe0000000p-12},
{0x1.00000007fff00p+898, 0x1.fffc1ffff87f0p+842, 0x1.0000007c00000p-1007, 0x0.0000000000000p+0, 0x1.00000007fff00p+898, 0x1.fffc1ffff87f0p+842},
{0x1.00000007fff00p+898, 0x1.fffc1ffff87f0p+842, -0x1.0000007c00000p-1007, -0x0.0000000000000p+0, 0x1.00000007fff00p+898, 0x1.fffc1ffff87f0p+842},
{-0x1.00000007fff00p+898, -0x1.fffc1ffff87f0p+842, 0x1.0000007c00000p-1007, 0x0.0000000000000p+0, -0x1.00000007fff00p+898, -0x1.fffc1ffff87f0p+842},
{-0x1.00000007fff00p+898, -0x1.fffc1ffff87f0p+842, -0x1.0000007c00000p-1007, -0x0.0000000000000p+0, -0x1.00000007fff00p+898, -0x1.fffc1ffff87f0p+842},
{0x1.fffffffe00000p+397, 0x1.e000001fff800p+333, 0x1.00000001ffffep+347, 0x0.0000000000000p+0, 0x1.fffffffe00004p+397, 0x1.e000801ff7800p+333},
{0x1.fffffffe00000p+397, 0x1.e000001fff800p+333, -0x1.00000001ffffep+347, 0x0.0000000000000p+0, 0x1.fffffffdffffcp+397, 0x1.dfff802007800p+333},
{-0x1.fffffffe00000p+397, -0x1.e000001fff800p+333, 0x1.00000001ffffep+347, 0x0.0000000000000p+0, -0x1.fffffffdffffcp+397, -0x1.dfff802007800p+333},
{-0x1.fffffffe00000p+397, -0x1.e000001fff800p+333, -0x1.00000001ffffep+347, 0x0.0000000000000p+0, -0x1.fffffffe00004p+397, -0x1.e000801ff7800p+333},
{0x1.0000000000008p-231, -0x1.fffff00400000p-303, 0x1.00003ffffe000p-829, 0x1.ffffe00000000p-901, 0x1.0000000000008p-231, -0x1.fffff00400000p-303},
{0x1.0000000000008p-231, -0x1.fffff00400000p-303, -0x1.00003ffffe000p-829, -0x1.ffffe00000000p-901, 0x1.0000000000008p-231, -0x1.fffff00400000p-303},
{-0x1.0000000000008p-231, 0x1.fffff00400000p-303, 0x1.00003ffffe000p-829, 0x1.ffffe00000000p-901, -0x1.0000000000008p-231, 0x1.fffff00400000p-303},
{-0x1.0000000000008p-231, 0x1.fffff00400000p-303, -0x1.00003ffffe000p-829, -0x1.ffffe00000000p-901, -0x1.0000000000008p-231, 0x1.fffff00400000p-303},
{0x1.0000000000000p-479, 0x1.ffffffffffc00p-542, 0x1.000001ffe0000p-355, 0x1.ff87ff9f801c0p-414, 0x1.000001ffe0000p-355, 0x1.ff87ff9f801c0p-414},
{0x1.0000000000000p-479, 0x1.ffffffffffc00p-542, -0x1.000001ffe0000p-355, -0x1.ff87ff9f801c0p-414, -0x1.000001ffe0000p-355, -0x1.ff87ff9f801c0p-414},
{-0x1.0000000000000p-479, -0x1.ffffffffffc00p-542, 0x1.000001ffe0000p-355, 0x1.ff87ff9f801c0p-414, 0x1.000001ffe0000p-355, 0x1.ff87ff9f801c0p-414},
{-0x1.0000000000000p-479, -0x1.ffffffffffc00p-542, -0x1.000001ffe0000p-355, -0x1.ff87ff9f801c0p-414, -0x1.000001ffe0000p-355, -0x1.ff87ff9f801c0p-414},
{0x1.0000000000400p-496, -0x1.f800000002000p-557, 0x1.03fffffffff00p-62, 0x1.fffe7fffffffep-116, 0x1.03fffffffff00p-62, 0x1.fffe7fffffffep-116},
{0x1.0000000000400p-496, -0x1.f800000002000p-557, -0x1.03fffffffff00p-62, -0x1.fffe7fffffffep-116, -0x1.03fffffffff00p-62, -0x1.fffe7fffffffep-116},
{-0x1.0000000000400p-496, 0x1.f800000002000p-557, 0x1.03fffffffff00p-62, 0x1.fffe7fffffffep-116, 0x1.03fffffffff00p-62, 0x1.fffe7fffffffep-116},
{-0x1.0000000000400p-496, 0x1.f800000002000p-557, -0x1.03fffffffff00p-62, -0x1.fffe7fffffffep-116, -0x1.03fffffffff00p-62, -0x1.fffe7fffffffep-116},
{0x1.0000000000000p+267, 0x1.fffffffffff00p+211, 0x1.000007c000000p+439, 0x1.fff8000000000p+383, 0x1.000007c000000p+439, 0x1.fff8000000000p+383},
{0x1.0000000000000p+267, 0x1.fffffffffff00p+211, -0x1.000007c000000p+439, -0x1.fff8000000000p+383, -0x1.000007c000000p+439, -0x1.fff8000000000p+383},
{-0x1.0000000000000p+267, -0x1.fffffffffff00p+211, 0x1.000007c000000p+439, 0x1.fff8000000000p+383, 0x1.000007c000000p+439, 0x1.fff8000000000p+383},
{-0x1.0000000000000p+267, -0x1.fffffffffff00p+211, -0x1.000007c000000p+439, -0x1.fff8000000000p+383, -0x1.000007c000000p+439, -0x1.fff8000000000p+383},
{0x1.0000000000400p-383, -0x1.fffc0003fffd0p-440, 0x1.0000000000000p+249, 0x1.fffffff800000p+193, 0x1.0000000000000p+249, 0x1.fffffff800000p+193},
{0x1.0000000000400p-383, -0x1.fffc0003fffd0p-440, -0x1.0000000000000p+249, -0x1.fffffff800000p+193, -0x1.0000000000000p+249, -0x1.fffffff800000p+193},
{-0x1.0000000000400p-383, 0x1.fffc0003fffd0p-440, 0x1.0000000000000p+249, 0x1.fffffff800000p+193, 0x1.0000000000000p+249, 0x1.fffffff800000p+193},
{-0x1.0000000000400p-383, 0x1.fffc0003fffd0p-440, -0x1.0000000000000p+249, -0x1.fffffff800000p+193, -0x1.0000000000000p+249, -0x1.fffffff800000p+193},
{0x1.0000000002000p-673, -0x1.ffff040000000p-744, 0x1.03fffff800000p+930, 0x1.fffffe0000000p+862, 0x1.03fffff800000p+930, 0x1.fffffe0000000p+862},
{0x1.0000000002000p-673, -0x1.ffff040000000p-744, -0x1.03fffff800000p+930, -0x1.fffffe0000000p+862, -0x1.03fffff800000p+930, -0x1.fffffe0000000p+862},
{-0x1.0000000002000p-673, 0x1.ffff040000000p-744, 0x1.03fffff800000p+930, 0x1.fffffe0000000p+862, 0x1.03fffff800000p+930, 0x1.fffffe0000000p+862},
{-0x1.0000000002000p-673, 0x1.ffff040000000p-744, -0x1.03fffff800000p+930, -0x1.fffffe0000000p+862, -0x1.03fffff800000p+930, -0x1.fffffe0000000p+862},
{0x1.0000008000000p+436, -0x1.ffffff8200000p+379, 0x1.000fff8000000p-8, 0x1.fffffe0000000p-70, 0x1.0000008000000p+436, -0x1.ffffff8200000p+379},
{0x1.0000008000000p+436, -0x1.ffffff8200000p+379, -0x1.000fff8000000p-8, -0x1.fffffe0000000p-70, 0x1.0000008000000p+436, -0x1.ffffff8200000p+379},
{-0x1.0000008000000p+436, 0x1.ffffff8200000p+379, 0x1.000fff8000000p-8, 0x1.fffffe0000000p-70, -0x1.0000008000000p+436, 0x1.ffffff8200000p+379},
{-0x1.0000008000000p+436, 0x1.ffffff8200000p+379, -0x1.000fff8000000p-8, -0x1.fffffe0000000p-70, -0x1.0000008000000p+436, 0x1.ffffff8200000p+379},
{0x1.0000000000000p+138, 0x1.0001f9e000000p+78, 0x1.0000000000040p+438, -0x1.fffffc0080000p+366, 0x1.0000000000040p+438, -0x1.fffffc0080000p+366},
{0x1.0000000000000p+138, 0x1.0001f9e000000p+78, -0x1.0000000000040p+438, 0x1.fffffc0080000p+366, -0x1.0000000000040p+438, 0x1.fffffc0080000p+366},
{-0x1.0000000000000p+138, -0x1.0001f9e000000p+78, 0x1.0000000000040p+438, -0x1.fffffc0080000p+366, 0x1.0000000000040p+438, -0x1.fffffc0080000p+366},
{-0x1.0000000000000p+138, -0x1.0001f9e000000p+78, -0x1.0000000000040p+438, 0x1.fffffc0080000p+366, -0x1.0000000000040p+438, 0x1.fffffc0080000p+366},
{0x1.0000ff0007e00p+955, 0x1.ffc001ff00000p+888, 0x1.0000010000000p+582, -0x1.ffc007ffffe20p+524, 0x1.0000ff0007e00p+955, 0x1.ffc001ff00000p+888},
{0x1.0000ff0007e00p+955, 0x1.ffc001ff00000p+888, -0x1.0000010000000p+582, 0x1.ffc007ffffe20p+524, 0x1.0000ff0007e00p+955, 0x1.ffc001ff00000p+888},
{-0x1.0000ff0007e00p+955, -0x1.ffc001ff00000p+888, 0x1.0000010000000p+582, -0x1.ffc007ffffe20p+524, -0x1.0000ff0007e00p+955, -0x1.ffc001ff00000p+888},
{-0x1.0000ff0007e00p+955, -0x1.ffc001ff00000p+888, -0x1.0000010000000p+582, 0x1.ffc007ffffe20p+524, -0x1.0000ff0007e00p+955, -0x1.ffc001ff00000p+888},
{0x1.07ffffff1fe00p+791, 0x1.fc00000000000p+694, 0x1.0040000000000p-580, -0x1.c000000400000p-641, 0x1.07ffffff1fe00p+791, 0x1.fc00000000000p+694},
{0x1.07ffffff1fe00p+791, 0x1.fc00000000000p+694, -0x1.0040000000000p-580, 0x1.c000000400000p-641, 0x1.07ffffff1fe00p+791, 0x1.fc00000000000p+694},
{-0x1.07ffffff1fe00p+791, -0x1.fc00000000000p+694, 0x1.0040000000000p-580, -0x1.c000000400000p-641, -0x1.07ffffff1fe00p+791, -0x1.fc00000000000p+694},
{-0x1.07ffffff1fe00p+791, -0x1.fc00000000000p+694, -0x1.0040000000000p-580, 0x1.c000000400000p-641, -0x1.07ffffff1fe00p+791, -0x1.fc00000000000p+694},
{0x1.0080000000000p-806, -0x1.0000000000000p-911, 0x1.00000001fffc0p+613, 0x1.fffff80f80000p+541, 0x1.00000001fffc0p+613, 0x1.fffff80f80000p+541},
{0x1.0080000000000p-806, -0x1.0000000000000p-911, -0x1.00000001fffc0p+613, -0x1.fffff80f80000p+541, -0x1.00000001fffc0p+613, -0x1.fffff80f80000p+541},
{-0x1.0080000000000p-806, 0x1.0000000000000p-911, 0x1.00000001fffc0p+613, 0x1.fffff80f80000p+541, 0x1.00000001fffc0p+613, 0x1.fffff80f80000p+541},
{-0x1.0080000000000p-806, 0x1.0000000000000p-911, -0x1.00000001fffc0p+613, -0x1.fffff80f80000p+541, -0x1.00000001fffc0p+613, -0x1.fffff80f80000p+541},
{0x1.000000003ff00p-698, 0x0.0000000000000p+0, 0x1.0000000000000p-302, 0x1.f83007ffffe00p-362, 0x1.0000000000000p-302, 0x1.f83007ffffe00p-362},
{0x1.000000003ff00p-698, 0x0.0000000000000p+0, -0x1.0000000000000p-302, -0x1.f83007ffffe00p-362, -0x1.0000000000000p-302, -0x1.f83007ffffe00p-362},
{-0x1.000000003ff00p-698, 0x0.0000000000000p+0, 0x1.0000000000000p-302, 0x1.f83007ffffe00p-362, 0x1.0000000000000p-302, 0x1.f83007ffffe00p-362},
{-0x1.000000003ff00p-698, 0x0.0000000000000p+0, -0x1.0000000000000p-302, -0x1.f83007ffffe00p-362, -0x1.0000000000000p-302, -0x1.f83007ffffe00p-362},
{0x1.00003fffffe00p-546, 0x1.fffffc0000380p-606, 0x1.0000010000000p-330, -0x1.ffffff8000008p-386, 0x1.0000010000000p-330, -0x1.ffffff8000008p-386},
{0x1.00003fffffe00p-546, 0x1.fffffc0000380p-606, -0x1.0000010000000p-330, 0x1.ffffff8000008p-386, -0x1.0000010000000p-330, 0x1.ffffff8000008p-386},
{-0x1.00003fffffe00p-546, -0x1.fffffc0000380p-606, 0x1.0000010000000p-330, -0x1.ffffff8000008p-386, 0x1.0000010000000p-330, -0x1.ffffff8000008p-386},
{-0x1.00003fffffe00p-546, -0x1.fffffc0000380p-606, -0x1.0000010000000p-330, 0x1.ffffff8000008p-386, -0x1.0000010000000p-330, 0x1.ffffff8000008p-386},
{0x1.0000000080000p-450, -0x1.fffffc0010000p-519, 0x1.3fffff8000000p-829, 0x1.fffff800fe000p-895, 0x1.0000000080000p-450, -0x1.fffffc0010000p-519},
{0x1.0000000080000p-450, -0x1.fffffc0010000p-519, -0x1.3fffff8000000p-829, -0x1.fffff800fe000p-895, 0x1.0000000080000p-450, -0x1.fffffc0010000p-519},
{-0x1.0000000080000p-450, 0x1.fffffc0010000p-519, 0x1.3fffff8000000p-829, 0x1.fffff800fe000p-895, -0x1.0000000080000p-450, 0x1.fffffc0010000p-519},
{-0x1.0000000080000p-450, 0x1.fffffc0010000p-519, -0x1.3fffff8000000p-829, -0x1.fffff800fe000p-895, -0x1.0000000080000p-450, 0x1.fffffc0010000p-519},
{0x1.0003f001fc3f0p+850, 0x1.fffffc0000000p+794, 0x1.0000000000000p+214, 0x1.fc1fffc0000fep+160, 0x1.0003f001fc3f0p+850, 0x1.fffffc0000000p+794},
{0x1.0003f001fc3f0p+850, 0x1.fffffc0000000p+794, -0x1.0000000000000p+214, -0x1.fc1fffc0000fep+160, 0x1.0003f001fc3f0p+850, 0x1.fffffc0000000p+794},
{-0x1.0003f001fc3f0p+850, -0x1.fffffc0000000p+794, 0x1.0000000000000p+214, 0x1.fc1fffc0000fep+160, -0x1.0003f001fc3f0p+850, -0x1.fffffc0000000p+794},
{-0x1.0003f001fc3f0p+850, -0x1.fffffc0000000p+794, -0x1.0000000000000p+214, -0x1.fc1fffc0000fep+160, -0x1.0003f001fc3f0p+850, -0x1.fffffc0000000p+794},
{0x1.0000000000000p+702, 0x1.fffffffffffc0p+643, 0x1.0000000000000p-646, 0x1.ffffffffffe00p-708, 0x1.0000000000000p+702, 0x1.fffffffffffc0p+643},
{0x1.0000000000000p+702, 0x1.fffffffffffc0p+643, -0x1.0000000000000p-646, -0x1.ffffffffffe00p-708, 0x1.0000000000000p+702, 0x1.fffffffffffc0p+643},
{-0x1.0000000000000p+702, -0x1.fffffffffffc0p+643, 0x1.0000000000000p-646, 0x1.ffffffffffe00p-708, -0x1.0000000000000p+702, -0x1.fffffffffffc0p+643},
{-0x1.0000000000000p+702, -0x1.fffffffffffc0p+643, -0x1.0000000000000p-646, -0x1.ffffffffffe00p-708, -0x1.0000000000000p+702, -0x1.fffffffffffc0p+643},
{0x1.0000000100000p+167, -0x1.0000000000000p+88, 0x1.0000004000000p+455, -0x1.0000000000000p+375, 0x1.0000004000000p+455, -0x1.0000000000000p+375},
{0x1.0000000100000p+167, -0x1.0000000000000p+88, -0x1.0000004000000p+455, 0x1.0000000000000p+375, -0x1.0000004000000p+455, 0x1.0000000000000p+375},
{-0x1.0000000100000p+167, 0x1.0000000000000p+88, 0x1.0000004000000p+455, -0x1.0000000000000p+375, 0x1.0000004000000p+455, -0x1.0000000000000p+375},
{-0x1.0000000100000p+167, 0x1.0000000000000p+88, -0x1.0000004000000p+455, 0x1.0000000000000p+375, -0x1.0000004000000p+455, 0x1.0000000000000p+375},
{0x1.0000010000000p+394, -0x1.ffffe00001000p+338, 0x1.000ffffff007fp-145, 0x1.ffffc00000000p-226, 0x1.0000010000000p+394, -0x1.ffffe00001000p+338},
{0x1.0000010000000p+394, -0x1.ffffe00001000p+338, -0x1.000ffffff007fp-145, -0x1.ffffc00000000p-226, 0x1.0000010000000p+394, -0x1.ffffe00001000p+338},
{-0x1.0000010000000p+394, 0x1.ffffe00001000p+338, 0x1.000ffffff007fp-145, 0x1.ffffc00000000p-226, -0x1.0000010000000p+394, 0x1.ffffe00001000p+338},
{-0x1.0000010000000p+394, 0x1.ffffe00001000p+338, -0x1.000ffffff007fp-145, -0x1.ffffc00000000p-226, -0x1.0000010000000p+394, 0x1.ffffe00001000p+338},
{0x1.1000fffffc000p+16, 0x1.fffffc003f000p-40, 0x1.0000000001000p+243, -0x1.0000000000000p+138, 0x1.0000000001000p+243, -0x1.0000000000000p+138},
{0x1.1000fffffc000p+16, 0x1.fffffc003f000p-40, -0x1.0000000001000p+243, 0x1.0000000000000p+138, -0x1.0000000001000p+243, 0x1.0000000000000p+138},
{-0x1.1000fffffc000p+16, -0x1.fffffc003f000p-40, 0x1.0000000001000p+243, -0x1.0000000000000p+138, 0x1.0000000001000p+243, -0x1.0000000000000p+138},
{-0x1.1000fffffc000p+16, -0x1.fffffc003f000p-40, -0x1.0000000001000p+243, 0x1.0000000000000p+138, -0x1.0000000001000p+243, 0x1.0000000000000p+138},
{0x1.00000fff80000p+494, 0x1.fffe000000000p+409, 0x1.0000001f00000p-623, 0x1.fffffff000000p-700, 0x1.00000fff80000p+494, 0x1.fffe000000000p+409},
{0x1.00000fff80000p+494, 0x1.fffe000000000p+409, -0x1.0000001f00000p-623, -0x1.fffffff000000p-700, 0x1.00000fff80000p+494, 0x1.fffe000000000p+409},
{-0x1.00000fff80000p+494, -0x1.fffe000000000p+409, 0x1.0000001f00000p-623, 0x1.fffffff000000p-700, -0x1.00000fff80000p+494, -0x1.fffe000000000p+409},
{-0x1.00000fff80000p+494, -0x1.fffe000000000p+409, -0x1.0000001f00000p-623, -0x1.fffffff000000p-700, -0x1.00000fff80000p+494, -0x1.fffe000000000p+409},
{0x1.3fdfffff02000p-533, -0x1.0000000000000p-611, 0x1.0000000000000p-602, 0x1.ffffffffffe00p-663, 0x1.3fdfffff02000p-533, 0x1.ff00000000000p-603},
{0x1.3fdfffff02000p-533, -0x1.0000000000000p-611, -0x1.0000000000000p-602, -0x1.ffffffffffe00p-663, 0x1.3fdfffff02000p-533, -0x1.0080000000000p-602},
{-0x1.3fdfffff02000p-533, 0x1.0000000000000p-611, 0x1.0000000000000p-602, 0x1.ffffffffffe00p-663, -0x1.3fdfffff02000p-533, 0x1.0080000000000p-602},
{-0x1.3fdfffff02000p-533, 0x1.0000000000000p-611, -0x1.0000000000000p-602, -0x1.ffffffffffe00p-663, -0x1.3fdfffff02000p-533, -0x1.ff00000000000p-603},
{0x1.0000000000008p-274, -0x1.e0ffffd002000p-340, 0x1.0000000000400p-229, 0x1.ffffffff03ffep-283, 0x1.0000000000480p-229, 0x1.ffffffff04ffep-283},
{0x1.0000000000008p-274, -0x1.e0ffffd002000p-340, -0x1.0000000000400p-229, -0x1.ffffffff03ffep-283, -0x1.0000000000380p-229, -0x1.ffffffff02ffep-283},
{-0x1.0000000000008p-274, 0x1.e0ffffd002000p-340, 0x1.0000000000400p-229, 0x1.ffffffff03ffep-283, 0x1.0000000000380p-229, 0x1.ffffffff02ffep-283},
{-0x1.0000000000008p-274, 0x1.e0ffffd002000p-340, -0x1.0000000000400p-229, -0x1.ffffffff03ffep-283, -0x1.0000000000480p-229, -0x1.ffffffff04ffep-283},
{0x1.000000fffe020p-113, -0x1.f800100000000p-186, 0x1.0001fff80ffc0p+13, 0x1.ff00000000000p-84, 0x1.0001fff80ffc0p+13, 0x1.ff00000000000p-84},
{0x1.000000fffe020p-113, -0x1.f800100000000p-186, -0x1.0001fff80ffc0p+13, -0x1.ff00000000000p-84, -0x1.0001fff80ffc0p+13, -0x1.ff00000000000p-84},
{-0x1.000000fffe020p-113, 0x1.f800100000000p-186, 0x1.0001fff80ffc0p+13, 0x1.ff00000000000p-84, 0x1.0001fff80ffc0p+13, 0x1.ff00000000000p-84},
{-0x1.000000fffe020p-113, 0x1.f800100000000p-186, -0x1.0001fff80ffc0p+13, -0x1.ff00000000000p-84, -0x1.0001fff80ffc0p+13, -0x1.ff00000000000p-84},
{0x1.4000000000000p+455, -0x1.0000000000000p+350, 0x1.0000000000010p-968, 0x1.e000000000000p-1036, 0x1.4000000000000p+455, -0x1.0000000000000p+350},
{0x1.4000000000000p+455, -0x1.0000000000000p+350, -0x1.0000000000010p-968, -0x1.e000000000000p-1036, 0x1.4000000000000p+455, -0x1.0000000000000p+350},
{-0x1.4000000000000p+455, 0x1.0000000000000p+350, 0x1.0000000000010p-968, 0x1.e000000000000p-1036, -0x1.4000000000000p+455, 0x1.0000000000000p+350},
{-0x1.4000000000000p+455, 0x1.0000000000000p+350, -0x1.0000000000010p-968, -0x1.e000000000000p-1036, -0x1.4000000000000p+455, 0x1.0000000000000p+350},
{0x1.2000000000000p+333, -0x1.fffffff01ff20p+276, 0x1.0000000000000p+701, 0x1.fff8003ff0000p+644, 0x1.0000000000000p+701, 0x1.fff8003ff0000p+644},
{0x1.2000000000000p+333, -0x1.fffffff01ff20p+276, -0x1.0000000000000p+701, -0x1.fff8003ff0000p+644, -0x1.0000000000000p+701, -0x1.fff8003ff0000p+644},
{-0x1.2000000000000p+333, 0x1.fffffff01ff20p+276, 0x1.0000000000000p+701, 0x1.fff8003ff0000p+644, 0x1.0000000000000p+701, 0x1.fff8003ff0000p+644},
{-0x1.2000000000000p+333, 0x1.fffffff01ff20p+276, -0x1.0000000000000p+701, -0x1.fff8003ff0000p+644, -0x1.0000000000000p+701, -0x1.fff8003ff0000p+644},
{0x1.2000000000000p+275, -0x1.fc08000000000p+183, 0x1.0000000038004p+750, -0x1.fffffc4000000p+681, 0x1.0000000038004p+750, -0x1.fffffc4000000p+681},
{0x1.2000000000000p+275, -0x1.fc08000000000p+183, -0x1.0000000038004p+750, 0x1.fffffc4000000p+681, -0x1.0000000038004p+750, 0x1.fffffc4000000p+681},
{-0x1.2000000000000p+275, 0x1.fc08000000000p+183, 0x1.0000000038004p+750, -0x1.fffffc4000000p+681, 0x1.0000000038004p+750, -0x1.fffffc4000000p+681},
{-0x1.2000000000000p+275, 0x1.fc08000000000p+183, -0x1.0000000038004p+750, 0x1.fffffc4000000p+681, -0x1.0000000038004p+750, 0x1.fffffc4000000p+681},
{0x1.0000000000200p+680, 0x0.0000000000000p+0, 0x1.00001ffff0000p-146, 0x1.ffff80001f000p-203, 0x1.0000000000200p+680, 0x0.0000000000000p+0},
{0x1.0000000000200p+680, 0x0.0000000000000p+0, -0x1.00001ffff0000p-146, -0x1.ffff80001f000p-203, 0x1.0000000000200p+680, 0x0.0000000000000p+0},
{-0x1.0000000000200p+680, 0x0.0000000000000p+0, 0x1.00001ffff0000p-146, 0x1.ffff80001f000p-203, -0x1.0000000000200p+680, 0x0.0000000000000p+0},
{-0x1.0000000000200p+680, 0x0.0000000000000p+0, -0x1.00001ffff0000p-146, -0x1.ffff80001f000p-203, -0x1.0000000000200p+680, 0x0.0000000000000p+0},
{0x1.0000100000000p-400, -0x1.0000000000000p-505, 0x1.0038000800000p-754, -0x1.fffffe0000080p-814, 0x1.0000100000000p-400, -0x1.0000000000000p-505},
{0x1.0000100000000p-400, -0x1.0000000000000p-505, -0x1.0038000800000p-754, 0x1.fffffe0000080p-814, 0x1.0000100000000p-400, -0x1.0000000000000p-505},
{-0x1.0000100000000p-400, 0x1.0000000000000p-505, 0x1.0038000800000p-754, -0x1.fffffe0000080p-814, -0x1.0000100000000p-400, 0x1.0000000000000p-505},
{-0x1.0000100000000p-400, 0x1.0000000000000p-505, -0x1.0038000800000p-754, 0x1.fffffe0000080p-814, -0x1.0000100000000p-400, 0x1.0000000000000p-505},
{0x1.0000000000004p+838, -0x1.ffffe00400000p+765, 0x1.0000000000000p-355, 0x1.ffffffffffff8p-411, 0x1.0000000000004p+838, -0x1.ffffe00400000p+765},
{0x1.0000000000004p+838, -0x1.ffffe00400000p+765, -0x1.0000000000000p-355, -0x1.ffffffffffff8p-411, 0x1.0000000000004p+838, -0x1.ffffe00400000p+765},
{-0x1.0000000000004p+838, 0x1.ffffe00400000p+765, 0x1.0000000000000p-355, 0x1.ffffffffffff8p-411, -0x1.0000000000004p+838, 0x1.ffffe00400000p+765},
{-0x1.0000000000004p+838, 0x1.ffffe00400000p+765, -0x1.0000000000000p-355, -0x1.ffffffffffff8p-411, -0x1.0000000000004p+838, 0x1.ffffe00400000p+765},
{0x1.0000000000100p+877, -0x1.ffffe7e400000p+807, 0x1.1ffe000008000p+922, -0x1.fffc000004000p+855, 0x1.1ffe000008080p+922, -0x1.fffbfc0004000p+855},
{0x1.0000000000100p+877, -0x1.ffffe7e400000p+807, -0x1.1ffe000008000p+922, 0x1.fffc000004000p+855, -0x1.1ffe000007f80p+922, 0x1.fffc040004000p+855},
{-0x1.0000000000100p+877, 0x1.ffffe7e400000p+807, 0x1.1ffe000008000p+922, -0x1.fffc000004000p+855, 0x1.1ffe000007f80p+922, -0x1.fffc040004000p+855},
{-0x1.0000000000100p+877, 0x1.ffffe7e400000p+807, -0x1.1ffe000008000p+922, 0x1.fffc000004000p+855, -0x1.1ffe000008080p+922, 0x1.fffbfc0004000p+855},
{0x1.0000000000004p-490, -0x1.0000000000000p-595, 0x1.00000007ffffep-352, 0x1.ffffe000f0000p-413, 0x1.00000007ffffep-352, 0x1.ffffe000f0000p-413},
{0x1.0000000000004p-490, -0x1.0000000000000p-595, -0x1.00000007ffffep-352, -0x1.ffffe000f0000p-413, -0x1.00000007ffffep-352, -0x1.ffffe000f0000p-413},
{-0x1.0000000000004p-490, 0x1.0000000000000p-595, 0x1.00000007ffffep-352, 0x1.ffffe000f0000p-413, 0x1.00000007ffffep-352, 0x1.ffffe000f0000p-413},
{-0x1.0000000000004p-490, 0x1.0000000000000p-595, -0x1.00000007ffffep-352, -0x1.ffffe000f0000p-413, -0x1.00000007ffffep-352, -0x1.ffffe000f0000p-413},
{0x1.0000000000000p-458, 0x1.fffffff000000p-519, 0x1.0001fffc00001p+791, -0x1.0000000000000p+712, 0x1.0001fffc00001p+791, -0x1.0000000000000p+712},
{0x1.0000000000000p-458, 0x1.fffffff000000p-519, -0x1.0001fffc00001p+791, 0x1.0000000000000p+712, -0x1.0001fffc00001p+791, 0x1.0000000000000p+712},
{-0x1.0000000000000p-458, -0x1.fffffff000000p-519, 0x1.0001fffc00001p+791, -0x1.0000000000000p+712, 0x1.0001fffc00001p+791, -0x1.0000000000000p+712},
{-0x1.0000000000000p-458, -0x1.fffffff000000p-519, -0x1.0001fffc00001p+791, 0x1.0000000000000p+712, -0x1.0001fffc00001p+791, 0x1.0000000000000p+712},
{0x1.00007fdfffff0p-745, 0x1.fff000f803fe0p-803, 0x1.0080000000000p-983, -0x0.0000000000000p+0, 0x1.00007fdfffff0p-745, 0x1.fff000f803fe0p-803},
{0x1.00007fdfffff0p-745, 0x1.fff000f803fe0p-803, -0x1.0080000000000p-983, 0x0.0000000000000p+0, 0x1.00007fdfffff0p-745, 0x1.fff000f803fe0p-803},
{-0x1.00007fdfffff0p-745, -0x1.fff000f803fe0p-803, 0x1.0080000000000p-983, -0x0.0000000000000p+0, -0x1.00007fdfffff0p-745, -0x1.fff000f803fe0p-803},
{-0x1.00007fdfffff0p-745, -0x1.fff000f803fe0p-803, -0x1.0080000000000p-983, 0x0.0000000000000p+0, -0x1.00007fdfffff0p-745, -0x1.fff000f803fe0p-803},
{0x1.0000000000080p-788, 0x0.0000000000000p+0, 0x1.0000080000000p+973, -0x1.0000000000000p+888, 0x1.0000080000000p+973, -0x1.0000000000000p+888},
{0x1.0000000000080p-788, 0x0.0000000000000p+0, -0x1.0000080000000p+973, 0x1.0000000000000p+888, -0x1.0000080000000p+973, 0x1.0000000000000p+888},
{-0x1.0000000000080p-788, 0x0.0000000000000p+0, 0x1.0000080000000p+973, -0x1.0000000000000p+888, 0x1.0000080000000p+973, -0x1.0000000000000p+888},
{-0x1.0000000000080p-788, 0x0.0000000000000p+0, -0x1.0000080000000p+973, 0x1.0000000000000p+888, -0x1.0000080000000p+973, 0x1.0000000000000p+888},
{0x1.0400000000000p-287, -0x1.0000000000000p-392, 0x1.0000000000400p+736, -0x1.fffff00008000p+680, 0x1.0000000000400p+736, -0x1.fffff00008000p+680},
{0x1.0400000000000p-287, -0x1.0000000000000p-392, -0x1.0000000000400p+736, 0x1.fffff00008000p+680, -0x1.0000000000400p+736, 0x1.fffff00008000p+680},
{-0x1.0400000000000p-287, 0x1.0000000000000p-392, 0x1.0000000000400p+736, -0x1.fffff00008000p+680, 0x1.0000000000400p+736, -0x1.fffff00008000p+680},
{-0x1.0400000000000p-287, 0x1.0000000000000p-392, -0x1.0000000000400p+736, 0x1.fffff00008000p+680, -0x1.0000000000400p+736, 0x1.fffff00008000p+680},
{0x1.0000000000001p+232, -0x1.fc0000c004000p+167, 0x1.0000000000200p+192, -0x1.ffe0000200000p+124, 0x1.0000000001001p+232, -0x1.fbffc0c004000p+167},
{0x1.0000000000001p+232, -0x1.fc0000c004000p+167, -0x1.0000000000200p+192, 0x1.ffe0000200000p+124, 0x1.fffffffffe002p+231, -0x1.fc0040c004000p+167},
{-0x1.0000000000001p+232, 0x1.fc0000c004000p+167, 0x1.0000000000200p+192, -0x1.ffe0000200000p+124, -0x1.fffffffffe002p+231, 0x1.fc0040c004000p+167},
{-0x1.0000000000001p+232, 0x1.fc0000c004000p+167, -0x1.0000000000200p+192, 0x1.ffe0000200000p+124, -0x1.0000000001001p+232, 0x1.fbffc0c004000p+167},
{0x1.00000c0000004p+583, -0x1.ff80000800000p+515, 0x1.18003ffff8000p-786, 0x1.ffffff8000000p-852, 0x1.00000c0000004p+583, -0x1.ff80000800000p+515},
{0x1.00000c0000004p+583, -0x1.ff80000800000p+515, -0x1.18003ffff8000p-786, -0x1.ffffff8000000p-852, 0x1.00000c0000004p+583, -0x1.ff80000800000p+515},
{-0x1.00000c0000004p+583, 0x1.ff80000800000p+515, 0x1.18003ffff8000p-786, 0x1.ffffff8000000p-852, -0x1.00000c0000004p+583, 0x1.ff80000800000p+515},
{-0x1.00000c0000004p+583, 0x1.ff80000800000p+515, -0x1.18003ffff8000p-786, -0x1.ffffff8000000p-852, -0x1.00000c0000004p+583, 0x1.ff80000800000p+515},
{0x1.1ffff80000040p+488, -0x1.0000000000000p+398, 0x1.000000000fe00p-510, 0x1.ffff7e0003ff8p-566, 0x1.1ffff80000040p+488, -0x1.0000000000000p+398},
{0x1.1ffff80000040p+488, -0x1.0000000000000p+398, -0x1.000000000fe00p-510, -0x1.ffff7e0003ff8p-566, 0x1.1ffff80000040p+488, -0x1.0000000000000p+398},
{-0x1.1ffff80000040p+488, 0x1.0000000000000p+398, 0x1.000000000fe00p-510, 0x1.ffff7e0003ff8p-566, -0x1.1ffff80000040p+488, 0x1.0000000000000p+398},
{-0x1.1ffff80000040p+488, 0x1.0000000000000p+398, -0x1.000000000fe00p-510, -0x1.ffff7e0003ff8p-566, -0x1.1ffff80000040p+488, 0x1.0000000000000p+398},
{0x1.3fc0000040000p+871, -0x1.fffffff802000p+815, 0x1.fffffffffff00p-663, 0x0.0000000000000p+0, 0x1.3fc0000040000p+871, -0x1.fffffff802000p+815},
{0x1.3fc0000040000p+871, -0x1.fffffff802000p+815, -0x1.fffffffffff00p-663, 0x0.0000000000000p+0, 0x1.3fc0000040000p+871, -0x1.fffffff802000p+815},
{-0x1.3fc0000040000p+871, 0x1.fffffff802000p+815, 0x1.fffffffffff00p-663, 0x0.0000000000000p+0, -0x1.3fc0000040000p+871, 0x1.fffffff802000p+815},
{-0x1.3fc0000040000p+871, 0x1.fffffff802000p+815, -0x1.fffffffffff00p-663, 0x0.0000000000000p+0, -0x1.3fc0000040000p+871, 0x1.fffffff802000p+815},
{0x1.00000001fffffp-628, 0x1.00001f8000000p-699, 0x1.0000000000200p-312, -0x1.0000000000000p-417, 0x1.0000000000200p-312, -0x1.0000000000000p-417},
{0x1.00000001fffffp-628, 0x1.00001f8000000p-699, -0x1.0000000000200p-312, 0x1.0000000000000p-417, -0x1.0000000000200p-312, 0x1.0000000000000p-417},
{-0x1.00000001fffffp-628, -0x1.00001f8000000p-699, 0x1.0000000000200p-312, -0x1.0000000000000p-417, 0x1.0000000000200p-312, -0x1.0000000000000p-417},
{-0x1.00000001fffffp-628, -0x1.00001f8000000p-699, -0x1.0000000000200p-312, 0x1.0000000000000p-417, -0x1.0000000000200p-312, 0x1.0000000000000p-417},
{0x1.0000000010000p+23, -0x1.0000000000000p-54, 0x1.1fff010000000p+322, 0x1.ff8c0000007fep+268, 0x1.1fff010000000p+322, 0x1.ff8c0000007fep+268},
{0x1.0000000010000p+23, -0x1.0000000000000p-54, -0x1.1fff010000000p+322, -0x1.ff8c0000007fep+268, -0x1.1fff010000000p+322, -0x1.ff8c0000007fep+268},
{-0x1.0000000010000p+23, 0x1.0000000000000p-54, 0x1.1fff010000000p+322, 0x1.ff8c0000007fep+268, 0x1.1fff010000000p+322, 0x1.ff8c0000007fep+268},
{-0x1.0000000010000p+23, 0x1.0000000000000p-54, -0x1.1fff010000000p+322, -0x1.ff8c0000007fep+268, -0x1.1fff010000000p+322, -0x1.ff8c0000007fep+268},
{0x1.0000000000002p-4, -0x1.0000000000000p-91, 0x1.0000000000200p+643, -0x1.0000000000000p+558, 0x1.0000000000200p+643, -0x1.0000000000000p+558},
{0x1.0000000000002p-4, -0x1.0000000000000p-91, -0x1.0000000000200p+643, 0x1.0000000000000p+558, -0x1.0000000000200p+643, 0x1.0000000000000p+558},
{-0x1.0000000000002p-4, 0x1.0000000000000p-91, 0x1.0000000000200p+643, -0x1.0000000000000p+558, 0x1.0000000000200p+643, -0x1.0000000000000p+558},
{-0x1.0000000000002p-4, 0x1.0000000000000p-91, -0x1.0000000000200p+643, 0x1.0000000000000p+558, -0x1.0000000000200p+643, 0x1.0000000000000p+558},
{0x1.0000000000002p-252, -0x1.ffff800008000p-319, 0x1.000000003fffep-236, 0x1.f81fc007c0000p-307, 0x1.000100003fffep-236, 0x1.1f7ffc00fc000p-303},
{0x1.0000000000002p-252, -0x1.ffff800008000p-319, -0x1.000000003fffep-236, -0x1.f81fc007c0000p-307, -0x1.fffe00007fffcp-237, 0x1.c0f8080008000p-304},
{-0x1.0000000000002p-252, 0x1.ffff800008000p-319, 0x1.000000003fffep-236, 0x1.f81fc007c0000p-307, 0x1.fffe00007fffcp-237, -0x1.c0f8080008000p-304},
{-0x1.0000000000002p-252, 0x1.ffff800008000p-319, -0x1.000000003fffep-236, -0x1.f81fc007c0000p-307, -0x1.000100003fffep-236, -0x1.1f7ffc00fc000p-303},
{0x1.7ffffffff0000p-1014, 0x0.0000000000000p+0, 0x1.000000000ffe0p-894, 0x1.ff0000003c000p-957, 0x1.000000000ffe0p-894, 0x1.ff0000003c000p-957},
{0x1.7ffffffff0000p-1014, 0x0.0000000000000p+0, -0x1.000000000ffe0p-894, -0x1.ff0000003c000p-957, -0x1.000000000ffe0p-894, -0x1.ff0000003c000p-957},
{-0x1.7ffffffff0000p-1014, -0x0.0000000000000p+0, 0x1.000000000ffe0p-894, 0x1.ff0000003c000p-957, 0x1.000000000ffe0p-894, 0x1.ff0000003c000p-957},
{-0x1.7ffffffff0000p-1014, -0x0.0000000000000p+0, -0x1.000000000ffe0p-894, -0x1.ff0000003c000p-957, -0x1.000000000ffe0p-894, -0x1.ff0000003c000p-957},
{0x1.00007fff00080p+981, -0x1.ffffff0001000p+916, 0x1.00000000007c0p-918, 0x1.ffffff8000000p-985, 0x1.00007fff00080p+981, -0x1.ffffff0001000p+916},
{0x1.00007fff00080p+981, -0x1.ffffff0001000p+916, -0x1.00000000007c0p-918, -0x1.ffffff8000000p-985, 0x1.00007fff00080p+981, -0x1.ffffff0001000p+916},
{-0x1.00007fff00080p+981, 0x1.ffffff0001000p+916, 0x1.00000000007c0p-918, 0x1.ffffff8000000p-985, -0x1.00007fff00080p+981, 0x1.ffffff0001000p+916},
{-0x1.00007fff00080p+981, 0x1.ffffff0001000p+916, -0x1.00000000007c0p-918, -0x1.ffffff8000000p-985, -0x1.00007fff00080p+981, 0x1.ffffff0001000p+916},
{0x1.1fff81ff00000p-768, 0x1.e007ffff00000p-825, 0x1.1ffffffff8000p-236, 0x1.fe001ffffffe0p-292, 0x1.1ffffffff8000p-236, 0x1.fe001ffffffe0p-292},
{0x1.1fff81ff00000p-768, 0x1.e007ffff00000p-825, -0x1.1ffffffff8000p-236, -0x1.fe001ffffffe0p-292, -0x1.1ffffffff8000p-236, -0x1.fe001ffffffe0p-292},
{-0x1.1fff81ff00000p-768, -0x1.e007ffff00000p-825, 0x1.1ffffffff8000p-236, 0x1.fe001ffffffe0p-292, 0x1.1ffffffff8000p-236, 0x1.fe001ffffffe0p-292},
{-0x1.1fff81ff00000p-768, -0x1.e007ffff00000p-825, -0x1.1ffffffff8000p-236, -0x1.fe001ffffffe0p-292, -0x1.1ffffffff8000p-236, -0x1.fe001ffffffe0p-292},
{0x1.0000000000000p-141, 0x1.fffc03fe00700p-202, 0x1.0000000000001p-845, -0x1.fff8060000f34p-900, 0x1.0000000000000p-141, 0x1.fffc03fe00700p-202},
{0x1.0000000000000p-141, 0x1.fffc03fe00700p-202, -0x1.0000000000001p-845, 0x1.fff8060000f34p-900, 0x1.0000000000000p-141, 0x1.fffc03fe00700p-202},
{-0x1.0000000000000p-141, -0x1.fffc03fe00700p-202, 0x1.0000000000001p-845, -0x1.fff8060000f34p-900, -0x1.0000000000000p-141, -0x1.fffc03fe00700p-202},
{-0x1.0000000000000p-141, -0x1.fffc03fe00700p-202, -0x1.0000000000001p-845, 0x1.fff8060000f34p-900, -0x1.0000000000000p-141, -0x1.fffc03fe00700p-202},
{0x1.001ffffc00080p+872, -0x1.fe00000200000p+811, 0x1.0000000001000p-359, -0x1.fe3ffffff0080p-419, 0x1.001ffffc00080p+872, -0x1.fe00000200000p+811},
{0x1.001ffffc00080p+872, -0x1.fe00000200000p+811, -0x1.0000000001000p-359, 0x1.fe3ffffff0080p-419, 0x1.001ffffc00080p+872, -0x1.fe00000200000p+811},
{-0x1.001ffffc00080p+872, 0x1.fe00000200000p+811, 0x1.0000000001000p-359, -0x1.fe3ffffff0080p-419, -0x1.001ffffc00080p+872, 0x1.fe00000200000p+811},
{-0x1.001ffffc00080p+872, 0x1.fe00000200000p+811, -0x1.0000000001000p-359, 0x1.fe3ffffff0080p-419, -0x1.001ffffc00080p+872, 0x1.fe00000200000p+811},
{0x1.00001fffff800p-894, 0x1.ffffff801ffc0p-953, 0x1.0000000002000p-720, -0x1.0000000000000p-825, 0x1.0000000002000p-720, -0x1.0000000000000p-825},
{0x1.00001fffff800p-894, 0x1.ffffff801ffc0p-953, -0x1.0000000002000p-720, 0x1.0000000000000p-825, -0x1.0000000002000p-720, 0x1.0000000000000p-825},
{-0x1.00001fffff800p-894, -0x1.ffffff801ffc0p-953, 0x1.0000000002000p-720, -0x1.0000000000000p-825, 0x1.0000000002000p-720, -0x1.0000000000000p-825},
{-0x1.00001fffff800p-894, -0x1.ffffff801ffc0p-953, -0x1.0000000002000p-720, 0x1.0000000000000p-825, -0x1.0000000002000p-720, 0x1.0000000000000p-825},
{0x1.0000000000000p-30, 0x1.ffff83ffbfff0p-86, 0x1.0000000000000p-240, 0x1.ffffffffe0000p-296, 0x1.0000000000000p-30, 0x1.ffff83ffbfff0p-86},
{0x1.0000000000000p-30, 0x1.ffff83ffbfff0p-86, -0x1.0000000000000p-240, -0x1.ffffffffe0000p-296, 0x1.0000000000000p-30, 0x1.ffff83ffbfff0p-86},
{-0x1.0000000000000p-30, -0x1.ffff83ffbfff0p-86, 0x1.0000000000000p-240, 0x1.ffffffffe0000p-296, -0x1.0000000000000p-30, -0x1.ffff83ffbfff0p-86},
{-0x1.0000000000000p-30, -0x1.ffff83ffbfff0p-86, -0x1.0000000000000p-240, -0x1.ffffffffe0000p-296, -0x1.0000000000000p-30, -0x1.ffff83ffbfff0p-86},
{0x1.0000000000000p-874, 0x1.fffe7ff3dff00p-935, 0x1.000007f830000p-977, 0x1.fff0001000000p-1038, 0x1.0000000000000p-874, 0x1.fffe7ff3e0300p-935},
{0x1.0000000000000p-874, 0x1.fffe7ff3dff00p-935, -0x1.000007f830000p-977, -0x1.fff0001000000p-1038, 0x1.0000000000000p-874, 0x1.fffe7ff3dfb00p-935},
{-0x1.0000000000000p-874, -0x1.fffe7ff3dff00p-935, 0x1.000007f830000p-977, 0x1.fff0001000000p-1038, -0x1.0000000000000p-874, -0x1.fffe7ff3dfb00p-935},
{-0x1.0000000000000p-874, -0x1.fffe7ff3dff00p-935, -0x1.000007f830000p-977, -0x1.fff0001000000p-1038, -0x1.0000000000000p-874, -0x1.fffe7ff3e0300p-935},
{0x1.00000ffffe000p-269, 0x1.fffffffffc000p-336, 0x1.0000000000002p+226, -0x1.fffffe01f8004p+171, 0x1.0000000000002p+226, -0x1.fffffe01f8004p+171},
{0x1.00000ffffe000p-269, 0x1.fffffffffc000p-336, -0x1.0000000000002p+226, 0x1.fffffe01f8004p+171, -0x1.0000000000002p+226, 0x1.fffffe01f8004p+171},
{-0x1.00000ffffe000p-269, -0x1.fffffffffc000p-336, 0x1.0000000000002p+226, -0x1.fffffe01f8004p+171, 0x1.0000000000002p+226, -0x1.fffffe01f8004p+171},
{-0x1.00000ffffe000p-269, -0x1.fffffffffc000p-336, -0x1.0000000000002p+226, 0x1.fffffe01f8004p+171, -0x1.0000000000002p+226, 0x1.fffffe01f8004p+171},
{0x1.0000000000060p+403, 0x1.fffe3fff00000p+340, 0x1.0000000002000p+783, -0x1.ffe0002000000p+726, 0x1.0000000002000p+783, -0x1.ffe0002000000p+726},
{0x1.0000000000060p+403, 0x1.fffe3fff00000p+340, -0x1.0000000002000p+783, 0x1.ffe0002000000p+726, -0x1.0000000002000p+783, 0x1.ffe0002000000p+726},
{-0x1.0000000000060p+403, -0x1.fffe3fff00000p+340, 0x1.0000000002000p+783, -0x1.ffe0002000000p+726, 0x1.0000000002000p+783, -0x1.ffe0002000000p+726},
{-0x1.0000000000060p+403, -0x1.fffe3fff00000p+340, -0x1.0000000002000p+783, 0x1.ffe0002000000p+726, -0x1.0000000002000p+783, 0x1.ffe0002000000p+726},
{0x1.0003ffffffff8p-273, 0x1.ffffe07e00000p-347, 0x1.07ff000008000p-591, -0x1.e000020000000p-652, 0x1.0003ffffffff8p-273, 0x1.ffffe07e00000p-347},
{0x1.0003ffffffff8p-273, 0x1.ffffe07e00000p-347, -0x1.07ff000008000p-591, 0x1.e000020000000p-652, 0x1.0003ffffffff8p-273, 0x1.ffffe07e00000p-347},
{-0x1.0003ffffffff8p-273, -0x1.ffffe07e00000p-347, 0x1.07ff000008000p-591, -0x1.e000020000000p-652, -0x1.0003ffffffff8p-273, -0x1.ffffe07e00000p-347},
{-0x1.0003ffffffff8p-273, -0x1.ffffe07e00000p-347, -0x1.07ff000008000p-591, 0x1.e000020000000p-652, -0x1.0003ffffffff8p-273, -0x1.ffffe07e00000p-347},
// Some test cases where a and b are guaranteed to overlap:
{0x1.0000000000002p-39, -0x1.ffc0ffff90000p-103, 0x1.0003ffffffffcp-35, 0x1.ffffc00000000p-122, 0x1.1003ffffffffcp-35, 0x1.ffe003f400068p-91},
{0x1.0000000000002p-39, -0x1.ffc0ffff90000p-103, -0x1.0003ffffffffcp-35, -0x1.ffffc00000000p-122, -0x1.e007ffffffff8p-36, 0x1.ffe003ec00078p-91},
{-0x1.0000000000002p-39, 0x1.ffc0ffff90000p-103, 0x1.0003ffffffffcp-35, 0x1.ffffc00000000p-122, 0x1.e007ffffffff8p-36, -0x1.ffe003ec00078p-91},
{-0x1.0000000000002p-39, 0x1.ffc0ffff90000p-103, -0x1.0003ffffffffcp-35, -0x1.ffffc00000000p-122, -0x1.1003ffffffffcp-35, -0x1.ffe003f400068p-91},
{0x1.0000100000000p+23, -0x1.0000000000000p-39, 0x1.0000000004000p-41, -0x1.ffffe00fc0200p-97, 0x1.0000100000000p+23, -0x1.7ffffffffe000p-40},
{0x1.0000100000000p+23, -0x1.0000000000000p-39, -0x1.0000000004000p-41, 0x1.ffffe00fc0200p-97, 0x1.0000100000000p+23, -0x1.4000000001000p-39},
{-0x1.0000100000000p+23, 0x1.0000000000000p-39, 0x1.0000000004000p-41, -0x1.ffffe00fc0200p-97, -0x1.0000100000000p+23, 0x1.4000000001000p-39},
{-0x1.0000100000000p+23, 0x1.0000000000000p-39, -0x1.0000000004000p-41, 0x1.ffffe00fc0200p-97, -0x1.0000100000000p+23, 0x1.7ffffffffe000p-40},
{0x1.0000000800000p-42, -0x1.0000000000000p-114, 0x1.000003fffc002p-41, -0x1.0000000000000p-146, 0x1.80000403fc002p-41, -0x1.0000000100000p-114},
{0x1.0000000800000p-42, -0x1.0000000000000p-114, -0x1.000003fffc002p-41, 0x1.0000000000000p-146, -0x1.000007f7f8004p-42, -0x1.fffffffe00000p-115},
{-0x1.0000000800000p-42, 0x1.0000000000000p-114, 0x1.000003fffc002p-41, -0x1.0000000000000p-146, 0x1.000007f7f8004p-42, 0x1.fffffffe00000p-115},
{-0x1.0000000800000p-42, 0x1.0000000000000p-114, -0x1.000003fffc002p-41, 0x1.0000000000000p-146, -0x1.80000403fc002p-41, 0x1.0000000100000p-114},
{0x1.0000000000000p+46, 0x1.ffc001ffffe00p-9, 0x1.000003fff8000p-19, 0x1.fffc000fc0000p-90, 0x1.0000000000000p+46, 0x1.000001007fef0p-8},
{0x1.0000000000000p+46, 0x1.ffc001ffffe00p-9, -0x1.000003fff8000p-19, -0x1.fffc000fc0000p-90, 0x1.0000000000000p+46, 0x1.ff8001feffe20p-9},
{-0x1.0000000000000p+46, -0x1.ffc001ffffe00p-9, 0x1.000003fff8000p-19, 0x1.fffc000fc0000p-90, -0x1.0000000000000p+46, -0x1.ff8001feffe20p-9},
{-0x1.0000000000000p+46, -0x1.ffc001ffffe00p-9, -0x1.000003fff8000p-19, -0x1.fffc000fc0000p-90, -0x1.0000000000000p+46, -0x1.000001007fef0p-8},
{0x1.0000000000003p+33, 0x1.fc00001fff800p-31, 0x1.0000000000020p+40, -0x1.fc01ffe002000p-24, 0x1.0200000000020p+40, 0x1.781fd80080f80p-18},
{0x1.0000000000003p+33, 0x1.fc00001fff800p-31, -0x1.0000000000020p+40, 0x1.fc01ffe002000p-24, -0x1.fc00000000040p+39, 0x1.87ffe7ff81080p-18},
{-0x1.0000000000003p+33, -0x1.fc00001fff800p-31, 0x1.0000000000020p+40, -0x1.fc01ffe002000p-24, 0x1.fc00000000040p+39, -0x1.87ffe7ff81080p-18},
{-0x1.0000000000003p+33, -0x1.fc00001fff800p-31, -0x1.0000000000020p+40, 0x1.fc01ffe002000p-24, -0x1.0200000000020p+40, -0x1.781fd80080f80p-18},
{0x1.0000000000000p-42, 0x1.fffff00001ffep-96, 0x1.0000000000000p-9, 0x1.ff00000ffff80p-69, 0x1.0000000080000p-9, 0x1.ff00004ffff80p-69},
{0x1.0000000000000p-42, 0x1.fffff00001ffep-96, -0x1.0000000000000p-9, -0x1.ff00000ffff80p-69, -0x1.ffffffff00000p-10, -0x1.feffffcffff80p-69},
{-0x1.0000000000000p-42, -0x1.fffff00001ffep-96, 0x1.0000000000000p-9, 0x1.ff00000ffff80p-69, 0x1.ffffffff00000p-10, 0x1.feffffcffff80p-69},
{-0x1.0000000000000p-42, -0x1.fffff00001ffep-96, -0x1.0000000000000p-9, -0x1.ff00000ffff80p-69, -0x1.0000000080000p-9, -0x1.ff00004ffff80p-69},
{0x1.0000000000000p-46, 0x1.ffffffffffc00p-109, 0x1.0000200000000p+11, -0x1.ffc0000080000p-61, 0x1.0000200000000p+11, 0x1.fff800fffffe0p-47},
{0x1.0000000000000p-46, 0x1.ffffffffffc00p-109, -0x1.0000200000000p+11, 0x1.ffc0000080000p-61, -0x1.0000200000000p+11, 0x1.0003ff8000010p-46},
{-0x1.0000000000000p-46, -0x1.ffffffffffc00p-109, 0x1.0000200000000p+11, -0x1.ffc0000080000p-61, 0x1.0000200000000p+11, -0x1.0003ff8000010p-46},
{-0x1.0000000000000p-46, -0x1.ffffffffffc00p-109, -0x1.0000200000000p+11, 0x1.ffc0000080000p-61, -0x1.0000200000000p+11, -0x1.fff800fffffe0p-47},
{0x1.0000000000800p-4, -0x1.fffc000000010p-61, 0x1.0000000000008p-2, -0x1.ffe03fffff010p-59, 0x1.4000000000208p-2, -0x1.3fef9fffff808p-58},
{0x1.0000000000800p-4, -0x1.fffc000000010p-61, -0x1.0000000000008p-2, 0x1.ffe03fffff010p-59, -0x1.7fffffffffc10p-3, 0x1.7fe13fffff010p-59},
{-0x1.0000000000800p-4, 0x1.fffc000000010p-61, 0x1.0000000000008p-2, -0x1.ffe03fffff010p-59, 0x1.7fffffffffc10p-3, -0x1.7fe13fffff010p-59},
{-0x1.0000000000800p-4, 0x1.fffc000000010p-61, -0x1.0000000000008p-2, 0x1.ffe03fffff010p-59, -0x1.4000000000208p-2, 0x1.3fef9fffff808p-58},
{0x1.0000000000000p-39, 0x1.fffef0ffff000p-102, 0x1.0000100000000p-7, -0x1.fff8000000800p-71, 0x1.0000100100000p-7, -0x1.fff7fffc00800p-71},
{0x1.0000000000000p-39, 0x1.fffef0ffff000p-102, -0x1.0000100000000p-7, 0x1.fff8000000800p-71, -0x1.00000fff00000p-7, 0x1.fff8000400800p-71},
{-0x1.0000000000000p-39, -0x1.fffef0ffff000p-102, 0x1.0000100000000p-7, -0x1.fff8000000800p-71, 0x1.00000fff00000p-7, -0x1.fff8000400800p-71},
{-0x1.0000000000000p-39, -0x1.fffef0ffff000p-102, -0x1.0000100000000p-7, 0x1.fff8000000800p-71, -0x1.0000100100000p-7, 0x1.fff7fffc00800p-71},
{0x1.07c00007ffff0p-15, 0x1.fffc00f800000p-91, 0x1.0000007ffffe0p-11, 0x1.f000000000000p-82, 0x1.107c00807ffdfp-11, 0x1.f0fffe0080000p-82},
{0x1.07c00007ffff0p-15, 0x1.fffc00f800000p-91, -0x1.0000007ffffe0p-11, -0x1.f000000000000p-82, -0x1.df0800fefffc2p-12, -0x1.ef0001ff80000p-82},
{-0x1.07c00007ffff0p-15, -0x1.fffc00f800000p-91, 0x1.0000007ffffe0p-11, 0x1.f000000000000p-82, 0x1.df0800fefffc2p-12, 0x1.ef0001ff80000p-82},
{-0x1.07c00007ffff0p-15, -0x1.fffc00f800000p-91, -0x1.0000007ffffe0p-11, -0x1.f000000000000p-82, -0x1.107c00807ffdfp-11, -0x1.f0fffe0080000p-82},
{0x1.0000000000000p+37, 0x1.ff800fffff800p-19, 0x1.0000000000000p-32, 0x1.fffc0000ffff8p-88, 0x1.0000000000000p+37, 0x1.ff880fffff800p-19},
{0x1.0000000000000p+37, 0x1.ff800fffff800p-19, -0x1.0000000000000p-32, -0x1.fffc0000ffff8p-88, 0x1.0000000000000p+37, 0x1.ff780fffff800p-19},
{-0x1.0000000000000p+37, -0x1.ff800fffff800p-19, 0x1.0000000000000p-32, 0x1.fffc0000ffff8p-88, -0x1.0000000000000p+37, -0x1.ff780fffff800p-19},
{-0x1.0000000000000p+37, -0x1.ff800fffff800p-19, -0x1.0000000000000p-32, -0x1.fffc0000ffff8p-88, -0x1.0000000000000p+37, -0x1.ff880fffff800p-19},
{0x1.0000000008000p+44, -0x1.ffff807ffe002p-10, 0x1.0000000800000p-6, -0x1.ffffff8000008p-62, 0x1.0000000008004p+44, -0x1.ffff7ffffe004p-10},
{0x1.0000000008000p+44, -0x1.ffff807ffe002p-10, -0x1.0000000800000p-6, 0x1.ffffff8000008p-62, 0x1.0000000007ffcp+44, -0x1.ffff80fffe000p-10},
{-0x1.0000000008000p+44, 0x1.ffff807ffe002p-10, 0x1.0000000800000p-6, -0x1.ffffff8000008p-62, -0x1.0000000007ffcp+44, 0x1.ffff80fffe000p-10},
{-0x1.0000000008000p+44, 0x1.ffff807ffe002p-10, -0x1.0000000800000p-6, 0x1.ffffff8000008p-62, -0x1.0000000008004p+44, 0x1.ffff7ffffe004p-10},
{0x1.0000000000020p-53, -0x1.0000000000000p-136, 0x1.0000000000000p+33, 0x1.fffff000003c0p-26, 0x1.0000000000000p+33, 0x1.fffff020003c0p-26},
{0x1.0000000000020p-53, -0x1.0000000000000p-136, -0x1.0000000000000p+33, -0x1.fffff000003c0p-26, -0x1.0000000000000p+33, -0x1.ffffefe0003c0p-26},
{-0x1.0000000000020p-53, 0x1.0000000000000p-136, 0x1.0000000000000p+33, 0x1.fffff000003c0p-26, 0x1.0000000000000p+33, 0x1.ffffefe0003c0p-26},
{-0x1.0000000000020p-53, 0x1.0000000000000p-136, -0x1.0000000000000p+33, -0x1.fffff000003c0p-26, -0x1.0000000000000p+33, -0x1.fffff020003c0p-26},
{0x1.0000001ffff80p-7, 0x1.fffffdfffff8ep-61, 0x1.0000000200000p-27, -0x1.0000000000000p-96, 0x1.0000101ffff82p-7, 0x1.fffffdffdff8ep-61},
{0x1.0000001ffff80p-7, 0x1.fffffdfffff8ep-61, -0x1.0000000200000p-27, 0x1.0000000000000p-96, 0x1.ffffe03fffefdp-8, -0x1.ffe0072000000p-85},
{-0x1.0000001ffff80p-7, -0x1.fffffdfffff8ep-61, 0x1.0000000200000p-27, -0x1.0000000000000p-96, -0x1.ffffe03fffefdp-8, 0x1.ffe0072000000p-85},
{-0x1.0000001ffff80p-7, -0x1.fffffdfffff8ep-61, -0x1.0000000200000p-27, 0x1.0000000000000p-96, -0x1.0000101ffff82p-7, -0x1.fffffdffdff8ep-61},
{0x1.01ffffff80400p-38, -0x1.c7ffffc000100p-96, 0x1.0000000000000p-47, 0x1.8fff01fffff00p-107, 0x1.027fffff80400p-38, -0x1.c7cdffdfc0100p-96},
{0x1.01ffffff80400p-38, -0x1.c7ffffc000100p-96, -0x1.0000000000000p-47, -0x1.8fff01fffff00p-107, 0x1.017fffff80400p-38, -0x1.c831ffa040100p-96},
{-0x1.01ffffff80400p-38, 0x1.c7ffffc000100p-96, 0x1.0000000000000p-47, 0x1.8fff01fffff00p-107, -0x1.017fffff80400p-38, 0x1.c831ffa040100p-96},
{-0x1.01ffffff80400p-38, 0x1.c7ffffc000100p-96, -0x1.0000000000000p-47, -0x1.8fff01fffff00p-107, -0x1.027fffff80400p-38, 0x1.c7cdffdfc0100p-96},
{0x1.0000000000008p-27, -0x1.ffffc00800000p-100, 0x1.00003ffe00002p+16, -0x1.ffffffc180080p-44, 0x1.00003ffe00202p+16, -0x1.ffffffc080080p-44},
{0x1.0000000000008p-27, -0x1.ffffc00800000p-100, -0x1.00003ffe00002p+16, 0x1.ffffffc180080p-44, -0x1.00003ffdffe02p+16, 0x1.ffffffc280080p-44},
{-0x1.0000000000008p-27, 0x1.ffffc00800000p-100, 0x1.00003ffe00002p+16, -0x1.ffffffc180080p-44, 0x1.00003ffdffe02p+16, -0x1.ffffffc280080p-44},
{-0x1.0000000000008p-27, 0x1.ffffc00800000p-100, -0x1.00003ffe00002p+16, 0x1.ffffffc180080p-44, -0x1.00003ffe00202p+16, 0x1.ffffffc080080p-44},
{0x1.000000ffffff8p+16, 0x1.ffffffc000000p-41, 0x1.0000000000000p+8, 0x1.ffffffe000000p-50, 0x1.010000ffffff8p+16, 0x1.007fffdff8000p-40},
{0x1.000000ffffff8p+16, 0x1.ffffffc000000p-41, -0x1.0000000000000p+8, -0x1.ffffffe000000p-50, 0x1.fe0001ffffff0p+15, 0x1.feffffc010000p-41},
{-0x1.000000ffffff8p+16, -0x1.ffffffc000000p-41, 0x1.0000000000000p+8, 0x1.ffffffe000000p-50, -0x1.fe0001ffffff0p+15, -0x1.feffffc010000p-41},
{-0x1.000000ffffff8p+16, -0x1.ffffffc000000p-41, -0x1.0000000000000p+8, -0x1.ffffffe000000p-50, -0x1.010000ffffff8p+16, -0x1.007fffdff8000p-40},
{0x1.0000000020000p-39, -0x1.0000000000000p-144, 0x1.0000000000400p-40, -0x1.ff1ffffff4000p-107, 0x1.8000000020200p-39, -0x1.ff20000000000p-107},
{0x1.0000000020000p-39, -0x1.0000000000000p-144, -0x1.0000000000400p-40, 0x1.ff1ffffff4000p-107, 0x1.000000003fc00p-40, 0x1.ff1fffffec000p-107},
{-0x1.0000000020000p-39, 0x1.0000000000000p-144, 0x1.0000000000400p-40, -0x1.ff1ffffff4000p-107, -0x1.000000003fc00p-40, -0x1.ff1fffffec000p-107},
{-0x1.0000000020000p-39, 0x1.0000000000000p-144, -0x1.0000000000400p-40, 0x1.ff1ffffff4000p-107, -0x1.8000000020200p-39, 0x1.ff20000000000p-107},
{0x1.0000ff8020000p-18, -0x1.fffffa0000000p-80, 0x1.0000000001ff0p+30, 0x1.f700000ffe000p-36, 0x1.0000000002000p+30, 0x1.7d402003ff800p-34},
{0x1.0000ff8020000p-18, -0x1.fffffa0000000p-80, -0x1.0000000001ff0p+30, -0x1.f700000ffe000p-36, -0x1.0000000001fe0p+30, 0x1.03803ff801000p-35},
{-0x1.0000ff8020000p-18, 0x1.fffffa0000000p-80, 0x1.0000000001ff0p+30, 0x1.f700000ffe000p-36, 0x1.0000000001fe0p+30, -0x1.03803ff801000p-35},
{-0x1.0000ff8020000p-18, 0x1.fffffa0000000p-80, -0x1.0000000001ff0p+30, -0x1.f700000ffe000p-36, -0x1.0000000002000p+30, -0x1.7d402003ff800p-34},
{0x1.0000020000000p+8, -0x1.0000000000000p-56, 0x1.003e00001f800p-14, 0x1.ffde000001e00p-76, 0x1.00000600f8000p+8, 0x1.f400007ff7800p-50},
{0x1.0000020000000p+8, -0x1.0000000000000p-56, -0x1.003e00001f800p-14, -0x1.ffde000001e00p-76, 0x1.fffffbfe10000p+7, -0x1.fc00007ff7800p-50},
{-0x1.0000020000000p+8, 0x1.0000000000000p-56, 0x1.003e00001f800p-14, 0x1.ffde000001e00p-76, -0x1.fffffbfe10000p+7, 0x1.fc00007ff7800p-50},
{-0x1.0000020000000p+8, 0x1.0000000000000p-56, -0x1.003e00001f800p-14, -0x1.ffde000001e00p-76, -0x1.00000600f8000p+8, -0x1.f400007ff7800p-50},
{0x1.0007e3ffe0000p-24, 0x1.fe0ffffffe000p-88, 0x1.0000000000000p+0, 0x1.fffffe07fff00p-61, 0x1.0000010007e40p+0, -0x1.000000dc1f080p-60},
{0x1.0007e3ffe0000p-24, 0x1.fe0ffffffe000p-88, -0x1.0000000000000p+0, -0x1.fffffe07fff00p-61, -0x1.fffffdfff0380p-1, -0x1.7fffff720f7c0p-59},
{-0x1.0007e3ffe0000p-24, -0x1.fe0ffffffe000p-88, 0x1.0000000000000p+0, 0x1.fffffe07fff00p-61, 0x1.fffffdfff0380p-1, 0x1.7fffff720f7c0p-59},
{-0x1.0007e3ffe0000p-24, -0x1.fe0ffffffe000p-88, -0x1.0000000000000p+0, -0x1.fffffe07fff00p-61, -0x1.0000010007e40p+0, 0x1.000000dc1f080p-60},
{0x1.00000000fe000p+38, 0x1.ff00004000000p-29, 0x1.000003fffff80p-32, 0x1.fff8400000000p-92, 0x1.00000000fe000p+38, 0x1.0f80006000000p-28},
{0x1.00000000fe000p+38, 0x1.ff00004000000p-29, -0x1.000003fffff80p-32, -0x1.fff8400000000p-92, 0x1.00000000fe000p+38, 0x1.deffffc000000p-29},
{-0x1.00000000fe000p+38, -0x1.ff00004000000p-29, 0x1.000003fffff80p-32, 0x1.fff8400000000p-92, -0x1.00000000fe000p+38, -0x1.deffffc000000p-29},
{-0x1.00000000fe000p+38, -0x1.ff00004000000p-29, -0x1.000003fffff80p-32, -0x1.fff8400000000p-92, -0x1.00000000fe000p+38, -0x1.0f80006000000p-28},
{0x1.0000800000000p-9, -0x1.0000000000000p-65, 0x1.0000000000000p-51, 0x1.fffffffc1fc00p-114, 0x1.0000800000400p-9, -0x1.fffffffffffe0p-66},
{0x1.0000800000000p-9, -0x1.0000000000000p-65, -0x1.0000000000000p-51, -0x1.fffffffc1fc00p-114, 0x1.00007fffffc00p-9, -0x1.0000000000010p-65},
{-0x1.0000800000000p-9, 0x1.0000000000000p-65, 0x1.0000000000000p-51, 0x1.fffffffc1fc00p-114, -0x1.00007fffffc00p-9, 0x1.0000000000010p-65},
{-0x1.0000800000000p-9, 0x1.0000000000000p-65, -0x1.0000000000000p-51, -0x1.fffffffc1fc00p-114, -0x1.0000800000400p-9, 0x1.fffffffffffe0p-66},
{0x1.0000000000000p-40, 0x1.ff000ff3ffe00p-94, 0x1.0000000000000p-18, 0x1.fe0003fffc000p-77, 0x1.0000040000000p-18, 0x1.fe01038003f80p-77},
{0x1.0000000000000p-40, 0x1.ff000ff3ffe00p-94, -0x1.0000000000000p-18, -0x1.fe0003fffc000p-77, -0x1.fffff80000000p-19, -0x1.fdff047ff4060p-77},
{-0x1.0000000000000p-40, -0x1.ff000ff3ffe00p-94, 0x1.0000000000000p-18, 0x1.fe0003fffc000p-77, 0x1.fffff80000000p-19, 0x1.fdff047ff4060p-77},
{-0x1.0000000000000p-40, -0x1.ff000ff3ffe00p-94, -0x1.0000000000000p-18, -0x1.fe0003fffc000p-77, -0x1.0000040000000p-18, -0x1.fe01038003f80p-77},
{0x1.000000e000010p-12, -0x1.ffffffc020000p-82, 0x1.00000001f0000p-4, 0x1.8000000000000p-108, 0x1.01000002d0000p-4, 0x1.fffff00000050p-61},
{0x1.000000e000010p-12, -0x1.ffffffc020000p-82, -0x1.00000001f0000p-4, -0x1.8000000000000p-108, -0x1.fe00000220000p-5, 0x1.ffffefffffff0p-61},
{-0x1.000000e000010p-12, 0x1.ffffffc020000p-82, 0x1.00000001f0000p-4, 0x1.8000000000000p-108, 0x1.fe00000220000p-5, -0x1.ffffefffffff0p-61},
{-0x1.000000e000010p-12, 0x1.ffffffc020000p-82, -0x1.00000001f0000p-4, -0x1.8000000000000p-108, -0x1.01000002d0000p-4, -0x1.fffff00000050p-61},
{0x1.000003ff00000p+23, 0x1.8000000000000p-54, 0x1.0001ffffe0000p-39, 0x1.f800000100000p-112, 0x1.000003ff00000p+23, 0x1.0004ffffe0000p-39},
{0x1.000003ff00000p+23, 0x1.8000000000000p-54, -0x1.0001ffffe0000p-39, -0x1.f800000100000p-112, 0x1.000003ff00000p+23, -0x1.fffdffffc0000p-40},
{-0x1.000003ff00000p+23, -0x1.8000000000000p-54, 0x1.0001ffffe0000p-39, 0x1.f800000100000p-112, -0x1.000003ff00000p+23, 0x1.fffdffffc0000p-40},
{-0x1.000003ff00000p+23, -0x1.8000000000000p-54, -0x1.0001ffffe0000p-39, -0x1.f800000100000p-112, -0x1.000003ff00000p+23, -0x1.0004ffffe0000p-39},
{0x1.0000000000200p+14, -0x1.ffffffffffc08p-42, 0x1.0000000000000p-48, 0x1.ffffffffffffep-102, 0x1.0000000000200p+14, -0x1.fbffffffffc08p-42},
{0x1.0000000000200p+14, -0x1.ffffffffffc08p-42, -0x1.0000000000000p-48, -0x1.ffffffffffffep-102, 0x1.0000000000200p+14, -0x1.01ffffffffe04p-41},
{-0x1.0000000000200p+14, 0x1.ffffffffffc08p-42, 0x1.0000000000000p-48, 0x1.ffffffffffffep-102, -0x1.0000000000200p+14, 0x1.01ffffffffe04p-41},
{-0x1.0000000000200p+14, 0x1.ffffffffffc08p-42, -0x1.0000000000000p-48, -0x1.ffffffffffffep-102, -0x1.0000000000200p+14, 0x1.fbffffffffc08p-42},
{0x1.0000000040000p+37, -0x1.fffff80002000p-29, 0x1.0000000000000p-21, 0x1.ffc01ffff3c00p-84, 0x1.0000000040000p+37, 0x1.fc00000ffffc0p-22},
{0x1.0000000040000p+37, -0x1.fffff80002000p-29, -0x1.0000000000000p-21, -0x1.ffc01ffff3c00p-84, 0x1.0000000040000p+37, -0x1.01fffff800020p-21},
{-0x1.0000000040000p+37, 0x1.fffff80002000p-29, 0x1.0000000000000p-21, 0x1.ffc01ffff3c00p-84, -0x1.0000000040000p+37, 0x1.01fffff800020p-21},
{-0x1.0000000040000p+37, 0x1.fffff80002000p-29, -0x1.0000000000000p-21, -0x1.ffc01ffff3c00p-84, -0x1.0000000040000p+37, -0x1.fc00000ffffc0p-22},
{0x1.0000800000000p+25, -0x1.0000000000000p-80, 0x1.0000003800001p-52, 0x1.fffffe000003ep-106, 0x1.0000800000000p+25, 0x1.0000003000000p-52},
{0x1.0000800000000p+25, -0x1.0000000000000p-80, -0x1.0000003800001p-52, -0x1.fffffe000003ep-106, 0x1.0000800000000p+25, -0x1.0000005000000p-52},
{-0x1.0000800000000p+25, 0x1.0000000000000p-80, 0x1.0000003800001p-52, 0x1.fffffe000003ep-106, -0x1.0000800000000p+25, 0x1.0000005000000p-52},
{-0x1.0000800000000p+25, 0x1.0000000000000p-80, -0x1.0000003800001p-52, -0x1.fffffe000003ep-106, -0x1.0000800000000p+25, -0x1.0000003000000p-52},
{0x1.0f00000020000p-4, -0x1.fffe000000100p-58, 0x1.0000000400000p-7, -0x1.ffffffc004000p-74, 0x1.2f000000a0000p-4, -0x1.ffffffffffd00p-58},
{0x1.0f00000020000p-4, -0x1.fffe000000100p-58, -0x1.0000000400000p-7, 0x1.ffffffc004000p-74, 0x1.ddffffff3ffffp-5, 0x1.ffffffd800000p-73},
{-0x1.0f00000020000p-4, 0x1.fffe000000100p-58, 0x1.0000000400000p-7, -0x1.ffffffc004000p-74, -0x1.ddffffff3ffffp-5, -0x1.ffffffd800000p-73},
{-0x1.0f00000020000p-4, 0x1.fffe000000100p-58, -0x1.0000000400000p-7, 0x1.ffffffc004000p-74, -0x1.2f000000a0000p-4, 0x1.ffffffffffd00p-58},
{0x1.003fe00020000p-13, -0x1.ffffffffe8000p-81, 0x1.0000000000400p-48, -0x1.ffff800007ffcp-103, 0x1.003fe00040000p-13, -0x1.ff8007ffe8000p-81},
{0x1.003fe00020000p-13, -0x1.ffffffffe8000p-81, -0x1.0000000000400p-48, 0x1.ffff800007ffcp-103, 0x1.003fe00000000p-13, -0x1.003ffbfff4000p-80},
{-0x1.003fe00020000p-13, 0x1.ffffffffe8000p-81, 0x1.0000000000400p-48, -0x1.ffff800007ffcp-103, -0x1.003fe00000000p-13, 0x1.003ffbfff4000p-80},
{-0x1.003fe00020000p-13, 0x1.ffffffffe8000p-81, -0x1.0000000000400p-48, 0x1.ffff800007ffcp-103, -0x1.003fe00040000p-13, 0x1.ff8007ffe8000p-81},
{0x1.ffff00fffffc4p-6, -0x1.8080000000000p-88, 0x1.0000020000000p-1, -0x1.0000000000000p-77, 0x1.0ffffa07ffffep-1, 0x1.ffffeffcff000p-57},
{0x1.ffff00fffffc4p-6, -0x1.8080000000000p-88, -0x1.0000020000000p-1, 0x1.0000000000000p-77, -0x1.e00013f000004p-2, 0x1.000007fe7f800p-56},
{-0x1.ffff00fffffc4p-6, 0x1.8080000000000p-88, 0x1.0000020000000p-1, -0x1.0000000000000p-77, 0x1.e00013f000004p-2, -0x1.000007fe7f800p-56},
{-0x1.ffff00fffffc4p-6, 0x1.8080000000000p-88, -0x1.0000020000000p-1, 0x1.0000000000000p-77, -0x1.0ffffa07ffffep-1, -0x1.ffffeffcff000p-57},
{0x1.0000000000040p+36, -0x1.8000000800000p-39, 0x1.0000000002000p+30, -0x1.0000000000000p-74, 0x1.04000000000c0p+36, -0x1.8000000800000p-39},
{0x1.0000000000040p+36, -0x1.8000000800000p-39, -0x1.0000000002000p+30, 0x1.0000000000000p-74, 0x1.f7fffffffff80p+35, -0x1.8000000800000p-39},
{-0x1.0000000000040p+36, 0x1.8000000800000p-39, 0x1.0000000002000p+30, -0x1.0000000000000p-74, -0x1.f7fffffffff80p+35, 0x1.8000000800000p-39},
{-0x1.0000000000040p+36, 0x1.8000000800000p-39, -0x1.0000000002000p+30, 0x1.0000000000000p-74, -0x1.04000000000c0p+36, 0x1.8000000800000p-39},
{0x1.0000000800000p-29, -0x1.ffffc02000000p-107, 0x1.7fffe00800000p-48, -0x1.0000000000000p-153, 0x1.00003007fc010p-29, -0x1.ffffc02000000p-107},
{0x1.0000000800000p-29, -0x1.ffffc02000000p-107, -0x1.7fffe00800000p-48, 0x1.0000000000000p-153, 0x1.ffffa01007fe0p-30, -0x1.ffffc02000000p-107},
{-0x1.0000000800000p-29, 0x1.ffffc02000000p-107, 0x1.7fffe00800000p-48, -0x1.0000000000000p-153, -0x1.ffffa01007fe0p-30, 0x1.ffffc02000000p-107},
{-0x1.0000000800000p-29, 0x1.ffffc02000000p-107, -0x1.7fffe00800000p-48, 0x1.0000000000000p-153, -0x1.00003007fc010p-29, 0x1.ffffc02000000p-107},
{0x1.0000400000000p+11, -0x1.0000000000000p-80, 0x1.4000000000000p+51, -0x1.ffffffff00020p-7, 0x1.4000000001000p+51, -0x1.fffffffe00040p-8},
{0x1.0000400000000p+11, -0x1.0000000000000p-80, -0x1.4000000000000p+51, 0x1.ffffffff00020p-7, -0x1.3fffffffff000p+51, 0x1.7fffffff80010p-6},
{-0x1.0000400000000p+11, 0x1.0000000000000p-80, 0x1.4000000000000p+51, -0x1.ffffffff00020p-7, 0x1.3fffffffff000p+51, -0x1.7fffffff80010p-6},
{-0x1.0000400000000p+11, 0x1.0000000000000p-80, -0x1.4000000000000p+51, 0x1.ffffffff00020p-7, -0x1.4000000001000p+51, 0x1.fffffffe00040p-8},
{0x1.0000000000000p+42, 0x1.03eff7ff3fff0p-15, 0x1.00001fffc0000p-1, 0x1.f80001f000f80p-61, 0x1.0000000000200p+42, 0x1.0befe7ff40070p-15},
{0x1.0000000000000p+42, 0x1.03eff7ff3fff0p-15, -0x1.00001fffc0000p-1, -0x1.f80001f000f80p-61, 0x1.ffffffffffc00p+41, 0x1.f7e00ffe7fee0p-16},
{-0x1.0000000000000p+42, -0x1.03eff7ff3fff0p-15, 0x1.00001fffc0000p-1, 0x1.f80001f000f80p-61, -0x1.ffffffffffc00p+41, -0x1.f7e00ffe7fee0p-16},
{-0x1.0000000000000p+42, -0x1.03eff7ff3fff0p-15, -0x1.00001fffc0000p-1, -0x1.f80001f000f80p-61, -0x1.0000000000200p+42, -0x1.0befe7ff40070p-15},
{0x1.03ffffffffffep-18, 0x1.8000000000000p-122, 0x1.0000040000000p+49, -0x1.0000000000000p-56, 0x1.0000040000000p+49, 0x1.03fffffffc000p-18},
{0x1.03ffffffffffep-18, 0x1.8000000000000p-122, -0x1.0000040000000p+49, 0x1.0000000000000p-56, -0x1.0000040000000p+49, 0x1.0400000004000p-18},
{-0x1.03ffffffffffep-18, -0x1.8000000000000p-122, 0x1.0000040000000p+49, -0x1.0000000000000p-56, 0x1.0000040000000p+49, -0x1.0400000004000p-18},
{-0x1.03ffffffffffep-18, -0x1.8000000000000p-122, -0x1.0000040000000p+49, 0x1.0000000000000p-56, -0x1.0000040000000p+49, -0x1.03fffffffc000p-18},
{0x1.0007ffffcfffcp+9, 0x1.ff01fffe00000p-65, 0x1.01fff08000000p-5, -0x1.fffffe0000020p-59, 0x1.000c07ff91ffcp+9, -0x1.f803f60008000p-59},
{0x1.0007ffffcfffcp+9, 0x1.ff01fffe00000p-65, -0x1.01fff08000000p-5, 0x1.fffffe0000020p-59, 0x1.0003f8000dffcp+9, 0x1.03fe02fffc000p-58},
{-0x1.0007ffffcfffcp+9, -0x1.ff01fffe00000p-65, 0x1.01fff08000000p-5, -0x1.fffffe0000020p-59, -0x1.0003f8000dffcp+9, -0x1.03fe02fffc000p-58},
{-0x1.0007ffffcfffcp+9, -0x1.ff01fffe00000p-65, -0x1.01fff08000000p-5, 0x1.fffffe0000020p-59, -0x1.000c07ff91ffcp+9, 0x1.f803f60008000p-59},
{0x1.00000000007fep-4, 0x1.ff00003ffff00p-65, 0x1.004000e000008p+5, -0x1.ffffe02000000p-73, 0x1.00c000e00000cp+5, -0x1.ff017fffd0100p-56},
{0x1.00000000007fep-4, 0x1.ff00003ffff00p-65, -0x1.004000e000008p+5, 0x1.ffffe02000000p-73, -0x1.ff8001c000008p+4, -0x1.feff7fffeff00p-56},
{-0x1.00000000007fep-4, -0x1.ff00003ffff00p-65, 0x1.004000e000008p+5, -0x1.ffffe02000000p-73, 0x1.ff8001c000008p+4, 0x1.feff7fffeff00p-56},
{-0x1.00000000007fep-4, -0x1.ff00003ffff00p-65, -0x1.004000e000008p+5, 0x1.ffffe02000000p-73, -0x1.00c000e00000cp+5, 0x1.ff017fffd0100p-56},
{0x1.00000003ff000p-41, 0x1.ffff000000000p-130, 0x1.0000fffff8000p-20, 0x1.f003ffe0003e0p-78, 0x1.000107fff8002p-20, 0x1.d003ffe0003e0p-78},
{0x1.00000003ff000p-41, 0x1.ffff000000000p-130, -0x1.0000fffff8000p-20, -0x1.f003ffe0003e0p-78, -0x1.0000f7fff7ffep-20, -0x1.0801fff0001f0p-77},
{-0x1.00000003ff000p-41, -0x1.ffff000000000p-130, 0x1.0000fffff8000p-20, 0x1.f003ffe0003e0p-78, 0x1.0000f7fff7ffep-20, 0x1.0801fff0001f0p-77},
{-0x1.00000003ff000p-41, -0x1.ffff000000000p-130, -0x1.0000fffff8000p-20, -0x1.f003ffe0003e0p-78, -0x1.000107fff8002p-20, -0x1.d003ffe0003e0p-78},
{0x1.0000000000000p-41, 0x1.ffe001c0003c0p-100, 0x1.01fffffe00000p+41, 0x1.ffffffffffc00p-17, 0x1.01fffffe00000p+41, 0x1.0000007fffe00p-16},
{0x1.0000000000000p-41, 0x1.ffe001c0003c0p-100, -0x1.01fffffe00000p+41, -0x1.ffffffffffc00p-17, -0x1.01fffffe00000p+41, -0x1.fffffeffffc00p-17},
{-0x1.0000000000000p-41, -0x1.ffe001c0003c0p-100, 0x1.01fffffe00000p+41, 0x1.ffffffffffc00p-17, 0x1.01fffffe00000p+41, 0x1.fffffeffffc00p-17},
{-0x1.0000000000000p-41, -0x1.ffe001c0003c0p-100, -0x1.01fffffe00000p+41, -0x1.ffffffffffc00p-17, -0x1.01fffffe00000p+41, -0x1.0000007fffe00p-16},
{0x1.00003ff808000p+36, -0x1.ffe0000800000p-25, 0x1.000001ffff000p+41, 0x1.ff80000000000p-33, 0x1.080003ffbf400p+41, -0x1.fde0800800000p-25},
{0x1.00003ff808000p+36, -0x1.ffe0000800000p-25, -0x1.000001ffff000p+41, -0x1.ff80000000000p-33, -0x1.f00000007d800p+40, -0x1.00efc00400000p-24},
{-0x1.00003ff808000p+36, 0x1.ffe0000800000p-25, 0x1.000001ffff000p+41, 0x1.ff80000000000p-33, 0x1.f00000007d800p+40, 0x1.00efc00400000p-24},
{-0x1.00003ff808000p+36, 0x1.ffe0000800000p-25, -0x1.000001ffff000p+41, -0x1.ff80000000000p-33, -0x1.080003ffbf400p+41, 0x1.fde0800800000p-25},
{0x1.0000000000000p+6, 0x1.fc007f0ff0000p-56, 0x1.0000000007f80p-50, 0x1.ff0007f802000p-114, 0x1.0000000000000p+6, 0x1.07f001fc47b80p-50},
{0x1.0000000000000p+6, 0x1.fc007f0ff0000p-56, -0x1.0000000007f80p-50, -0x1.ff0007f802000p-114, 0x1.0000000000000p+6, -0x1.f01ffc0790700p-51},
{-0x1.0000000000000p+6, -0x1.fc007f0ff0000p-56, 0x1.0000000007f80p-50, 0x1.ff0007f802000p-114, -0x1.0000000000000p+6, 0x1.f01ffc0790700p-51},
{-0x1.0000000000000p+6, -0x1.fc007f0ff0000p-56, -0x1.0000000007f80p-50, -0x1.ff0007f802000p-114, -0x1.0000000000000p+6, -0x1.07f001fc47b80p-50},
{0x1.0020000000000p+12, -0x1.0000000000000p-65, 0x1.000001ffff802p-4, -0x1.ffffffc000040p-63, 0x1.0021000002000p+12, -0x1.ff811fffffe00p-46},
{0x1.0020000000000p+12, -0x1.0000000000000p-65, -0x1.000001ffff802p-4, 0x1.ffffffc000040p-63, 0x1.001efffffe000p+12, 0x1.ff80dfffffe00p-46},
{-0x1.0020000000000p+12, 0x1.0000000000000p-65, 0x1.000001ffff802p-4, -0x1.ffffffc000040p-63, -0x1.001efffffe000p+12, -0x1.ff80dfffffe00p-46},
{-0x1.0020000000000p+12, 0x1.0000000000000p-65, -0x1.000001ffff802p-4, 0x1.ffffffc000040p-63, -0x1.0021000002000p+12, 0x1.ff811fffffe00p-46},
{0x1.0000000000800p-45, -0x1.ffe0000400000p-107, 0x1.0000000000038p+25, 0x1.e100000000000p-49, 0x1.0000000000038p+25, 0x1.1e10000000000p-45},
{0x1.0000000000800p-45, -0x1.ffe0000400000p-107, -0x1.0000000000038p+25, -0x1.e100000000000p-49, -0x1.0000000000038p+25, 0x1.c3e0000000000p-46},
{-0x1.0000000000800p-45, 0x1.ffe0000400000p-107, 0x1.0000000000038p+25, 0x1.e100000000000p-49, 0x1.0000000000038p+25, -0x1.c3e0000000000p-46},
{-0x1.0000000000800p-45, 0x1.ffe0000400000p-107, -0x1.0000000000038p+25, -0x1.e100000000000p-49, -0x1.0000000000038p+25, -0x1.1e10000000000p-45},
{0x1.0000000000020p-31, -0x1.8007ffff80c04p-86, 0x1.0000000000000p+50, 0x1.ffffe00000c00p-13, 0x1.0000000000000p+50, 0x1.0000100000600p-12},
{0x1.0000000000020p-31, -0x1.8007ffff80c04p-86, -0x1.0000000000000p+50, -0x1.ffffe00000c00p-13, -0x1.0000000000000p+50, -0x1.ffffa00000c00p-13},
{-0x1.0000000000020p-31, 0x1.8007ffff80c04p-86, 0x1.0000000000000p+50, 0x1.ffffe00000c00p-13, 0x1.0000000000000p+50, 0x1.ffffa00000c00p-13},
{-0x1.0000000000020p-31, 0x1.8007ffff80c04p-86, -0x1.0000000000000p+50, -0x1.ffffe00000c00p-13, -0x1.0000000000000p+50, -0x1.0000100000600p-12},
{0x1.7fffffff80000p+1, 0x1.f000000000000p-76, 0x1.000007ffffffep-41, 0x1.ffffff8000000p-117, 0x1.7fffffff80400p+1, 0x1.0007bffc00000p-62},
{0x1.7fffffff80000p+1, 0x1.f000000000000p-76, -0x1.000007ffffffep-41, -0x1.ffffff8000000p-117, 0x1.7fffffff7fc00p+1, -0x1.fff07ff800000p-63},
{-0x1.7fffffff80000p+1, -0x1.f000000000000p-76, 0x1.000007ffffffep-41, 0x1.ffffff8000000p-117, -0x1.7fffffff7fc00p+1, 0x1.fff07ff800000p-63},
{-0x1.7fffffff80000p+1, -0x1.f000000000000p-76, -0x1.000007ffffffep-41, -0x1.ffffff8000000p-117, -0x1.7fffffff80400p+1, -0x1.0007bffc00000p-62},
{0x1.0000000010000p+45, -0x1.0000000000000p-41, 0x1.0000000000000p+5, 0x1.c000003ffc3fep-49, 0x1.0000000011000p+45, -0x1.fc80000000000p-42},
{0x1.0000000010000p+45, -0x1.0000000000000p-41, -0x1.0000000000000p+5, -0x1.c000003ffc3fep-49, 0x1.000000000f000p+45, -0x1.01c0000000000p-41},
{-0x1.0000000010000p+45, 0x1.0000000000000p-41, 0x1.0000000000000p+5, 0x1.c000003ffc3fep-49, -0x1.000000000f000p+45, 0x1.01c0000000000p-41},
{-0x1.0000000010000p+45, 0x1.0000000000000p-41, -0x1.0000000000000p+5, -0x1.c000003ffc3fep-49, -0x1.0000000011000p+45, 0x1.fc80000000000p-42},
{0x1.00000ffff0800p-14, -0x1.fff80000ffc20p-72, 0x1.0000000000002p-19, -0x1.0000000000000p-124, 0x1.08000ffff0800p-14, 0x1.0003ffff801f0p-71},
{0x1.00000ffff0800p-14, -0x1.fff80000ffc20p-72, -0x1.0000000000002p-19, 0x1.0000000000000p-124, 0x1.f0001fffe1000p-15, -0x1.7ffe00003ff08p-70},
{-0x1.00000ffff0800p-14, 0x1.fff80000ffc20p-72, 0x1.0000000000002p-19, -0x1.0000000000000p-124, -0x1.f0001fffe1000p-15, 0x1.7ffe00003ff08p-70},
{-0x1.00000ffff0800p-14, 0x1.fff80000ffc20p-72, -0x1.0000000000002p-19, 0x1.0000000000000p-124, -0x1.08000ffff0800p-14, -0x1.0003ffff801f0p-71},
{0x1.000ffffc00000p+49, 0x1.fffff00000000p-20, 0x1.0000000000040p+45, -0x1.0000000000000p-60, 0x1.100ffffc00004p+49, 0x1.fffff00000000p-20},
{0x1.000ffffc00000p+49, 0x1.fffff00000000p-20, -0x1.0000000000040p+45, 0x1.0000000000000p-60, 0x1.e01ffff7ffff8p+48, 0x1.fffff00000000p-20},
{-0x1.000ffffc00000p+49, -0x1.fffff00000000p-20, 0x1.0000000000040p+45, -0x1.0000000000000p-60, -0x1.e01ffff7ffff8p+48, -0x1.fffff00000000p-20},
{-0x1.000ffffc00000p+49, -0x1.fffff00000000p-20, -0x1.0000000000040p+45, 0x1.0000000000000p-60, -0x1.100ffffc00004p+49, -0x1.fffff00000000p-20},
{0x1.0004000000000p+14, -0x1.0000000000000p-54, 0x1.0000000200000p-4, -0x1.ffbffffff8020p-62, 0x1.0004400000008p+14, -0x1.01ffc00000000p-54},
{0x1.0004000000000p+14, -0x1.0000000000000p-54, -0x1.0000000200000p-4, 0x1.ffbffffff8020p-62, 0x1.0003bfffffff8p+14, -0x1.fc00800000000p-55},
{-0x1.0004000000000p+14, 0x1.0000000000000p-54, 0x1.0000000200000p-4, -0x1.ffbffffff8020p-62, -0x1.0003bfffffff8p+14, 0x1.fc00800000000p-55},
{-0x1.0004000000000p+14, 0x1.0000000000000p-54, -0x1.0000000200000p-4, 0x1.ffbffffff8020p-62, -0x1.0004400000008p+14, 0x1.01ffc00000000p-54},
{0x1.0000000000000p-1, 0x1.e0001e7fbff00p-62, 0x1.0000001fffffep+43, 0x0.0000000000000p+0, 0x1.00000020000fep+43, 0x1.0000000000000p-61},
{0x1.0000000000000p-1, 0x1.e0001e7fbff00p-62, -0x1.0000001fffffep+43, 0x0.0000000000000p+0, -0x1.0000001fffefep+43, 0x1.0000000000000p-61},
{-0x1.0000000000000p-1, -0x1.e0001e7fbff00p-62, 0x1.0000001fffffep+43, 0x0.0000000000000p+0, 0x1.0000001fffefep+43, -0x1.0000000000000p-61},
{-0x1.0000000000000p-1, -0x1.e0001e7fbff00p-62, -0x1.0000001fffffep+43, 0x0.0000000000000p+0, -0x1.00000020000fep+43, -0x1.0000000000000p-61},
{0x1.0000000000fffp-39, 0x1.fffff3e000000p-112, 0x1.001ffe0000000p-15, 0x1.fffffffffffc0p-74, 0x1.001fff0000000p-15, 0x1.03ffc00003fe0p-73},
{0x1.0000000000fffp-39, 0x1.fffff3e000000p-112, -0x1.001ffe0000000p-15, -0x1.fffffffffffc0p-74, -0x1.001ffd0000000p-15, -0x1.f8007ffff7fc0p-74},
{-0x1.0000000000fffp-39, -0x1.fffff3e000000p-112, 0x1.001ffe0000000p-15, 0x1.fffffffffffc0p-74, 0x1.001ffd0000000p-15, 0x1.f8007ffff7fc0p-74},
{-0x1.0000000000fffp-39, -0x1.fffff3e000000p-112, -0x1.001ffe0000000p-15, -0x1.fffffffffffc0p-74, -0x1.001fff0000000p-15, -0x1.03ffc00003fe0p-73},
{0x1.0000080000000p-27, -0x1.0000000000000p-132, 0x1.0000000080000p-48, -0x1.0000000000000p-103, 0x1.0000100000000p-27, 0x1.fffff7ffffffcp-82},
{0x1.0000080000000p-27, -0x1.0000000000000p-132, -0x1.0000000080000p-48, 0x1.0000000000000p-103, 0x1.0000000000000p-27, -0x1.fffff80000004p-82},
{-0x1.0000080000000p-27, 0x1.0000000000000p-132, 0x1.0000000080000p-48, -0x1.0000000000000p-103, -0x1.0000000000000p-27, 0x1.fffff80000004p-82},
{-0x1.0000080000000p-27, 0x1.0000000000000p-132, -0x1.0000000080000p-48, 0x1.0000000000000p-103, -0x1.0000100000000p-27, -0x1.fffff7ffffffcp-82},
{0x1.003ffffffffe0p-31, 0x1.ffffe03800000p-107, 0x1.0000000002000p-53, -0x1.0000000000000p-106, 0x1.004003fffffe0p-31, 0x1.ffffffff80e00p-93},
{0x1.003ffffffffe0p-31, 0x1.ffffe03800000p-107, -0x1.0000000002000p-53, 0x1.0000000000000p-106, 0x1.003ffbfffffe0p-31, -0x1.fff000007f200p-93},
{-0x1.003ffffffffe0p-31, -0x1.ffffe03800000p-107, 0x1.0000000002000p-53, -0x1.0000000000000p-106, -0x1.003ffbfffffe0p-31, 0x1.fff000007f200p-93},
{-0x1.003ffffffffe0p-31, -0x1.ffffe03800000p-107, -0x1.0000000002000p-53, 0x1.0000000000000p-106, -0x1.004003fffffe0p-31, -0x1.ffffffff80e00p-93},
{0x1.0000000000008p+23, -0x1.ff000007ffc00p-37, 0x1.0000000060000p+36, 0x1.ffffff8000000p-44, 0x1.0008000060000p+36, 0x1.ff813ffffdc00p-27},
{0x1.0000000000008p+23, -0x1.ff000007ffc00p-37, -0x1.0000000060000p+36, -0x1.ffffff8000000p-44, -0x1.fff00000c0000p+35, 0x1.ff7f3ffffe400p-27},
{-0x1.0000000000008p+23, 0x1.ff000007ffc00p-37, 0x1.0000000060000p+36, 0x1.ffffff8000000p-44, 0x1.fff00000c0000p+35, -0x1.ff7f3ffffe400p-27},
{-0x1.0000000000008p+23, 0x1.ff000007ffc00p-37, -0x1.0000000060000p+36, -0x1.ffffff8000000p-44, -0x1.0008000060000p+36, -0x1.ff813ffffdc00p-27},
{0x1.0000000000080p+25, -0x1.0000000000000p-80, 0x1.01ffffff80001p+15, -0x1.ff90000000000p-78, 0x1.00407fffffe80p+25, 0x1.fffffffffdc00p-38},
{0x1.0000000000080p+25, -0x1.0000000000000p-80, -0x1.01ffffff80001p+15, 0x1.ff90000000000p-78, 0x1.ff7f000000500p+24, -0x1.fffffffffe400p-38},
{-0x1.0000000000080p+25, 0x1.0000000000000p-80, 0x1.01ffffff80001p+15, -0x1.ff90000000000p-78, -0x1.ff7f000000500p+24, 0x1.fffffffffe400p-38},
{-0x1.0000000000080p+25, 0x1.0000000000000p-80, -0x1.01ffffff80001p+15, 0x1.ff90000000000p-78, -0x1.00407fffffe80p+25, -0x1.fffffffffdc00p-38},
{0x1.0000000000800p+7, -0x1.fffff80002000p-59, 0x1.0008000000000p+50, -0x1.0000000000000p-29, 0x1.0008000000200p+50, -0x1.f000000000000p-30},
{0x1.0000000000800p+7, -0x1.fffff80002000p-59, -0x1.0008000000000p+50, 0x1.0000000000000p-29, -0x1.0007ffffffe00p+50, 0x1.0800000000000p-29},
{-0x1.0000000000800p+7, 0x1.fffff80002000p-59, 0x1.0008000000000p+50, -0x1.0000000000000p-29, 0x1.0007ffffffe00p+50, -0x1.0800000000000p-29},
{-0x1.0000000000800p+7, 0x1.fffff80002000p-59, -0x1.0008000000000p+50, 0x1.0000000000000p-29, -0x1.0008000000200p+50, 0x1.f000000000000p-30},
{0x1.0000000000002p+26, -0x1.ffffffe200000p-48, 0x1.0003fffffffc0p+27, 0x1.ffffe00000400p-36, 0x1.8003fffffffc1p+27, 0x1.ffdfe00002000p-36},
{0x1.0000000000002p+26, -0x1.ffffffe200000p-48, -0x1.0003fffffffc0p+27, -0x1.ffffe00000400p-36, -0x1.0007fffffff7ep+26, -0x1.000fefffff300p-35},
{-0x1.0000000000002p+26, 0x1.ffffffe200000p-48, 0x1.0003fffffffc0p+27, 0x1.ffffe00000400p-36, 0x1.0007fffffff7ep+26, 0x1.000fefffff300p-35},
{-0x1.0000000000002p+26, 0x1.ffffffe200000p-48, -0x1.0003fffffffc0p+27, -0x1.ffffe00000400p-36, -0x1.8003fffffffc1p+27, -0x1.ffdfe00002000p-36},
{0x1.000003f000000p+13, 0x1.fffffffe00000p-48, 0x1.0000000000ff0p+50, 0x1.c03fff0e00000p-24, 0x1.0000000008ff0p+50, 0x1.f807010004380p-10},
{0x1.000003f000000p+13, 0x1.fffffffe00000p-48, -0x1.0000000000ff0p+50, -0x1.c03fff0e00000p-24, -0x1.fffffffff1fe0p+49, 0x1.f7f8ff000bc80p-10},
{-0x1.000003f000000p+13, -0x1.fffffffe00000p-48, 0x1.0000000000ff0p+50, 0x1.c03fff0e00000p-24, 0x1.fffffffff1fe0p+49, -0x1.f7f8ff000bc80p-10},
{-0x1.000003f000000p+13, -0x1.fffffffe00000p-48, -0x1.0000000000ff0p+50, -0x1.c03fff0e00000p-24, -0x1.0000000008ff0p+50, -0x1.f807010004380p-10},
{0x1.0000000018000p-6, 0x1.fffffe0000000p-88, 0x1.0000100000000p+32, -0x1.0000000000000p-73, 0x1.0000100004000p+32, 0x1.7ffffffe00000p-42},
{0x1.0000000018000p-6, 0x1.fffffe0000000p-88, -0x1.0000100000000p+32, 0x1.0000000000000p-73, -0x1.00000ffffc000p+32, 0x1.8000000200000p-42},
{-0x1.0000000018000p-6, -0x1.fffffe0000000p-88, 0x1.0000100000000p+32, -0x1.0000000000000p-73, 0x1.00000ffffc000p+32, -0x1.8000000200000p-42},
{-0x1.0000000018000p-6, -0x1.fffffe0000000p-88, -0x1.0000100000000p+32, 0x1.0000000000000p-73, -0x1.0000100004000p+32, -0x1.7ffffffe00000p-42},
{0x1.0000000000001p+35, -0x1.fffffc1000000p-32, 0x1.1ffffff01f000p+36, 0x1.ffffc000060fcp-19, 0x1.9ffffff01f001p+36, -0x1.00081fffed382p-18},
{0x1.0000000000001p+35, -0x1.fffffc1000000p-32, -0x1.1ffffff01f000p+36, -0x1.ffffc000060fcp-19, -0x1.3fffffe03e000p+35, 0x1.fff0400019704p-19},
{-0x1.0000000000001p+35, 0x1.fffffc1000000p-32, 0x1.1ffffff01f000p+36, 0x1.ffffc000060fcp-19, 0x1.3fffffe03e000p+35, -0x1.fff0400019704p-19},
{-0x1.0000000000001p+35, 0x1.fffffc1000000p-32, -0x1.1ffffff01f000p+36, -0x1.ffffc000060fcp-19, -0x1.9ffffff01f001p+36, 0x1.00081fffed382p-18},
{0x1.0000003800000p+43, 0x1.ffe000001fffcp-12, 0x1.0000007fffff0p+50, 0x1.fe1ffffc00000p-12, 0x1.020000806fff0p+50, 0x1.fefffffe10000p-11},
{0x1.0000003800000p+43, 0x1.ffe000001fffcp-12, -0x1.0000007fffff0p+50, -0x1.fe1ffffc00000p-12, -0x1.fc0000ff1ffe0p+49, 0x1.c000042000000p-20},
{-0x1.0000003800000p+43, -0x1.ffe000001fffcp-12, 0x1.0000007fffff0p+50, 0x1.fe1ffffc00000p-12, 0x1.fc0000ff1ffe0p+49, -0x1.c000042000000p-20},
{-0x1.0000003800000p+43, -0x1.ffe000001fffcp-12, -0x1.0000007fffff0p+50, -0x1.fe1ffffc00000p-12, -0x1.020000806fff0p+50, -0x1.fefffffe10000p-11},
{0x1.0000001fffc00p+32, 0x1.f00007ffe0000p-32, 0x1.0000000000180p+25, 0x1.fffffe0000800p-39, 0x1.0200001fffc03p+32, 0x1.f40007fbe0000p-32},
{0x1.0000001fffc00p+32, 0x1.f00007ffe0000p-32, -0x1.0000000000180p+25, -0x1.fffffe0000800p-39, 0x1.fc00003fff7fap+31, 0x1.ec000803e0000p-32},
{-0x1.0000001fffc00p+32, -0x1.f00007ffe0000p-32, 0x1.0000000000180p+25, 0x1.fffffe0000800p-39, -0x1.fc00003fff7fap+31, -0x1.ec000803e0000p-32},
{-0x1.0000001fffc00p+32, -0x1.f00007ffe0000p-32, -0x1.0000000000180p+25, -0x1.fffffe0000800p-39, -0x1.0200001fffc03p+32, -0x1.f40007fbe0000p-32},
{0x1.0000000008000p+19, -0x1.c0000007ffe04p-36, 0x1.00000fffffffep-53, 0x1.fc00000000000p-152, 0x1.0000000008000p+19, -0x1.bfff8007f7e04p-36},
{0x1.0000000008000p+19, -0x1.c0000007ffe04p-36, -0x1.00000fffffffep-53, -0x1.fc00000000000p-152, 0x1.0000000008000p+19, -0x1.c000800807e04p-36},
{-0x1.0000000008000p+19, 0x1.c0000007ffe04p-36, 0x1.00000fffffffep-53, 0x1.fc00000000000p-152, -0x1.0000000008000p+19, 0x1.c000800807e04p-36},
{-0x1.0000000008000p+19, 0x1.c0000007ffe04p-36, -0x1.00000fffffffep-53, -0x1.fc00000000000p-152, -0x1.0000000008000p+19, 0x1.bfff8007f7e04p-36},
{0x1.0000003e00004p+43, -0x1.fe00040000000p-27, 0x1.00007c0001e00p+8, 0x1.fffe13ffe0000p-56, 0x1.0000003e20005p+43, -0x1.000f78001f800p-14},
{0x1.0000003e00004p+43, -0x1.fe00040000000p-27, -0x1.00007c0001e00p+8, -0x1.fffe13ffe0000p-56, 0x1.0000003de0003p+43, 0x1.ffdf2fffbf000p-15},
{-0x1.0000003e00004p+43, 0x1.fe00040000000p-27, 0x1.00007c0001e00p+8, 0x1.fffe13ffe0000p-56, -0x1.0000003de0003p+43, -0x1.ffdf2fffbf000p-15},
{-0x1.0000003e00004p+43, 0x1.fe00040000000p-27, -0x1.00007c0001e00p+8, -0x1.fffe13ffe0000p-56, -0x1.0000003e20005p+43, 0x1.000f78001f800p-14},
{0x1.03fffffffc000p+44, 0x1.ffffffc000000p-31, 0x1.0003ffffff000p-28, 0x1.ffc0000000000p-123, 0x1.03fffffffc000p+44, 0x1.4003fff800000p-28},
{0x1.03fffffffc000p+44, 0x1.ffffffc000000p-31, -0x1.0003ffffff000p-28, -0x1.ffc0000000000p-123, 0x1.03fffffffc000p+44, -0x1.8008001000000p-29},
{-0x1.03fffffffc000p+44, -0x1.ffffffc000000p-31, 0x1.0003ffffff000p-28, 0x1.ffc0000000000p-123, -0x1.03fffffffc000p+44, 0x1.8008001000000p-29},
{-0x1.03fffffffc000p+44, -0x1.ffffffc000000p-31, -0x1.0003ffffff000p-28, -0x1.ffc0000000000p-123, -0x1.03fffffffc000p+44, -0x1.4003fff800000p-28},
{0x1.07ffffff80000p+48, 0x1.8007ffffc0000p-23, 0x1.0000000000004p-5, -0x1.fffc000200000p-74, 0x1.07ffffff80001p+48, -0x1.ffff3ffbffffap-6},
{0x1.07ffffff80000p+48, 0x1.8007ffffc0000p-23, -0x1.0000000000004p-5, 0x1.fffc000200000p-74, 0x1.07ffffff80000p+48, -0x1.ffff3ffc0000ap-6},
{-0x1.07ffffff80000p+48, -0x1.8007ffffc0000p-23, 0x1.0000000000004p-5, -0x1.fffc000200000p-74, -0x1.07ffffff80000p+48, 0x1.ffff3ffc0000ap-6},
{-0x1.07ffffff80000p+48, -0x1.8007ffffc0000p-23, -0x1.0000000000004p-5, 0x1.fffc000200000p-74, -0x1.07ffffff80001p+48, 0x1.ffff3ffbffffap-6},
{0x1.00003ffffff80p-25, 0x1.ffff0000ffe00p-87, 0x1.00003fffffff8p-48, 0x0.0000000000000p+0, 0x1.0000420000780p-25, 0x1.ffbf0000ffe00p-87},
{0x1.00003ffffff80p-25, 0x1.ffff0000ffe00p-87, -0x1.00003fffffff8p-48, 0x0.0000000000000p+0, 0x1.00003dffff780p-25, 0x1.001f80007ff00p-86},
{-0x1.00003ffffff80p-25, -0x1.ffff0000ffe00p-87, 0x1.00003fffffff8p-48, 0x0.0000000000000p+0, -0x1.00003dffff780p-25, -0x1.001f80007ff00p-86},
{-0x1.00003ffffff80p-25, -0x1.ffff0000ffe00p-87, -0x1.00003fffffff8p-48, 0x0.0000000000000p+0, -0x1.0000420000780p-25, -0x1.ffbf0000ffe00p-87},
{0x1.0000010000000p+29, -0x1.0000000000000p-76, 0x1.00f81ffffff81p-9, 0x1.fffff03c00000p-71, 0x1.000001000403ep+29, 0x1.ffffff81007e0p-29},
{0x1.0000010000000p+29, -0x1.0000000000000p-76, -0x1.00f81ffffff81p-9, -0x1.fffff03c00000p-71, 0x1.000000fffbfc2p+29, -0x1.ffffff8100820p-29},
{-0x1.0000010000000p+29, 0x1.0000000000000p-76, 0x1.00f81ffffff81p-9, 0x1.fffff03c00000p-71, -0x1.000000fffbfc2p+29, 0x1.ffffff8100820p-29},
{-0x1.0000010000000p+29, 0x1.0000000000000p-76, -0x1.00f81ffffff81p-9, -0x1.fffff03c00000p-71, -0x1.000001000403ep+29, -0x1.ffffff81007e0p-29},
{0x1.0000000000004p+5, -0x1.fff0008000000p-59, 0x1.0000000100000p+48, -0x1.fffffe0004000p-19, 0x1.0000000100200p+48, -0x1.fffffdc004000p-19},
{0x1.0000000000004p+5, -0x1.fff0008000000p-59, -0x1.0000000100000p+48, 0x1.fffffe0004000p-19, -0x1.00000000ffe00p+48, 0x1.fffffe4004000p-19},
{-0x1.0000000000004p+5, 0x1.fff0008000000p-59, 0x1.0000000100000p+48, -0x1.fffffe0004000p-19, 0x1.00000000ffe00p+48, -0x1.fffffe4004000p-19},
{-0x1.0000000000004p+5, 0x1.fff0008000000p-59, -0x1.0000000100000p+48, 0x1.fffffe0004000p-19, -0x1.0000000100200p+48, 0x1.fffffdc004000p-19},
{0x1.0000000000800p+23, -0x1.0000000000000p-55, 0x1.0000010000000p+11, -0x1.0000000000000p-92, 0x1.0010000010800p+23, -0x1.0000000000000p-55},
{0x1.0000000000800p+23, -0x1.0000000000000p-55, -0x1.0000010000000p+11, 0x1.0000000000000p-92, 0x1.ffdfffffe1000p+22, -0x1.0000000000000p-55},
{-0x1.0000000000800p+23, 0x1.0000000000000p-55, 0x1.0000010000000p+11, -0x1.0000000000000p-92, -0x1.ffdfffffe1000p+22, 0x1.0000000000000p-55},
{-0x1.0000000000800p+23, 0x1.0000000000000p-55, -0x1.0000010000000p+11, 0x1.0000000000000p-92, -0x1.0010000010800p+23, 0x1.0000000000000p-55},
{0x1.00000007fff00p-25, 0x1.fffc1ffff87f0p-81, 0x1.0000007c00000p-2, 0x1.fff0000000000p-73, 0x1.0000027c00001p-2, -0x1.bfc2007c00000p-70},
{0x1.00000007fff00p-25, 0x1.fffc1ffff87f0p-81, -0x1.0000007c00000p-2, -0x1.fff0000000000p-73, -0x1.fffffcf7ffffep-3, -0x1.1fdf003e00000p-69},
{-0x1.00000007fff00p-25, -0x1.fffc1ffff87f0p-81, 0x1.0000007c00000p-2, 0x1.fff0000000000p-73, 0x1.fffffcf7ffffep-3, 0x1.1fdf003e00000p-69},
{-0x1.00000007fff00p-25, -0x1.fffc1ffff87f0p-81, -0x1.0000007c00000p-2, -0x1.fff0000000000p-73, -0x1.0000027c00001p-2, 0x1.bfc2007c00000p-70},
{0x1.fffffffe00000p+40, 0x1.e000001fff800p-24, 0x1.00000001ffffep-36, 0x0.0000000000000p+0, 0x1.fffffffe00000p+40, 0x1.e010001fff800p-24},
{0x1.fffffffe00000p+40, 0x1.e000001fff800p-24, -0x1.00000001ffffep-36, 0x0.0000000000000p+0, 0x1.fffffffe00000p+40, 0x1.dff0001fff800p-24},
{-0x1.fffffffe00000p+40, -0x1.e000001fff800p-24, 0x1.00000001ffffep-36, 0x0.0000000000000p+0, -0x1.fffffffe00000p+40, -0x1.dff0001fff800p-24},
{-0x1.fffffffe00000p+40, -0x1.e000001fff800p-24, -0x1.00000001ffffep-36, 0x0.0000000000000p+0, -0x1.fffffffe00000p+40, -0x1.e010001fff800p-24},
{0x1.0000000000008p+12, -0x1.fffff00400000p-60, 0x1.00003ffffe000p-40, 0x1.ffffe00000000p-112, 0x1.0000000000009p+12, 0x1.ffffeffc00000p-60},
{0x1.0000000000008p+12, -0x1.fffff00400000p-60, -0x1.00003ffffe000p-40, -0x1.ffffe00000000p-112, 0x1.0000000000007p+12, -0x1.7ffff40100000p-58},
{-0x1.0000000000008p+12, 0x1.fffff00400000p-60, 0x1.00003ffffe000p-40, 0x1.ffffe00000000p-112, -0x1.0000000000007p+12, 0x1.7ffff40100000p-58},
{-0x1.0000000000008p+12, 0x1.fffff00400000p-60, -0x1.00003ffffe000p-40, -0x1.ffffe00000000p-112, -0x1.0000000000009p+12, -0x1.ffffeffc00000p-60},
{0x1.0000000000000p-26, 0x1.ffffffffffc00p-89, 0x1.000001ffe0000p+32, 0x1.ff87ff9f801c0p-27, 0x1.000001ffe0000p+32, 0x1.ffc3ffcfc00e0p-26},
{0x1.0000000000000p-26, 0x1.ffffffffffc00p-89, -0x1.000001ffe0000p+32, -0x1.ff87ff9f801c0p-27, -0x1.000001ffe0000p+32, 0x1.e00181ff90000p-37},
{-0x1.0000000000000p-26, -0x1.ffffffffffc00p-89, 0x1.000001ffe0000p+32, 0x1.ff87ff9f801c0p-27, 0x1.000001ffe0000p+32, -0x1.e00181ff90000p-37},
{-0x1.0000000000000p-26, -0x1.ffffffffffc00p-89, -0x1.000001ffe0000p+32, -0x1.ff87ff9f801c0p-27, -0x1.000001ffe0000p+32, -0x1.ffc3ffcfc00e0p-26},
{0x1.0000000000400p+5, -0x1.f800000002000p-56, 0x1.03fffffffff00p+2, 0x1.fffe7fffffffep-52, 0x1.20800000003e0p+5, 0x1.e07e7fffffe00p-52},
{0x1.0000000000400p+5, -0x1.f800000002000p-56, -0x1.03fffffffff00p+2, -0x1.fffe7fffffffep-52, 0x1.bf00000000840p+4, -0x1.0fbf400000100p-51},
{-0x1.0000000000400p+5, 0x1.f800000002000p-56, 0x1.03fffffffff00p+2, 0x1.fffe7fffffffep-52, -0x1.bf00000000840p+4, 0x1.0fbf400000100p-51},
{-0x1.0000000000400p+5, 0x1.f800000002000p-56, -0x1.03fffffffff00p+2, -0x1.fffe7fffffffep-52, -0x1.20800000003e0p+5, -0x1.e07e7fffffe00p-52},
{0x1.0000000000000p+13, 0x1.fffffffffff00p-43, 0x1.000007c000000p-28, 0x1.fff8000000000p-84, 0x1.0000000000800p+13, 0x1.01f0000000780p-42},
{0x1.0000000000000p+13, 0x1.fffffffffff00p-43, -0x1.000007c000000p-28, -0x1.fff8000000000p-84, 0x1.ffffffffff000p+12, 0x1.fc1ffffffef00p-43},
{-0x1.0000000000000p+13, -0x1.fffffffffff00p-43, 0x1.000007c000000p-28, 0x1.fff8000000000p-84, -0x1.ffffffffff000p+12, -0x1.fc1ffffffef00p-43},
{-0x1.0000000000000p+13, -0x1.fffffffffff00p-43, -0x1.000007c000000p-28, -0x1.fff8000000000p-84, -0x1.0000000000800p+13, -0x1.01f0000000780p-42},
{0x1.0000000000400p-35, -0x1.fffc0003fffd0p-92, 0x1.0000000000000p+4, 0x1.fffffff800000p-52, 0x1.0000000002000p+4, 0x1.0000003bff000p-51},
{0x1.0000000000400p-35, -0x1.fffc0003fffd0p-92, -0x1.0000000000000p+4, -0x1.fffffff800000p-52, -0x1.fffffffffc000p+3, -0x1.ffffff7802000p-52},
{-0x1.0000000000400p-35, 0x1.fffc0003fffd0p-92, 0x1.0000000000000p+4, 0x1.fffffff800000p-52, 0x1.fffffffffc000p+3, 0x1.ffffff7802000p-52},
{-0x1.0000000000400p-35, 0x1.fffc0003fffd0p-92, -0x1.0000000000000p+4, -0x1.fffffff800000p-52, -0x1.0000000002000p+4, -0x1.0000003bff000p-51},
{0x1.0000000002000p-51, -0x1.ffff040000000p-122, 0x1.03fffff800000p-40, 0x1.fffffe0000000p-108, 0x1.041ffff800004p-40, 0x1.fff7fe03f0000p-108},
{0x1.0000000002000p-51, -0x1.ffff040000000p-122, -0x1.03fffff800000p-40, -0x1.fffffe0000000p-108, -0x1.03dffff7ffffcp-40, -0x1.0003fefe08000p-107},
{-0x1.0000000002000p-51, 0x1.ffff040000000p-122, 0x1.03fffff800000p-40, 0x1.fffffe0000000p-108, 0x1.03dffff7ffffcp-40, 0x1.0003fefe08000p-107},
{-0x1.0000000002000p-51, 0x1.ffff040000000p-122, -0x1.03fffff800000p-40, -0x1.fffffe0000000p-108, -0x1.041ffff800004p-40, -0x1.fff7fe03f0000p-108},
{0x1.0000008000000p-21, -0x1.ffffff8200000p-78, 0x1.000fff8000000p+9, 0x1.fffffe0000000p-53, 0x1.000fff8400000p+9, 0x1.03fffffa00000p-46},
{0x1.0000008000000p-21, -0x1.ffffff8200000p-78, -0x1.000fff8000000p+9, -0x1.fffffe0000000p-53, -0x1.000fff7c00000p+9, 0x1.f800000400000p-47},
{-0x1.0000008000000p-21, 0x1.ffffff8200000p-78, 0x1.000fff8000000p+9, 0x1.fffffe0000000p-53, 0x1.000fff7c00000p+9, -0x1.f800000400000p-47},
{-0x1.0000008000000p-21, 0x1.ffffff8200000p-78, -0x1.000fff8000000p+9, -0x1.fffffe0000000p-53, -0x1.000fff8400000p+9, -0x1.03fffffa00000p-46},
{0x1.0000000000000p-33, 0x1.0001f9e000000p-93, 0x1.0000000000040p+19, -0x1.fffffc0080000p-53, 0x1.0000000000041p+19, -0x1.fffffc0080000p-53},
{0x1.0000000000000p-33, 0x1.0001f9e000000p-93, -0x1.0000000000040p+19, 0x1.fffffc0080000p-53, -0x1.000000000003fp+19, 0x1.fffffc0080000p-53},
{-0x1.0000000000000p-33, -0x1.0001f9e000000p-93, 0x1.0000000000040p+19, -0x1.fffffc0080000p-53, 0x1.000000000003fp+19, -0x1.fffffc0080000p-53},
{-0x1.0000000000000p-33, -0x1.0001f9e000000p-93, -0x1.0000000000040p+19, 0x1.fffffc0080000p-53, -0x1.0000000000041p+19, 0x1.fffffc0080000p-53},
{0x1.0000ff0007e00p-13, 0x1.ffc001ff00000p-80, 0x1.0000010000000p-33, -0x1.ffc007ffffe20p-91, 0x1.00010f0007f00p-13, 0x1.ff8009fe00000p-80},
{0x1.0000ff0007e00p-13, 0x1.ffc001ff00000p-80, -0x1.0000010000000p-33, 0x1.ffc007ffffe20p-91, 0x1.0000ef0007d00p-13, 0x1.fffffa0000000p-80},
{-0x1.0000ff0007e00p-13, -0x1.ffc001ff00000p-80, 0x1.0000010000000p-33, -0x1.ffc007ffffe20p-91, -0x1.0000ef0007d00p-13, -0x1.fffffa0000000p-80},
{-0x1.0000ff0007e00p-13, -0x1.ffc001ff00000p-80, -0x1.0000010000000p-33, 0x1.ffc007ffffe20p-91, -0x1.00010f0007f00p-13, -0x1.ff8009fe00000p-80},
{0x1.07ffffff1fe00p-53, 0x1.fc00000000000p-150, 0x1.0040000000000p-23, -0x1.c000000400000p-84, 0x1.0040000420000p-23, -0x1.1808000200000p-83},
{0x1.07ffffff1fe00p-53, 0x1.fc00000000000p-150, -0x1.0040000000000p-23, 0x1.c000000400000p-84, -0x1.003ffffbe0000p-23, 0x1.4ff0000400000p-84},
{-0x1.07ffffff1fe00p-53, -0x1.fc00000000000p-150, 0x1.0040000000000p-23, -0x1.c000000400000p-84, 0x1.003ffffbe0000p-23, -0x1.4ff0000400000p-84},
{-0x1.07ffffff1fe00p-53, -0x1.fc00000000000p-150, -0x1.0040000000000p-23, 0x1.c000000400000p-84, -0x1.0040000420000p-23, 0x1.1808000200000p-83},
{0x1.0080000000000p+15, -0x1.0000000000000p-90, 0x1.00000001fffc0p-35, 0x1.fffff80f80000p-107, 0x1.0080000000004p+15, 0x1.fffbfe0000000p-67},
{0x1.0080000000000p+15, -0x1.0000000000000p-90, -0x1.00000001fffc0p-35, -0x1.fffff80f80000p-107, 0x1.007fffffffffcp+15, -0x1.fffc020000000p-67},
{-0x1.0080000000000p+15, 0x1.0000000000000p-90, 0x1.00000001fffc0p-35, 0x1.fffff80f80000p-107, -0x1.007fffffffffcp+15, 0x1.fffc020000000p-67},
{-0x1.0080000000000p+15, 0x1.0000000000000p-90, -0x1.00000001fffc0p-35, -0x1.fffff80f80000p-107, -0x1.0080000000004p+15, -0x1.fffbfe0000000p-67},
{0x1.000000003ff00p+2, 0x0.0000000000000p+0, 0x1.0000000000000p+46, 0x1.f83007ffffe00p-14, 0x1.0000000000100p+46, 0x1.f83047efffe00p-14},
{0x1.000000003ff00p+2, 0x0.0000000000000p+0, -0x1.0000000000000p+46, -0x1.f83007ffffe00p-14, -0x1.ffffffffffe00p+45, -0x1.f82fc80fffe00p-14},
{-0x1.000000003ff00p+2, 0x0.0000000000000p+0, 0x1.0000000000000p+46, 0x1.f83007ffffe00p-14, 0x1.ffffffffffe00p+45, 0x1.f82fc80fffe00p-14},
{-0x1.000000003ff00p+2, 0x0.0000000000000p+0, -0x1.0000000000000p+46, -0x1.f83007ffffe00p-14, -0x1.0000000000100p+46, -0x1.f83047efffe00p-14},
{0x1.00003fffffe00p+13, 0x1.fffffc0000380p-47, 0x1.0000010000000p-8, -0x1.ffffff8000008p-64, 0x1.000047ffffe80p+13, 0x1.fffefc0000780p-47},
{0x1.00003fffffe00p+13, 0x1.fffffc0000380p-47, -0x1.0000010000000p-8, 0x1.ffffff8000008p-64, 0x1.000037ffffd80p+13, 0x1.00007dfffffc0p-46},
{-0x1.00003fffffe00p+13, -0x1.fffffc0000380p-47, 0x1.0000010000000p-8, -0x1.ffffff8000008p-64, -0x1.000037ffffd80p+13, -0x1.00007dfffffc0p-46},
{-0x1.00003fffffe00p+13, -0x1.fffffc0000380p-47, -0x1.0000010000000p-8, 0x1.ffffff8000008p-64, -0x1.000047ffffe80p+13, -0x1.fffefc0000780p-47},
{0x1.0000000080000p+13, -0x1.fffffc0010000p-56, 0x1.3fffff8000000p-19, 0x1.fffff800fe000p-85, 0x1.00000001c0000p+13, -0x1.001fffffbf010p-44},
{0x1.0000000080000p+13, -0x1.fffffc0010000p-56, -0x1.3fffff8000000p-19, -0x1.fffff800fe000p-85, 0x1.fffffffe80000p+12, 0x1.ffc000007dfe0p-45},
{-0x1.0000000080000p+13, 0x1.fffffc0010000p-56, 0x1.3fffff8000000p-19, 0x1.fffff800fe000p-85, -0x1.fffffffe80000p+12, -0x1.ffc000007dfe0p-45},
{-0x1.0000000080000p+13, 0x1.fffffc0010000p-56, -0x1.3fffff8000000p-19, -0x1.fffff800fe000p-85, -0x1.00000001c0000p+13, 0x1.001fffffbf010p-44},
{0x1.0003f001fc3f0p-13, 0x1.fffffc0000000p-69, 0x1.0000000000000p+27, 0x1.fc1fffc0000fep-27, 0x1.0000000001001p+27, -0x1.07df81303f702p-27},
{0x1.0003f001fc3f0p-13, 0x1.fffffc0000000p-69, -0x1.0000000000000p+27, -0x1.fc1fffc0000fep-27, -0x1.fffffffffe001p+26, 0x1.ffc0fe9f80e04p-28},
{-0x1.0003f001fc3f0p-13, -0x1.fffffc0000000p-69, 0x1.0000000000000p+27, 0x1.fc1fffc0000fep-27, 0x1.fffffffffe001p+26, -0x1.ffc0fe9f80e04p-28},
{-0x1.0003f001fc3f0p-13, -0x1.fffffc0000000p-69, -0x1.0000000000000p+27, -0x1.fc1fffc0000fep-27, -0x1.0000000001001p+27, 0x1.07df81303f702p-27},
{0x1.0000000000000p+49, 0x1.fffffffffffc0p-10, 0x1.0000000000000p+12, 0x1.ffffffffffe00p-50, 0x1.0000000008000p+49, 0x1.0000000000fe0p-9},
{0x1.0000000000000p+49, 0x1.fffffffffffc0p-10, -0x1.0000000000000p+12, -0x1.ffffffffffe00p-50, 0x1.fffffffff0000p+48, 0x1.fffffffffdfc0p-10},
{-0x1.0000000000000p+49, -0x1.fffffffffffc0p-10, 0x1.0000000000000p+12, 0x1.ffffffffffe00p-50, -0x1.fffffffff0000p+48, -0x1.fffffffffdfc0p-10},
{-0x1.0000000000000p+49, -0x1.fffffffffffc0p-10, -0x1.0000000000000p+12, -0x1.ffffffffffe00p-50, -0x1.0000000008000p+49, -0x1.0000000000fe0p-9},
{0x1.0000000100000p-51, -0x1.0000000000000p-130, 0x1.0000004000000p+23, -0x1.0000000000000p-57, 0x1.0000004000000p+23, 0x1.f800000000000p-52},
{0x1.0000000100000p-51, -0x1.0000000000000p-130, -0x1.0000004000000p+23, 0x1.0000000000000p-57, -0x1.0000004000000p+23, 0x1.0400000000000p-51},
{-0x1.0000000100000p-51, 0x1.0000000000000p-130, 0x1.0000004000000p+23, -0x1.0000000000000p-57, 0x1.0000004000000p+23, -0x1.0400000000000p-51},
{-0x1.0000000100000p-51, 0x1.0000000000000p-130, -0x1.0000004000000p+23, 0x1.0000000000000p-57, -0x1.0000004000000p+23, -0x1.f800000000000p-52},
{0x1.0000010000000p-27, -0x1.ffffe00001000p-83, 0x1.000ffffff007fp-2, 0x1.ffffc00000000p-83, 0x1.0010007ff0087p-2, -0x1.0000000000000p-102},
{0x1.0000010000000p-27, -0x1.ffffe00001000p-83, -0x1.000ffffff007fp-2, -0x1.ffffc00000000p-83, -0x1.000fff7ff0077p-2, -0x1.ffffd00000000p-82},
{-0x1.0000010000000p-27, 0x1.ffffe00001000p-83, 0x1.000ffffff007fp-2, 0x1.ffffc00000000p-83, 0x1.000fff7ff0077p-2, 0x1.ffffd00000000p-82},
{-0x1.0000010000000p-27, 0x1.ffffe00001000p-83, -0x1.000ffffff007fp-2, -0x1.ffffc00000000p-83, -0x1.0010007ff0087p-2, 0x1.0000000000000p-102},
{0x1.1000fffffc000p+50, 0x1.fffffc003f000p-6, 0x1.0000000001000p+30, -0x1.0000000000000p-75, 0x1.10010ffffc000p+50, 0x1.07fffe001f800p-5},
{0x1.1000fffffc000p+50, 0x1.fffffc003f000p-6, -0x1.0000000001000p+30, 0x1.0000000000000p-75, 0x1.1000effffc000p+50, 0x1.effffc003f000p-6},
{-0x1.1000fffffc000p+50, -0x1.fffffc003f000p-6, 0x1.0000000001000p+30, -0x1.0000000000000p-75, -0x1.1000effffc000p+50, -0x1.effffc003f000p-6},
{-0x1.1000fffffc000p+50, -0x1.fffffc003f000p-6, -0x1.0000000001000p+30, 0x1.0000000000000p-75, -0x1.10010ffffc000p+50, -0x1.07fffe001f800p-5},
{0x1.00000fff80000p+32, 0x1.fffe000000000p-53, 0x1.0000001f00000p+41, 0x1.fffffff000000p-36, 0x1.00800026ffc00p+41, 0x1.00007ff800000p-35},
{0x1.00000fff80000p+32, 0x1.fffe000000000p-53, -0x1.0000001f00000p+41, -0x1.fffffff000000p-36, -0x1.ff00002e00800p+40, -0x1.fffefff000000p-36},
{-0x1.00000fff80000p+32, -0x1.fffe000000000p-53, 0x1.0000001f00000p+41, 0x1.fffffff000000p-36, 0x1.ff00002e00800p+40, 0x1.fffefff000000p-36},
{-0x1.00000fff80000p+32, -0x1.fffe000000000p-53, -0x1.0000001f00000p+41, -0x1.fffffff000000p-36, -0x1.00800026ffc00p+41, -0x1.00007ff800000p-35},
{0x1.3fdfffff02000p+34, -0x1.0000000000000p-44, 0x1.0000000000000p+31, 0x1.ffffffffffe00p-30, 0x1.5fdfffff02000p+34, 0x1.fffc000000000p-30},
{0x1.3fdfffff02000p+34, -0x1.0000000000000p-44, -0x1.0000000000000p+31, -0x1.ffffffffffe00p-30, 0x1.1fdfffff02000p+34, -0x1.0002000000000p-29},
{-0x1.3fdfffff02000p+34, 0x1.0000000000000p-44, 0x1.0000000000000p+31, 0x1.ffffffffffe00p-30, -0x1.1fdfffff02000p+34, 0x1.0002000000000p-29},
{-0x1.3fdfffff02000p+34, 0x1.0000000000000p-44, -0x1.0000000000000p+31, -0x1.ffffffffffe00p-30, -0x1.5fdfffff02000p+34, -0x1.fffc000000000p-30},
{0x1.0000000000008p+41, -0x1.e0ffffd002000p-25, 0x1.0000000000400p+36, 0x1.ffffffff03ffep-18, 0x1.0800000000028p+41, 0x1.fc3dffff63fc0p-18},
{0x1.0000000000008p+41, -0x1.e0ffffd002000p-25, -0x1.0000000000400p+36, -0x1.ffffffff03ffep-18, 0x1.effffffffffd0p+40, -0x1.01e0ffff52020p-17},
{-0x1.0000000000008p+41, 0x1.e0ffffd002000p-25, 0x1.0000000000400p+36, 0x1.ffffffff03ffep-18, -0x1.effffffffffd0p+40, 0x1.01e0ffff52020p-17},
{-0x1.0000000000008p+41, 0x1.e0ffffd002000p-25, -0x1.0000000000400p+36, -0x1.ffffffff03ffep-18, -0x1.0800000000028p+41, -0x1.fc3dffff63fc0p-18},
{0x1.000000fffe020p+17, -0x1.f800100000000p-56, 0x1.0001fff80ffc0p+20, 0x1.ff00000000000p-77, 0x1.200200180fbc4p+20, -0x1.f800000800000p-56},
{0x1.000000fffe020p+17, -0x1.f800100000000p-56, -0x1.0001fff80ffc0p+20, -0x1.ff00000000000p-77, -0x1.c003ffb020778p+19, -0x1.f8001ff800000p-56},
{-0x1.000000fffe020p+17, 0x1.f800100000000p-56, 0x1.0001fff80ffc0p+20, 0x1.ff00000000000p-77, 0x1.c003ffb020778p+19, 0x1.f8001ff800000p-56},
{-0x1.000000fffe020p+17, 0x1.f800100000000p-56, -0x1.0001fff80ffc0p+20, -0x1.ff00000000000p-77, -0x1.200200180fbc4p+20, 0x1.f800000800000p-56},
{0x1.4000000000000p+46, -0x1.0000000000000p-59, 0x1.0000000000010p-11, 0x1.e000000000000p-79, 0x1.4000000000000p+46, 0x1.0000000000000p-11},
{0x1.4000000000000p+46, -0x1.0000000000000p-59, -0x1.0000000000010p-11, -0x1.e000000000000p-79, 0x1.4000000000000p+46, -0x1.0000000000020p-11},
{-0x1.4000000000000p+46, 0x1.0000000000000p-59, 0x1.0000000000010p-11, 0x1.e000000000000p-79, -0x1.4000000000000p+46, 0x1.0000000000020p-11},
{-0x1.4000000000000p+46, 0x1.0000000000000p-59, -0x1.0000000000010p-11, -0x1.e000000000000p-79, -0x1.4000000000000p+46, -0x1.0000000000000p-11},
{0x1.2000000000000p+0, -0x1.fffffff01ff20p-57, 0x1.0000000000000p-33, 0x1.fff8003ff0000p-90, 0x1.2000000080000p+0, -0x1.ffffffef1ff60p-57},
{0x1.2000000000000p+0, -0x1.fffffff01ff20p-57, -0x1.0000000000000p-33, -0x1.fff8003ff0000p-90, 0x1.1fffffff80000p+0, -0x1.fffffff11fee0p-57},
{-0x1.2000000000000p+0, 0x1.fffffff01ff20p-57, 0x1.0000000000000p-33, 0x1.fff8003ff0000p-90, -0x1.1fffffff80000p+0, 0x1.fffffff11fee0p-57},
{-0x1.2000000000000p+0, 0x1.fffffff01ff20p-57, -0x1.0000000000000p-33, -0x1.fff8003ff0000p-90, -0x1.2000000080000p+0, 0x1.ffffffef1ff60p-57},
{0x1.2000000000000p+14, -0x1.fc08000000000p-78, 0x1.0000000038004p+6, -0x1.fffffc4000000p-63, 0x1.2100000000380p+14, 0x1.ffff7fff02ec0p-45},
{0x1.2000000000000p+14, -0x1.fc08000000000p-78, -0x1.0000000038004p+6, 0x1.fffffc4000000p-63, 0x1.1effffffffc80p+14, -0x1.ffff8000fef40p-45},
{-0x1.2000000000000p+14, 0x1.fc08000000000p-78, 0x1.0000000038004p+6, -0x1.fffffc4000000p-63, -0x1.1effffffffc80p+14, 0x1.ffff8000fef40p-45},
{-0x1.2000000000000p+14, 0x1.fc08000000000p-78, -0x1.0000000038004p+6, 0x1.fffffc4000000p-63, -0x1.2100000000380p+14, -0x1.ffff7fff02ec0p-45},
{0x1.0000000000200p+30, 0x0.0000000000000p+0, 0x1.00001ffff0000p+10, 0x1.ffff80001f000p-47, 0x1.0000100002200p+30, -0x1.ffffe00008000p-27},
{0x1.0000000000200p+30, 0x0.0000000000000p+0, -0x1.00001ffff0000p+10, -0x1.ffff80001f000p-47, 0x1.ffffdffffc400p+29, 0x1.ffffe00008000p-27},
{-0x1.0000000000200p+30, 0x0.0000000000000p+0, 0x1.00001ffff0000p+10, 0x1.ffff80001f000p-47, -0x1.ffffdffffc400p+29, -0x1.ffffe00008000p-27},
{-0x1.0000000000200p+30, 0x0.0000000000000p+0, -0x1.00001ffff0000p+10, -0x1.ffff80001f000p-47, -0x1.0000100002200p+30, 0x1.ffffe00008000p-27},
{0x1.0000100000000p-17, -0x1.0000000000000p-122, 0x1.0038000800000p+15, -0x1.fffffe0000080p-45, 0x1.0038000900001p+15, -0x1.fffffe0000080p-45},
{0x1.0000100000000p-17, -0x1.0000000000000p-122, -0x1.0038000800000p+15, 0x1.fffffe0000080p-45, -0x1.00380006fffffp+15, 0x1.fffffe0000080p-45},
{-0x1.0000100000000p-17, 0x1.0000000000000p-122, 0x1.0038000800000p+15, -0x1.fffffe0000080p-45, 0x1.00380006fffffp+15, -0x1.fffffe0000080p-45},
{-0x1.0000100000000p-17, 0x1.0000000000000p-122, -0x1.0038000800000p+15, 0x1.fffffe0000080p-45, -0x1.0038000900001p+15, 0x1.fffffe0000080p-45},
{0x1.0000000000004p-11, -0x1.ffffe00400000p-84, 0x1.0000000000000p-12, 0x1.ffffffffffff8p-68, 0x1.8000000000004p-11, 0x1.fffe00001ffc0p-68},
{0x1.0000000000004p-11, -0x1.ffffe00400000p-84, -0x1.0000000000000p-12, -0x1.ffffffffffff8p-68, 0x1.0000000000008p-12, -0x1.0000fffff001cp-67},
{-0x1.0000000000004p-11, 0x1.ffffe00400000p-84, 0x1.0000000000000p-12, 0x1.ffffffffffff8p-68, -0x1.0000000000008p-12, 0x1.0000fffff001cp-67},
{-0x1.0000000000004p-11, 0x1.ffffe00400000p-84, -0x1.0000000000000p-12, -0x1.ffffffffffff8p-68, -0x1.8000000000004p-11, -0x1.fffe00001ffc0p-68},
{0x1.0000000000100p-33, -0x1.ffffe7e400000p-103, 0x1.1ffe000008000p-33, -0x1.fffc000004000p-100, 0x1.0fff000004080p-32, -0x1.1ffdfe7e40000p-99},
{0x1.0000000000100p-33, -0x1.ffffe7e400000p-103, -0x1.1ffe000008000p-33, 0x1.fffc000004000p-100, -0x1.ffe000007f000p-37, 0x1.bffc030384000p-100},
{-0x1.0000000000100p-33, 0x1.ffffe7e400000p-103, 0x1.1ffe000008000p-33, -0x1.fffc000004000p-100, 0x1.ffe000007f000p-37, -0x1.bffc030384000p-100},
{-0x1.0000000000100p-33, 0x1.ffffe7e400000p-103, -0x1.1ffe000008000p-33, 0x1.fffc000004000p-100, -0x1.0fff000004080p-32, 0x1.1ffdfe7e40000p-99},
{0x1.0000000000004p-52, -0x1.0000000000000p-157, 0x1.00000007ffffep-10, 0x1.ffffe000f0000p-71, 0x1.00000008003fep-10, 0x1.ffffe002f0000p-71},
{0x1.0000000000004p-52, -0x1.0000000000000p-157, -0x1.00000007ffffep-10, -0x1.ffffe000f0000p-71, -0x1.00000007ffbfep-10, -0x1.ffffdffef0000p-71},
{-0x1.0000000000004p-52, 0x1.0000000000000p-157, 0x1.00000007ffffep-10, 0x1.ffffe000f0000p-71, 0x1.00000007ffbfep-10, 0x1.ffffdffef0000p-71},
{-0x1.0000000000004p-52, 0x1.0000000000000p-157, -0x1.00000007ffffep-10, -0x1.ffffe000f0000p-71, -0x1.00000008003fep-10, -0x1.ffffe002f0000p-71},
{0x1.0000000000000p+3, 0x1.fffffff000000p-58, 0x1.0001fffc00001p-42, -0x1.0000000000000p-121, 0x1.0000000000080p+3, 0x1.fffdfff808000p-57},
{0x1.0000000000000p+3, 0x1.fffffff000000p-58, -0x1.0001fffc00001p-42, 0x1.0000000000000p-121, 0x1.fffffffffff00p+2, 0x1.fff7f80000000p-73},
{-0x1.0000000000000p+3, -0x1.fffffff000000p-58, 0x1.0001fffc00001p-42, -0x1.0000000000000p-121, -0x1.fffffffffff00p+2, -0x1.fff7f80000000p-73},
{-0x1.0000000000000p+3, -0x1.fffffff000000p-58, -0x1.0001fffc00001p-42, 0x1.0000000000000p-121, -0x1.0000000000080p+3, -0x1.fffdfff808000p-57},
{0x1.00007fdfffff0p+50, 0x1.fff000f803fe0p-8, 0x1.0080000000000p-8, -0x1.0000000000000p-100, 0x1.00007fdfffff0p+50, 0x1.8038007c01ff0p-7},
{0x1.00007fdfffff0p+50, 0x1.fff000f803fe0p-8, -0x1.0080000000000p-8, 0x1.0000000000000p-100, 0x1.00007fdfffff0p+50, 0x1.fee001f007fc0p-9},
{-0x1.00007fdfffff0p+50, -0x1.fff000f803fe0p-8, 0x1.0080000000000p-8, -0x1.0000000000000p-100, -0x1.00007fdfffff0p+50, -0x1.fee001f007fc0p-9},
{-0x1.00007fdfffff0p+50, -0x1.fff000f803fe0p-8, -0x1.0080000000000p-8, 0x1.0000000000000p-100, -0x1.00007fdfffff0p+50, -0x1.8038007c01ff0p-7},
{0x1.0000000000080p+4, 0x0.0000000000000p+0, 0x1.0000080000000p+20, -0x1.0000000000000p-65, 0x1.0001080000000p+20, 0x1.fffffe0000000p-42},
{0x1.0000000000080p+4, 0x0.0000000000000p+0, -0x1.0000080000000p+20, 0x1.0000000000000p-65, -0x1.fffe100000000p+19, 0x1.0000010000000p-41},
{-0x1.0000000000080p+4, 0x0.0000000000000p+0, 0x1.0000080000000p+20, -0x1.0000000000000p-65, 0x1.fffe100000000p+19, -0x1.0000010000000p-41},
{-0x1.0000000000080p+4, 0x0.0000000000000p+0, -0x1.0000080000000p+20, 0x1.0000000000000p-65, -0x1.0001080000000p+20, -0x1.fffffe0000000p-42},
{0x1.0400000000000p+20, -0x1.0000000000000p-85, 0x1.0000000000400p+5, -0x1.fffff00008000p-51, 0x1.0402000000000p+20, 0x1.fff000007ffe0p-38},
{0x1.0400000000000p+20, -0x1.0000000000000p-85, -0x1.0000000000400p+5, 0x1.fffff00008000p-51, 0x1.03fe000000000p+20, -0x1.fff0000080020p-38},
{-0x1.0400000000000p+20, 0x1.0000000000000p-85, 0x1.0000000000400p+5, -0x1.fffff00008000p-51, -0x1.03fe000000000p+20, 0x1.fff0000080020p-38},
{-0x1.0400000000000p+20, 0x1.0000000000000p-85, -0x1.0000000000400p+5, 0x1.fffff00008000p-51, -0x1.0402000000000p+20, -0x1.fff000007ffe0p-38},
{0x1.0000000000001p+43, -0x1.fc0000c004000p-22, 0x1.0000000000200p+29, -0x1.ffe0000200000p-39, 0x1.0004000000001p+43, 0x1.fc07fdfe9ff80p-15},
{0x1.0000000000001p+43, -0x1.fc0000c004000p-22, -0x1.0000000000200p+29, 0x1.ffe0000200000p-39, 0x1.fff8000000002p+42, -0x1.01fbff00d0040p-14},
{-0x1.0000000000001p+43, 0x1.fc0000c004000p-22, 0x1.0000000000200p+29, -0x1.ffe0000200000p-39, -0x1.fff8000000002p+42, 0x1.01fbff00d0040p-14},
{-0x1.0000000000001p+43, 0x1.fc0000c004000p-22, -0x1.0000000000200p+29, 0x1.ffe0000200000p-39, -0x1.0004000000001p+43, -0x1.fc07fdfe9ff80p-15},
{0x1.00000c0000004p+1, -0x1.ff80000800000p-67, 0x1.18003ffff8000p+4, 0x1.ffffff8000000p-62, 0x1.3800417ff8001p+4, -0x1.ffe0ffc008040p-50},
{0x1.00000c0000004p+1, -0x1.ff80000800000p-67, -0x1.18003ffff8000p+4, -0x1.ffffff8000000p-62, -0x1.f0007cffeffffp+3, -0x1.07fdffc020000p-61},
{-0x1.00000c0000004p+1, 0x1.ff80000800000p-67, 0x1.18003ffff8000p+4, 0x1.ffffff8000000p-62, 0x1.f0007cffeffffp+3, 0x1.07fdffc020000p-61},
{-0x1.00000c0000004p+1, 0x1.ff80000800000p-67, -0x1.18003ffff8000p+4, -0x1.ffffff8000000p-62, -0x1.3800417ff8001p+4, 0x1.ffe0ffc008040p-50},
{0x1.1ffff80000040p+31, -0x1.0000000000000p-59, 0x1.000000000fe00p+44, 0x1.ffff7e0003ff8p-12, 0x1.0008ffffcfe00p+44, 0x1.0fffbf0001fecp-11},
{0x1.1ffff80000040p+31, -0x1.0000000000000p-59, -0x1.000000000fe00p+44, -0x1.ffff7e0003ff8p-12, -0x1.ffee00009fc00p+43, -0x1.dfff7e0004018p-12},
{-0x1.1ffff80000040p+31, 0x1.0000000000000p-59, 0x1.000000000fe00p+44, 0x1.ffff7e0003ff8p-12, 0x1.ffee00009fc00p+43, 0x1.dfff7e0004018p-12},
{-0x1.1ffff80000040p+31, 0x1.0000000000000p-59, -0x1.000000000fe00p+44, -0x1.ffff7e0003ff8p-12, -0x1.0008ffffcfe00p+44, -0x1.0fffbf0001fecp-11},
{0x1.3fc0000040000p-13, -0x1.fffffff802000p-69, 0x1.fffffffffff00p+30, 0x0.0000000000000p+0, 0x1.00000000000c0p+31, -0x1.fffffe0000100p-24},
{0x1.3fc0000040000p-13, -0x1.fffffff802000p-69, -0x1.fffffffffff00p+30, 0x0.0000000000000p+0, -0x1.ffffffffffc80p+30, -0x1.fffffe0000100p-24},
{-0x1.3fc0000040000p-13, 0x1.fffffff802000p-69, 0x1.fffffffffff00p+30, 0x0.0000000000000p+0, 0x1.ffffffffffc80p+30, 0x1.fffffe0000100p-24},
{-0x1.3fc0000040000p-13, 0x1.fffffff802000p-69, -0x1.fffffffffff00p+30, 0x0.0000000000000p+0, -0x1.00000000000c0p+31, 0x1.fffffe0000100p-24},
{0x1.00000001fffffp-1, 0x1.00001f8000000p-72, 0x1.0000000000200p-15, -0x1.0000000000000p-120, 0x1.00040001fffffp-1, 0x1.000400007e000p-58},
{0x1.00000001fffffp-1, 0x1.00001f8000000p-72, -0x1.0000000000200p-15, 0x1.0000000000000p-120, 0x1.fff80003ffffep-2, -0x1.fff7ffff04000p-59},
{-0x1.00000001fffffp-1, -0x1.00001f8000000p-72, 0x1.0000000000200p-15, -0x1.0000000000000p-120, -0x1.fff80003ffffep-2, 0x1.fff7ffff04000p-59},
{-0x1.00000001fffffp-1, -0x1.00001f8000000p-72, -0x1.0000000000200p-15, 0x1.0000000000000p-120, -0x1.00040001fffffp-1, -0x1.000400007e000p-58},
{0x1.0000000010000p+26, -0x1.0000000000000p-51, 0x1.1fff010000000p-53, 0x1.ff8c0000007fep-107, 0x1.0000000010000p+26, -0x1.70007f8000000p-52},
{0x1.0000000010000p+26, -0x1.0000000000000p-51, -0x1.1fff010000000p-53, -0x1.ff8c0000007fep-107, 0x1.0000000010000p+26, -0x1.47ffc04000000p-51},
{-0x1.0000000010000p+26, 0x1.0000000000000p-51, 0x1.1fff010000000p-53, 0x1.ff8c0000007fep-107, -0x1.0000000010000p+26, 0x1.47ffc04000000p-51},
{-0x1.0000000010000p+26, 0x1.0000000000000p-51, -0x1.1fff010000000p-53, -0x1.ff8c0000007fep-107, -0x1.0000000010000p+26, 0x1.70007f8000000p-52},
{0x1.0000000000002p-44, -0x1.0000000000000p-131, 0x1.0000000000200p+33, -0x1.0000000000000p-52, 0x1.0000000000200p+33, 0x1.fe00000000000p-45},
{0x1.0000000000002p-44, -0x1.0000000000000p-131, -0x1.0000000000200p+33, 0x1.0000000000000p-52, -0x1.0000000000200p+33, 0x1.0100000000000p-44},
{-0x1.0000000000002p-44, 0x1.0000000000000p-131, 0x1.0000000000200p+33, -0x1.0000000000000p-52, 0x1.0000000000200p+33, -0x1.0100000000000p-44},
{-0x1.0000000000002p-44, 0x1.0000000000000p-131, -0x1.0000000000200p+33, 0x1.0000000000000p-52, -0x1.0000000000200p+33, -0x1.fe00000000000p-45},
{0x1.0000000000002p+32, -0x1.ffff800008000p-35, 0x1.000000003fffep-25, 0x1.f81fc007c0000p-96, 0x1.0000000000002p+32, 0x1.ff0000407ffc0p-26},
{0x1.0000000000002p+32, -0x1.ffff800008000p-35, -0x1.000000003fffep-25, -0x1.f81fc007c0000p-96, 0x1.0000000000002p+32, -0x1.007fffe040020p-25},
{-0x1.0000000000002p+32, 0x1.ffff800008000p-35, 0x1.000000003fffep-25, 0x1.f81fc007c0000p-96, -0x1.0000000000002p+32, 0x1.007fffe040020p-25},
{-0x1.0000000000002p+32, 0x1.ffff800008000p-35, -0x1.000000003fffep-25, -0x1.f81fc007c0000p-96, -0x1.0000000000002p+32, -0x1.ff0000407ffc0p-26},
{0x1.7ffffffff0000p+18, 0x1.ffff07fffc000p-46, 0x1.000000000ffe0p-32, 0x1.ff0000003c000p-95, 0x1.7ffffffff0004p+18, 0x1.ffff0bff7c000p-46},
{0x1.7ffffffff0000p+18, 0x1.ffff07fffc000p-46, -0x1.000000000ffe0p-32, -0x1.ff0000003c000p-95, 0x1.7fffffffefffcp+18, 0x1.ffff04007c000p-46},
{-0x1.7ffffffff0000p+18, -0x1.ffff07fffc000p-46, 0x1.000000000ffe0p-32, 0x1.ff0000003c000p-95, -0x1.7fffffffefffcp+18, -0x1.ffff04007c000p-46},
{-0x1.7ffffffff0000p+18, -0x1.ffff07fffc000p-46, -0x1.000000000ffe0p-32, -0x1.ff0000003c000p-95, -0x1.7ffffffff0004p+18, -0x1.ffff0bff7c000p-46},
{0x1.00007fff00080p-52, -0x1.ffffff0001000p-117, 0x1.00000000007c0p+41, 0x1.ffffff8000000p-26, 0x1.00000000007c0p+41, 0x1.ffffffc000000p-26},
{0x1.00007fff00080p-52, -0x1.ffffff0001000p-117, -0x1.00000000007c0p+41, -0x1.ffffff8000000p-26, -0x1.00000000007c0p+41, -0x1.ffffff4000000p-26},
{-0x1.00007fff00080p-52, 0x1.ffffff0001000p-117, 0x1.00000000007c0p+41, 0x1.ffffff8000000p-26, 0x1.00000000007c0p+41, 0x1.ffffff4000000p-26},
{-0x1.00007fff00080p-52, 0x1.ffffff0001000p-117, -0x1.00000000007c0p+41, -0x1.ffffff8000000p-26, -0x1.00000000007c0p+41, -0x1.ffffffc000000p-26},
{0x1.1fff81ff00000p+13, 0x1.e007ffff00000p-44, 0x1.1ffffffff8000p-5, 0x1.fe001ffffffe0p-61, 0x1.1fffc9ff00000p+13, -0x1.0ffb808078000p-43},
{0x1.1fff81ff00000p+13, 0x1.e007ffff00000p-44, -0x1.1ffffffff8000p-5, -0x1.fe001ffffffe0p-61, 0x1.1fff39ff00000p+13, 0x1.7801c03fbc000p-42},
{-0x1.1fff81ff00000p+13, -0x1.e007ffff00000p-44, 0x1.1ffffffff8000p-5, 0x1.fe001ffffffe0p-61, -0x1.1fff39ff00000p+13, -0x1.7801c03fbc000p-42},
{-0x1.1fff81ff00000p+13, -0x1.e007ffff00000p-44, -0x1.1ffffffff8000p-5, -0x1.fe001ffffffe0p-61, -0x1.1fffc9ff00000p+13, 0x1.0ffb808078000p-43},
{0x1.0000000000000p+15, 0x1.fffc03fe00700p-46, 0x1.0000000000001p+24, -0x1.fff8060000f34p-31, 0x1.0080000000001p+24, -0x1.fff40607f8f74p-31},
{0x1.0000000000000p+15, 0x1.fffc03fe00700p-46, -0x1.0000000000001p+24, 0x1.fff8060000f34p-31, -0x1.ff00000000002p+23, 0x1.fffc05f808ef4p-31},
{-0x1.0000000000000p+15, -0x1.fffc03fe00700p-46, 0x1.0000000000001p+24, -0x1.fff8060000f34p-31, 0x1.ff00000000002p+23, -0x1.fffc05f808ef4p-31},
{-0x1.0000000000000p+15, -0x1.fffc03fe00700p-46, -0x1.0000000000001p+24, 0x1.fff8060000f34p-31, -0x1.0080000000001p+24, 0x1.fff40607f8f74p-31},
{0x1.001ffffc00080p+19, -0x1.fe00000200000p-42, 0x1.0000000001000p-24, -0x1.fe3ffffff0080p-84, 0x1.001ffffc00280p+19, -0x1.fdfffc0200800p-42},
{0x1.001ffffc00080p+19, -0x1.fe00000200000p-42, -0x1.0000000001000p-24, 0x1.fe3ffffff0080p-84, 0x1.001ffffbffe80p+19, -0x1.fe000401ff800p-42},
{-0x1.001ffffc00080p+19, 0x1.fe00000200000p-42, 0x1.0000000001000p-24, -0x1.fe3ffffff0080p-84, -0x1.001ffffbffe80p+19, 0x1.fe000401ff800p-42},
{-0x1.001ffffc00080p+19, 0x1.fe00000200000p-42, -0x1.0000000001000p-24, 0x1.fe3ffffff0080p-84, -0x1.001ffffc00280p+19, 0x1.fdfffc0200800p-42},
{0x1.00001fffff800p+37, 0x1.ffffff801ffc0p-22, 0x1.0000000002000p+12, -0x1.0000000000000p-93, 0x1.0000207fff800p+37, 0x1.03ffffc00ffe0p-21},
{0x1.00001fffff800p+37, 0x1.ffffff801ffc0p-22, -0x1.0000000002000p+12, 0x1.0000000000000p-93, 0x1.00001f7fff800p+37, 0x1.f7ffff801ffc0p-22},
{-0x1.00001fffff800p+37, -0x1.ffffff801ffc0p-22, 0x1.0000000002000p+12, -0x1.0000000000000p-93, -0x1.00001f7fff800p+37, -0x1.f7ffff801ffc0p-22},
{-0x1.00001fffff800p+37, -0x1.ffffff801ffc0p-22, -0x1.0000000002000p+12, 0x1.0000000000000p-93, -0x1.0000207fff800p+37, -0x1.03ffffc00ffe0p-21},
{0x1.0000000000000p-40, 0x1.ffff83ffbfff0p-96, 0x1.0000000000000p-34, 0x1.ffffffffe0000p-90, 0x1.0400000000000p-34, 0x1.03ffff07ef800p-89},
{0x1.0000000000000p-40, 0x1.ffff83ffbfff0p-96, -0x1.0000000000000p-34, -0x1.ffffffffe0000p-90, -0x1.f800000000000p-35, -0x1.f80001efe1000p-90},
{-0x1.0000000000000p-40, -0x1.ffff83ffbfff0p-96, 0x1.0000000000000p-34, 0x1.ffffffffe0000p-90, 0x1.f800000000000p-35, 0x1.f80001efe1000p-90},
{-0x1.0000000000000p-40, -0x1.ffff83ffbfff0p-96, -0x1.0000000000000p-34, -0x1.ffffffffe0000p-90, -0x1.0400000000000p-34, -0x1.03ffff07ef800p-89},
{0x1.0000000000000p+39, 0x1.fffe7ff3dff00p-22, 0x1.000007f830000p+13, 0x1.fff0000ffe100p-48, 0x1.0000004000020p+39, -0x1.e4000bfc61208p-17},
{0x1.0000000000000p+39, 0x1.fffe7ff3dff00p-22, -0x1.000007f830000p+13, -0x1.fff0000ffe100p-48, 0x1.ffffff7ffffc0p+38, 0x1.01fff9fdcf8fcp-16},
{-0x1.0000000000000p+39, -0x1.fffe7ff3dff00p-22, 0x1.000007f830000p+13, 0x1.fff0000ffe100p-48, -0x1.ffffff7ffffc0p+38, -0x1.01fff9fdcf8fcp-16},
{-0x1.0000000000000p+39, -0x1.fffe7ff3dff00p-22, -0x1.000007f830000p+13, -0x1.fff0000ffe100p-48, -0x1.0000004000020p+39, 0x1.e4000bfc61208p-17},
{0x1.00000ffffe000p+18, 0x1.fffffffffc000p-49, 0x1.0000000000002p-46, -0x1.fffffe01f8004p-101, 0x1.00000ffffe000p+18, 0x1.3fffffffff800p-46},
{0x1.00000ffffe000p+18, 0x1.fffffffffc000p-49, -0x1.0000000000002p-46, 0x1.fffffe01f8004p-101, 0x1.00000ffffe000p+18, -0x1.8000000001000p-47},
{-0x1.00000ffffe000p+18, -0x1.fffffffffc000p-49, 0x1.0000000000002p-46, -0x1.fffffe01f8004p-101, -0x1.00000ffffe000p+18, 0x1.8000000001000p-47},
{-0x1.00000ffffe000p+18, -0x1.fffffffffc000p-49, -0x1.0000000000002p-46, 0x1.fffffe01f8004p-101, -0x1.00000ffffe000p+18, -0x1.3fffffffff800p-46},
{0x1.0000000000060p+41, 0x1.fffe3fff00000p-22, 0x1.0000000002000p+4, -0x1.ffe0002000000p-53, 0x1.0000000008060p+41, 0x1.00031ffd80200p-21},
{0x1.0000000000060p+41, 0x1.fffe3fff00000p-22, -0x1.0000000002000p+4, 0x1.ffe0002000000p-53, 0x1.fffffffff00c0p+40, 0x1.fff64002ffc00p-22},
{-0x1.0000000000060p+41, -0x1.fffe3fff00000p-22, 0x1.0000000002000p+4, -0x1.ffe0002000000p-53, -0x1.fffffffff00c0p+40, -0x1.fff64002ffc00p-22},
{-0x1.0000000000060p+41, -0x1.fffe3fff00000p-22, -0x1.0000000002000p+4, 0x1.ffe0002000000p-53, -0x1.0000000008060p+41, -0x1.00031ffd80200p-21},
{0x1.0003ffffffff8p-50, 0x1.ffffe07e00000p-124, 0x1.07ff000008000p+19, -0x1.e000020000000p-42, 0x1.07ff000008000p+19, -0x1.defffe0000000p-42},
{0x1.0003ffffffff8p-50, 0x1.ffffe07e00000p-124, -0x1.07ff000008000p+19, 0x1.e000020000000p-42, -0x1.07ff000008000p+19, 0x1.e100060000000p-42},
{-0x1.0003ffffffff8p-50, -0x1.ffffe07e00000p-124, 0x1.07ff000008000p+19, -0x1.e000020000000p-42, 0x1.07ff000008000p+19, -0x1.e100060000000p-42},
{-0x1.0003ffffffff8p-50, -0x1.ffffe07e00000p-124, -0x1.07ff000008000p+19, 0x1.e000020000000p-42, -0x1.07ff000008000p+19, 0x1.defffe0000000p-42},
// Some test cases drawn only from the [0.5,1] binade:
{0x1.0000000000002p-1, -0x1.ffc0ffff90000p-65, 0x1.0003ffffffffcp-1, 0x1.ffffc00000000p-88, 0x1.0001fffffffffp+0, -0x1.ffc0fbff90000p-65},
{0x1.0000000000002p-1, -0x1.ffc0ffff90000p-65, -0x1.0003ffffffffcp-1, -0x1.ffffc00000000p-88, -0x1.ffffffffd0010p-16, 0x1.f7e0038400000p-76},
{-0x1.0000000000002p-1, 0x1.ffc0ffff90000p-65, 0x1.0003ffffffffcp-1, 0x1.ffffc00000000p-88, 0x1.ffffffffd0010p-16, -0x1.f7e0038400000p-76},
{-0x1.0000000000002p-1, 0x1.ffc0ffff90000p-65, -0x1.0003ffffffffcp-1, -0x1.ffffc00000000p-88, -0x1.0001fffffffffp+0, 0x1.ffc0fbff90000p-65},
{0x1.0000100000000p-1, -0x1.0000000000000p-63, 0x1.0000000004000p-1, -0x1.ffffe00fc0200p-57, 0x1.0000080002000p+0, -0x1.01fff007e0100p-56},
{0x1.0000100000000p-1, -0x1.0000000000000p-63, -0x1.0000000004000p-1, 0x1.ffffe00fc0200p-57, 0x1.ffff80003f800p-22, -0x1.ff03fe0000000p-77},
{-0x1.0000100000000p-1, 0x1.0000000000000p-63, 0x1.0000000004000p-1, -0x1.ffffe00fc0200p-57, -0x1.ffff80003f800p-22, 0x1.ff03fe0000000p-77},
{-0x1.0000100000000p-1, 0x1.0000000000000p-63, -0x1.0000000004000p-1, 0x1.ffffe00fc0200p-57, -0x1.0000080002000p+0, 0x1.01fff007e0100p-56},
{0x1.0000000800000p-1, -0x1.0000000000000p-73, 0x1.000003fffc002p-1, -0x1.0000000000000p-106, 0x1.00000203fe001p+0, -0x1.0000000000000p-73},
{0x1.0000000800000p-1, -0x1.0000000000000p-73, -0x1.000003fffc002p-1, 0x1.0000000000000p-106, -0x1.fbfe001000008p-24, 0x1.0000000000000p-106},
{-0x1.0000000800000p-1, 0x1.0000000000000p-73, 0x1.000003fffc002p-1, -0x1.0000000000000p-106, 0x1.fbfe001000008p-24, -0x1.0000000000000p-106},
{-0x1.0000000800000p-1, 0x1.0000000000000p-73, -0x1.000003fffc002p-1, 0x1.0000000000000p-106, -0x1.00000203fe001p+0, 0x1.0000000000000p-73},
{0x1.0000000000000p-1, 0x1.ffc001ffffe00p-56, 0x1.000003fff8000p-1, 0x1.fffc000fc0000p-72, 0x1.000001fffc000p+0, 0x1.ffc201fbfff00p-56},
{0x1.0000000000000p-1, 0x1.ffc001ffffe00p-56, -0x1.000003fff8000p-1, -0x1.fffc000fc0000p-72, -0x1.fffbfffe00420p-24, 0x1.01ffe82000000p-79},
{-0x1.0000000000000p-1, -0x1.ffc001ffffe00p-56, 0x1.000003fff8000p-1, 0x1.fffc000fc0000p-72, 0x1.fffbfffe00420p-24, -0x1.01ffe82000000p-79},
{-0x1.0000000000000p-1, -0x1.ffc001ffffe00p-56, -0x1.000003fff8000p-1, -0x1.fffc000fc0000p-72, -0x1.000001fffc000p+0, -0x1.ffc201fbfff00p-56},
{0x1.0000000000003p-1, 0x1.fc00001fff800p-65, 0x1.0000000000020p-1, -0x1.fc01ffe002000p-65, 0x1.0000000000011p+0, 0x1.ffffffc007ffcp-54},
{0x1.0000000000003p-1, 0x1.fc00001fff800p-65, -0x1.0000000000020p-1, 0x1.fc01ffe002000p-65, -0x1.cffc07fe00000p-49, 0x1.8000000000000p-105},
{-0x1.0000000000003p-1, -0x1.fc00001fff800p-65, 0x1.0000000000020p-1, -0x1.fc01ffe002000p-65, 0x1.cffc07fe00000p-49, -0x1.8000000000000p-105},
{-0x1.0000000000003p-1, -0x1.fc00001fff800p-65, -0x1.0000000000020p-1, 0x1.fc01ffe002000p-65, -0x1.0000000000011p+0, -0x1.ffffffc007ffcp-54},
{0x1.0000000000000p-1, 0x1.fffff00001ffep-55, 0x1.0000000000000p-1, 0x1.ff00000ffff80p-61, 0x1.0000000000000p+0, 0x1.03fdf80020ffep-54},
{0x1.0000000000000p-1, 0x1.fffff00001ffep-55, -0x1.0000000000000p-1, -0x1.ff00000ffff80p-61, 0x1.f803efffc2000p-55, 0x0.0000000000000p+0},
{-0x1.0000000000000p-1, -0x1.fffff00001ffep-55, 0x1.0000000000000p-1, 0x1.ff00000ffff80p-61, -0x1.f803efffc2000p-55, 0x0.0000000000000p+0},
{-0x1.0000000000000p-1, -0x1.fffff00001ffep-55, -0x1.0000000000000p-1, -0x1.ff00000ffff80p-61, -0x1.0000000000000p+0, -0x1.03fdf80020ffep-54},
{0x1.0000000000000p-1, 0x1.ffffffffffc00p-64, 0x1.0000200000000p-1, -0x1.ffc0000080000p-73, 0x1.0000100000000p+0, 0x1.ff001fffff800p-64},
{0x1.0000000000000p-1, 0x1.ffffffffffc00p-64, -0x1.0000200000000p-1, 0x1.ffc0000080000p-73, -0x1.ffffffffffbfep-21, -0x1.0000000000000p-83},
{-0x1.0000000000000p-1, -0x1.ffffffffffc00p-64, 0x1.0000200000000p-1, -0x1.ffc0000080000p-73, 0x1.ffffffffffbfep-21, 0x1.0000000000000p-83},
{-0x1.0000000000000p-1, -0x1.ffffffffffc00p-64, -0x1.0000200000000p-1, 0x1.ffc0000080000p-73, -0x1.0000100000000p+0, -0x1.ff001fffff800p-64},
{0x1.0000000000800p-1, -0x1.fffc000000010p-58, 0x1.0000000000008p-1, -0x1.ffe03fffff010p-58, 0x1.0000000000404p+0, -0x1.ffee1fffff810p-57},
{0x1.0000000000800p-1, -0x1.fffc000000010p-58, -0x1.0000000000008p-1, 0x1.ffe03fffff010p-58, 0x1.fdffffc880000p-43, -0x1.0000000000000p-98},
{-0x1.0000000000800p-1, 0x1.fffc000000010p-58, 0x1.0000000000008p-1, -0x1.ffe03fffff010p-58, -0x1.fdffffc880000p-43, 0x1.0000000000000p-98},
{-0x1.0000000000800p-1, 0x1.fffc000000010p-58, -0x1.0000000000008p-1, 0x1.ffe03fffff010p-58, -0x1.0000000000404p+0, 0x1.ffee1fffff810p-57},
{0x1.0000000000000p-1, 0x1.fffef0ffff000p-64, 0x1.0000100000000p-1, -0x1.fff8000000800p-65, 0x1.0000080000000p+0, 0x1.0002f0ffff000p-64},
{0x1.0000000000000p-1, 0x1.fffef0ffff000p-64, -0x1.0000100000000p-1, 0x1.fff8000000800p-65, -0x1.ffffffffff400p-22, -0x1.43c0003000000p-78},
{-0x1.0000000000000p-1, -0x1.fffef0ffff000p-64, 0x1.0000100000000p-1, -0x1.fff8000000800p-65, 0x1.ffffffffff400p-22, 0x1.43c0003000000p-78},
{-0x1.0000000000000p-1, -0x1.fffef0ffff000p-64, -0x1.0000100000000p-1, 0x1.fff8000000800p-65, -0x1.0000080000000p+0, -0x1.0002f0ffff000p-64},
{0x1.07c00007ffff0p-1, 0x1.fffc00f800000p-77, 0x1.0000007ffffe0p-1, 0x1.f000000000000p-72, 0x1.03e00043fffe8p+0, 0x1.ffffe00800000p-72},
{0x1.07c00007ffff0p-1, 0x1.fffc00f800000p-77, -0x1.0000007ffffe0p-1, -0x1.f000000000000p-72, 0x1.efffe20000400p-7, -0x1.e0001ff840000p-72},
{-0x1.07c00007ffff0p-1, -0x1.fffc00f800000p-77, 0x1.0000007ffffe0p-1, 0x1.f000000000000p-72, -0x1.efffe20000400p-7, 0x1.e0001ff840000p-72},
{-0x1.07c00007ffff0p-1, -0x1.fffc00f800000p-77, -0x1.0000007ffffe0p-1, -0x1.f000000000000p-72, -0x1.03e00043fffe8p+0, -0x1.ffffe00800000p-72},
{0x1.0000000000000p-1, 0x1.ff800fffff800p-57, 0x1.0000000000000p-1, 0x1.fffc0000ffff8p-57, 0x1.0000000000000p+0, 0x1.ffbe08007fc00p-56},
{0x1.0000000000000p-1, 0x1.ff800fffff800p-57, -0x1.0000000000000p-1, -0x1.fffc0000ffff8p-57, -0x1.efc00401fe000p-67, 0x0.0000000000000p+0},
{-0x1.0000000000000p-1, -0x1.ff800fffff800p-57, 0x1.0000000000000p-1, 0x1.fffc0000ffff8p-57, 0x1.efc00401fe000p-67, 0x0.0000000000000p+0},
{-0x1.0000000000000p-1, -0x1.ff800fffff800p-57, -0x1.0000000000000p-1, -0x1.fffc0000ffff8p-57, -0x1.0000000000000p+0, -0x1.ffbe08007fc00p-56},
{0x1.0000000008000p-1, -0x1.ffff807ffe002p-55, 0x1.0000000800000p-1, -0x1.ffffff8000008p-57, 0x1.0000000404000p+0, -0x1.3fffc02fff002p-54},
{0x1.0000000008000p-1, -0x1.ffff807ffe002p-55, -0x1.0000000800000p-1, 0x1.ffffff8000008p-57, -0x1.fe00017fff80ap-31, 0x1.0000000000000p-94},
{-0x1.0000000008000p-1, 0x1.ffff807ffe002p-55, 0x1.0000000800000p-1, -0x1.ffffff8000008p-57, 0x1.fe00017fff80ap-31, -0x1.0000000000000p-94},
{-0x1.0000000008000p-1, 0x1.ffff807ffe002p-55, -0x1.0000000800000p-1, 0x1.ffffff8000008p-57, -0x1.0000000404000p+0, 0x1.3fffc02fff002p-54},
{0x1.0000000000020p-1, -0x1.0000000000000p-84, 0x1.0000000000000p-1, 0x1.fffff000003c0p-60, 0x1.0000000000010p+0, 0x1.ffffef0000400p-60},
{0x1.0000000000020p-1, -0x1.0000000000000p-84, -0x1.0000000000000p-1, -0x1.fffff000003c0p-60, 0x1.ffc00001e0000p-49, -0x1.e000000000000p-103},
{-0x1.0000000000020p-1, 0x1.0000000000000p-84, 0x1.0000000000000p-1, 0x1.fffff000003c0p-60, -0x1.ffc00001e0000p-49, 0x1.e000000000000p-103},
{-0x1.0000000000020p-1, 0x1.0000000000000p-84, -0x1.0000000000000p-1, -0x1.fffff000003c0p-60, -0x1.0000000000010p+0, -0x1.ffffef0000400p-60},
{0x1.0000001ffff80p-1, 0x1.fffffdfffff8ep-55, 0x1.0000000200000p-1, -0x1.0000000000000p-70, 0x1.00000010fffc0p+0, 0x1.fffdfdfffff90p-55},
{0x1.0000001ffff80p-1, 0x1.fffffdfffff8ep-55, -0x1.0000000200000p-1, 0x1.0000000000000p-70, 0x1.dfff8080007f8p-29, -0x1.c800000000000p-101},
{-0x1.0000001ffff80p-1, -0x1.fffffdfffff8ep-55, 0x1.0000000200000p-1, -0x1.0000000000000p-70, -0x1.dfff8080007f8p-29, 0x1.c800000000000p-101},
{-0x1.0000001ffff80p-1, -0x1.fffffdfffff8ep-55, -0x1.0000000200000p-1, 0x1.0000000000000p-70, -0x1.00000010fffc0p+0, -0x1.fffdfdfffff90p-55},
{0x1.01ffffff80400p-1, -0x1.c7ffffc000100p-59, 0x1.0000000000000p-1, 0x1.8fff01fffff00p-61, 0x1.00ffffffc0200p+0, -0x1.64003f4000140p-59},
{0x1.01ffffff80400p-1, -0x1.c7ffffc000100p-59, -0x1.0000000000000p-1, -0x1.8fff01fffff00p-61, 0x1.ffffff803fff7p-9, 0x1.4003fbffff400p-63},
{-0x1.01ffffff80400p-1, 0x1.c7ffffc000100p-59, 0x1.0000000000000p-1, 0x1.8fff01fffff00p-61, -0x1.ffffff803fff7p-9, -0x1.4003fbffff400p-63},
{-0x1.01ffffff80400p-1, 0x1.c7ffffc000100p-59, -0x1.0000000000000p-1, -0x1.8fff01fffff00p-61, -0x1.00ffffffc0200p+0, 0x1.64003f4000140p-59},
{0x1.0000000000008p-1, -0x1.ffffc00800000p-74, 0x1.00003ffe00002p-1, -0x1.ffffffc180080p-61, 0x1.00001fff00005p+0, -0x1.0007ffdfc0200p-60},
{0x1.0000000000008p-1, -0x1.ffffc00800000p-74, -0x1.00003ffe00002p-1, 0x1.ffffffc180080p-61, -0x1.ffeffffcff001p-20, 0x1.fff86ff900000p-74},
{-0x1.0000000000008p-1, 0x1.ffffc00800000p-74, 0x1.00003ffe00002p-1, -0x1.ffffffc180080p-61, 0x1.ffeffffcff001p-20, -0x1.fff86ff900000p-74},
{-0x1.0000000000008p-1, 0x1.ffffc00800000p-74, -0x1.00003ffe00002p-1, 0x1.ffffffc180080p-61, -0x1.00001fff00005p+0, 0x1.0007ffdfc0200p-60},
{0x1.000000ffffff8p-1, 0x1.ffffffc000000p-58, 0x1.0000000000000p-1, 0x1.ffffffe000000p-59, 0x1.0000007fffffcp+0, 0x1.7fffffd800000p-57},
{0x1.000000ffffff8p-1, 0x1.ffffffc000000p-58, -0x1.0000000000000p-1, -0x1.ffffffe000000p-59, 0x1.ffffff0100000p-26, -0x1.8000000000000p-85},
{-0x1.000000ffffff8p-1, -0x1.ffffffc000000p-58, 0x1.0000000000000p-1, 0x1.ffffffe000000p-59, -0x1.ffffff0100000p-26, 0x1.8000000000000p-85},
{-0x1.000000ffffff8p-1, -0x1.ffffffc000000p-58, -0x1.0000000000000p-1, -0x1.ffffffe000000p-59, -0x1.0000007fffffcp+0, -0x1.7fffffd800000p-57},
{0x1.0000000020000p-1, -0x1.0000000000000p-106, 0x1.0000000000400p-1, -0x1.ff1ffffff4000p-68, 0x1.0000000010200p+0, -0x1.ff1ffffff8000p-68},
{0x1.0000000020000p-1, -0x1.0000000000000p-106, -0x1.0000000000400p-1, 0x1.ff1ffffff4000p-68, 0x1.fc000003fe400p-37, -0x1.0000000000000p-104},
{-0x1.0000000020000p-1, 0x1.0000000000000p-106, 0x1.0000000000400p-1, -0x1.ff1ffffff4000p-68, -0x1.fc000003fe400p-37, 0x1.0000000000000p-104},
{-0x1.0000000020000p-1, 0x1.0000000000000p-106, -0x1.0000000000400p-1, 0x1.ff1ffffff4000p-68, -0x1.0000000010200p+0, 0x1.ff1ffffff8000p-68},
{0x1.0000ff8020000p-1, -0x1.fffffa0000000p-63, 0x1.0000000001ff0p-1, 0x1.f700000ffe000p-67, 0x1.00007fc010ff8p+0, -0x1.e08ff9ff00000p-63},
{0x1.0000ff8020000p-1, -0x1.fffffa0000000p-63, -0x1.0000000001ff0p-1, -0x1.f700000ffe000p-67, 0x1.ff003c01ffef0p-18, 0x1.200bfe0040000p-72},
{-0x1.0000ff8020000p-1, 0x1.fffffa0000000p-63, 0x1.0000000001ff0p-1, 0x1.f700000ffe000p-67, -0x1.ff003c01ffef0p-18, -0x1.200bfe0040000p-72},
{-0x1.0000ff8020000p-1, 0x1.fffffa0000000p-63, -0x1.0000000001ff0p-1, -0x1.f700000ffe000p-67, -0x1.00007fc010ff8p+0, 0x1.e08ff9ff00000p-63},
{0x1.0000020000000p-1, -0x1.0000000000000p-65, 0x1.003e00001f800p-1, 0x1.ffde000001e00p-63, 0x1.001f01000fc00p+0, 0x1.bfde000002000p-63},
{0x1.0000020000000p-1, -0x1.0000000000000p-65, -0x1.003e00001f800p-1, -0x1.ffde000001e00p-63, -0x1.eff000fc00004p-12, -0x1.fef000000f000p-66},
{-0x1.0000020000000p-1, 0x1.0000000000000p-65, 0x1.003e00001f800p-1, 0x1.ffde000001e00p-63, 0x1.eff000fc00004p-12, 0x1.fef000000f000p-66},
{-0x1.0000020000000p-1, 0x1.0000000000000p-65, -0x1.003e00001f800p-1, -0x1.ffde000001e00p-63, -0x1.001f01000fc00p+0, -0x1.bfde000002000p-63},
{0x1.0007e3ffe0000p-1, 0x1.fe0ffffffe000p-65, 0x1.0000000000000p-1, 0x1.fffffe07fff00p-62, 0x1.0003f1fff0000p+0, 0x1.1fe0ff03ffe00p-61},
{0x1.0007e3ffe0000p-1, 0x1.fe0ffffffe000p-65, -0x1.0000000000000p-1, -0x1.fffffe07fff00p-62, 0x1.f8fff7fffffc8p-15, -0x1.eff0400180000p-73},
{-0x1.0007e3ffe0000p-1, -0x1.fe0ffffffe000p-65, 0x1.0000000000000p-1, 0x1.fffffe07fff00p-62, -0x1.f8fff7fffffc8p-15, 0x1.eff0400180000p-73},
{-0x1.0007e3ffe0000p-1, -0x1.fe0ffffffe000p-65, -0x1.0000000000000p-1, -0x1.fffffe07fff00p-62, -0x1.0003f1fff0000p+0, -0x1.1fe0ff03ffe00p-61},
{0x1.00000000fe000p-1, 0x1.ff00004000000p-68, 0x1.000003fffff80p-1, 0x1.fff8400000000p-61, 0x1.000002007efc0p+0, 0x1.01fb200040000p-60},
{0x1.00000000fe000p-1, 0x1.ff00004000000p-68, -0x1.000003fffff80p-1, -0x1.fff8400000000p-61, -0x1.ff80fc000fdfdp-24, -0x1.fffc000000000p-80},
{-0x1.00000000fe000p-1, -0x1.ff00004000000p-68, 0x1.000003fffff80p-1, 0x1.fff8400000000p-61, 0x1.ff80fc000fdfdp-24, 0x1.fffc000000000p-80},
{-0x1.00000000fe000p-1, -0x1.ff00004000000p-68, -0x1.000003fffff80p-1, -0x1.fff8400000000p-61, -0x1.000002007efc0p+0, -0x1.01fb200040000p-60},
{0x1.0000800000000p-1, -0x1.0000000000000p-57, 0x1.0000000000000p-1, 0x1.fffffffc1fc00p-64, 0x1.0000400000000p+0, -0x1.f80000000f800p-58},
{0x1.0000800000000p-1, -0x1.0000000000000p-57, -0x1.0000000000000p-1, -0x1.fffffffc1fc00p-64, 0x1.fffffffffbf00p-19, 0x1.f020000000000p-95},
{-0x1.0000800000000p-1, 0x1.0000000000000p-57, 0x1.0000000000000p-1, 0x1.fffffffc1fc00p-64, -0x1.fffffffffbf00p-19, -0x1.f020000000000p-95},
{-0x1.0000800000000p-1, 0x1.0000000000000p-57, -0x1.0000000000000p-1, -0x1.fffffffc1fc00p-64, -0x1.0000400000000p+0, 0x1.f80000000f800p-58},
{0x1.0000000000000p-1, 0x1.ff000ff3ffe00p-55, 0x1.0000000000000p-1, 0x1.fe0003fffc000p-60, 0x1.0000000000000p+0, 0x1.07780809ffe00p-54},
{0x1.0000000000000p-1, 0x1.ff000ff3ffe00p-55, -0x1.0000000000000p-1, -0x1.fe0003fffc000p-60, 0x1.ef100fd400000p-55, 0x0.0000000000000p+0},
{-0x1.0000000000000p-1, -0x1.ff000ff3ffe00p-55, 0x1.0000000000000p-1, 0x1.fe0003fffc000p-60, -0x1.ef100fd400000p-55, 0x0.0000000000000p+0},
{-0x1.0000000000000p-1, -0x1.ff000ff3ffe00p-55, -0x1.0000000000000p-1, -0x1.fe0003fffc000p-60, -0x1.0000000000000p+0, -0x1.07780809ffe00p-54},
{0x1.000000e000010p-1, -0x1.ffffffc020000p-71, 0x1.00000001f0000p-1, 0x1.8000000000000p-105, 0x1.00000070f8008p+0, -0x1.ffffffbfc0000p-71},
{0x1.000000e000010p-1, -0x1.ffffffc020000p-71, -0x1.00000001f0000p-1, -0x1.8000000000000p-105, 0x1.bc2001fffff00p-26, 0x1.fc00000000000p-98},
{-0x1.000000e000010p-1, 0x1.ffffffc020000p-71, 0x1.00000001f0000p-1, 0x1.8000000000000p-105, -0x1.bc2001fffff00p-26, -0x1.fc00000000000p-98},
{-0x1.000000e000010p-1, 0x1.ffffffc020000p-71, -0x1.00000001f0000p-1, -0x1.8000000000000p-105, -0x1.00000070f8008p+0, 0x1.ffffffbfc0000p-71},
{0x1.000003ff00000p-1, 0x1.8000000000000p-78, 0x1.0001ffffe0000p-1, 0x1.f800000100000p-74, 0x1.000101ff70000p+0, 0x1.0800000000000p-73},
{0x1.000003ff00000p-1, 0x1.8000000000000p-78, -0x1.0001ffffe0000p-1, -0x1.f800000100000p-74, -0x1.fc00e00000000p-17, -0x1.e000000100000p-74},
{-0x1.000003ff00000p-1, -0x1.8000000000000p-78, 0x1.0001ffffe0000p-1, 0x1.f800000100000p-74, 0x1.fc00e00000000p-17, 0x1.e000000100000p-74},
{-0x1.000003ff00000p-1, -0x1.8000000000000p-78, -0x1.0001ffffe0000p-1, -0x1.f800000100000p-74, -0x1.000101ff70000p+0, -0x1.0800000000000p-73},
{0x1.0000000000200p-1, -0x1.ffffffffffc08p-57, 0x1.0000000000000p-1, 0x1.ffffffffffffep-55, 0x1.0000000000100p+0, 0x1.80000000000fcp-55},
{0x1.0000000000200p-1, -0x1.ffffffffffc08p-57, -0x1.0000000000000p-1, -0x1.ffffffffffffep-55, 0x1.ff60000000000p-45, 0x1.0000000000000p-99},
{-0x1.0000000000200p-1, 0x1.ffffffffffc08p-57, 0x1.0000000000000p-1, 0x1.ffffffffffffep-55, -0x1.ff60000000000p-45, -0x1.0000000000000p-99},
{-0x1.0000000000200p-1, 0x1.ffffffffffc08p-57, -0x1.0000000000000p-1, -0x1.ffffffffffffep-55, -0x1.0000000000100p+0, -0x1.80000000000fcp-55},
{0x1.0000000040000p-1, -0x1.fffff80002000p-67, 0x1.0000000000000p-1, 0x1.ffc01ffff3c00p-64, 0x1.0000000020000p+0, 0x1.bfc020fff3800p-64},
{0x1.0000000040000p-1, -0x1.fffff80002000p-67, -0x1.0000000000000p-1, -0x1.ffc01ffff3c00p-64, 0x1.ffffffdc03fe1p-36, 0x1.8000000000000p-101},
{-0x1.0000000040000p-1, 0x1.fffff80002000p-67, 0x1.0000000000000p-1, 0x1.ffc01ffff3c00p-64, -0x1.ffffffdc03fe1p-36, -0x1.8000000000000p-101},
{-0x1.0000000040000p-1, 0x1.fffff80002000p-67, -0x1.0000000000000p-1, -0x1.ffc01ffff3c00p-64, -0x1.0000000020000p+0, -0x1.bfc020fff3800p-64},
{0x1.0000800000000p-1, -0x1.0000000000000p-106, 0x1.0000003800001p-1, 0x1.fffffe000003ep-55, 0x1.0000401c00001p+0, -0x1.000000fffffe2p-54},
{0x1.0000800000000p-1, -0x1.0000000000000p-106, -0x1.0000003800001p-1, -0x1.fffffe000003ep-55, 0x1.ff1fffffa0000p-19, 0x1.fffffc0000000p-79},
{-0x1.0000800000000p-1, 0x1.0000000000000p-106, 0x1.0000003800001p-1, 0x1.fffffe000003ep-55, -0x1.ff1fffffa0000p-19, -0x1.fffffc0000000p-79},
{-0x1.0000800000000p-1, 0x1.0000000000000p-106, -0x1.0000003800001p-1, -0x1.fffffe000003ep-55, -0x1.0000401c00001p+0, 0x1.000000fffffe2p-54},
{0x1.0f00000020000p-1, -0x1.fffe000000100p-55, 0x1.0000000400000p-1, -0x1.ffffffc004000p-68, 0x1.0780000210000p+0, -0x1.0006ffffff080p-54},
{0x1.0f00000020000p-1, -0x1.fffe000000100p-55, -0x1.0000000400000p-1, 0x1.ffffffc004000p-68, 0x1.dfffff83ffff0p-6, 0x1.1fffffdf02000p-67},
{-0x1.0f00000020000p-1, 0x1.fffe000000100p-55, 0x1.0000000400000p-1, -0x1.ffffffc004000p-68, -0x1.dfffff83ffff0p-6, -0x1.1fffffdf02000p-67},
{-0x1.0f00000020000p-1, 0x1.fffe000000100p-55, -0x1.0000000400000p-1, 0x1.ffffffc004000p-68, -0x1.0780000210000p+0, 0x1.0006ffffff080p-54},
{0x1.003fe00020000p-1, -0x1.ffffffffe8000p-69, 0x1.0000000000400p-1, -0x1.ffff800007ffcp-56, 0x1.001ff00010200p+0, -0x1.0007c00003ff8p-55},
{0x1.003fe00020000p-1, -0x1.ffffffffe8000p-69, -0x1.0000000000400p-1, 0x1.ffff800007ffcp-56, 0x1.ff0000fe00200p-12, -0x1.07ffff7ff8000p-68},
{-0x1.003fe00020000p-1, 0x1.ffffffffe8000p-69, 0x1.0000000000400p-1, -0x1.ffff800007ffcp-56, -0x1.ff0000fe00200p-12, 0x1.07ffff7ff8000p-68},
{-0x1.003fe00020000p-1, 0x1.ffffffffe8000p-69, -0x1.0000000000400p-1, 0x1.ffff800007ffcp-56, -0x1.001ff00010200p+0, 0x1.0007c00003ff8p-55},
{0x1.ffff00fffffc4p-1, -0x1.8080000000000p-83, 0x1.0000020000000p-1, -0x1.0000000000000p-77, 0x1.7fff817ffffe2p+0, -0x1.0602000000000p-77},
{0x1.ffff00fffffc4p-1, -0x1.8080000000000p-83, -0x1.0000020000000p-1, 0x1.0000000000000p-77, 0x1.fffdfdfffff88p-2, 0x1.f3fc000000000p-78},
{-0x1.ffff00fffffc4p-1, 0x1.8080000000000p-83, 0x1.0000020000000p-1, -0x1.0000000000000p-77, -0x1.fffdfdfffff88p-2, -0x1.f3fc000000000p-78},
{-0x1.ffff00fffffc4p-1, 0x1.8080000000000p-83, -0x1.0000020000000p-1, 0x1.0000000000000p-77, -0x1.7fff817ffffe2p+0, 0x1.0602000000000p-77},
{0x1.0000000000040p-1, -0x1.8000000800000p-76, 0x1.0000000002000p-1, -0x1.0000000000000p-105, 0x1.0000000001020p+0, -0x1.8000001000000p-76},
{0x1.0000000000040p-1, -0x1.8000000800000p-76, -0x1.0000000002000p-1, 0x1.0000000000000p-105, -0x1.fc00000030000p-41, 0x0.0000000000000p+0},
{-0x1.0000000000040p-1, 0x1.8000000800000p-76, 0x1.0000000002000p-1, -0x1.0000000000000p-105, 0x1.fc00000030000p-41, 0x0.0000000000000p+0},
{-0x1.0000000000040p-1, 0x1.8000000800000p-76, -0x1.0000000002000p-1, 0x1.0000000000000p-105, -0x1.0000000001020p+0, 0x1.8000001000000p-76},
{0x1.0000000800000p-1, -0x1.ffffc02000000p-79, 0x1.7fffe00800000p-1, -0x1.0000000000000p-106, 0x1.3ffff00800000p+0, -0x1.ffffc04000000p-79},
{0x1.0000000800000p-1, -0x1.ffffc02000000p-79, -0x1.7fffe00800000p-1, 0x1.0000000000000p-106, -0x1.ffff800000000p-3, -0x1.ffffc00000000p-79},
{-0x1.0000000800000p-1, 0x1.ffffc02000000p-79, 0x1.7fffe00800000p-1, -0x1.0000000000000p-106, 0x1.ffff800000000p-3, 0x1.ffffc00000000p-79},
{-0x1.0000000800000p-1, 0x1.ffffc02000000p-79, -0x1.7fffe00800000p-1, 0x1.0000000000000p-106, -0x1.3ffff00800000p+0, 0x1.ffffc04000000p-79},
{0x1.0000400000000p-1, -0x1.0000000000000p-92, 0x1.4000000000000p-1, -0x1.ffffffff00020p-59, 0x1.2000200000000p+0, -0x1.ffffffff80000p-59},
{0x1.0000400000000p-1, -0x1.0000000000000p-92, -0x1.4000000000000p-1, 0x1.ffffffff00020p-59, -0x1.fffe000000000p-4, 0x1.fffffffe80020p-59},
{-0x1.0000400000000p-1, 0x1.0000000000000p-92, 0x1.4000000000000p-1, -0x1.ffffffff00020p-59, 0x1.fffe000000000p-4, -0x1.fffffffe80020p-59},
{-0x1.0000400000000p-1, 0x1.0000000000000p-92, -0x1.4000000000000p-1, 0x1.ffffffff00020p-59, -0x1.2000200000000p+0, 0x1.ffffffff80000p-59},
{0x1.0000000000000p-1, 0x1.03eff7ff3fff0p-58, 0x1.00001fffc0000p-1, 0x1.f80001f000f80p-61, 0x1.00000fffe0000p+0, 0x1.42eff83d401e0p-58},
{0x1.0000000000000p-1, 0x1.03eff7ff3fff0p-58, -0x1.00001fffc0000p-1, -0x1.f80001f000f80p-61, -0x1.fffbfffff9d88p-21, -0x1.07d8040000000p-79},
{-0x1.0000000000000p-1, -0x1.03eff7ff3fff0p-58, 0x1.00001fffc0000p-1, 0x1.f80001f000f80p-61, 0x1.fffbfffff9d88p-21, 0x1.07d8040000000p-79},
{-0x1.0000000000000p-1, -0x1.03eff7ff3fff0p-58, -0x1.00001fffc0000p-1, -0x1.f80001f000f80p-61, -0x1.00000fffe0000p+0, -0x1.42eff83d401e0p-58},
{0x1.03ffffffffffep-1, 0x1.8000000000000p-105, 0x1.0000040000000p-1, -0x1.0000000000000p-106, 0x1.020001fffffffp+0, 0x1.0000000000000p-105},
{0x1.03ffffffffffep-1, 0x1.8000000000000p-105, -0x1.0000040000000p-1, 0x1.0000000000000p-106, 0x1.fffdfffffff00p-8, 0x1.0000000000000p-104},
{-0x1.03ffffffffffep-1, -0x1.8000000000000p-105, 0x1.0000040000000p-1, -0x1.0000000000000p-106, -0x1.fffdfffffff00p-8, -0x1.0000000000000p-104},
{-0x1.03ffffffffffep-1, -0x1.8000000000000p-105, -0x1.0000040000000p-1, 0x1.0000000000000p-106, -0x1.020001fffffffp+0, -0x1.0000000000000p-105},
{0x1.0007ffffcfffcp-1, 0x1.ff01fffe00000p-75, 0x1.01fff08000000p-1, -0x1.fffffe0000020p-55, 0x1.0103f83fe7ffep+0, -0x1.ffffde0fe0020p-55},
{0x1.0007ffffcfffcp-1, 0x1.ff01fffe00000p-75, -0x1.01fff08000000p-1, 0x1.fffffe0000020p-55, -0x1.f7f0803000380p-9, 0x1.df02001e00000p-75},
{-0x1.0007ffffcfffcp-1, -0x1.ff01fffe00000p-75, 0x1.01fff08000000p-1, -0x1.fffffe0000020p-55, 0x1.f7f0803000380p-9, -0x1.df02001e00000p-75},
{-0x1.0007ffffcfffcp-1, -0x1.ff01fffe00000p-75, -0x1.01fff08000000p-1, 0x1.fffffe0000020p-55, -0x1.0103f83fe7ffep+0, 0x1.ffffde0fe0020p-55},
{0x1.00000000007fep-1, 0x1.ff00003ffff00p-62, 0x1.004000e000008p-1, -0x1.ffffe02000000p-79, 0x1.0020007000403p+0, 0x1.feff00400fe00p-62},
{0x1.00000000007fep-1, 0x1.ff00003ffff00p-62, -0x1.004000e000008p-1, 0x1.ffffe02000000p-79, -0x1.00037ffe027fcp-11, -0x1.fdff802000000p-71},
{-0x1.00000000007fep-1, -0x1.ff00003ffff00p-62, 0x1.004000e000008p-1, -0x1.ffffe02000000p-79, 0x1.00037ffe027fcp-11, 0x1.fdff802000000p-71},
{-0x1.00000000007fep-1, -0x1.ff00003ffff00p-62, -0x1.004000e000008p-1, 0x1.ffffe02000000p-79, -0x1.0020007000403p+0, -0x1.feff00400fe00p-62},
{0x1.00000003ff000p-1, 0x1.ffff000000000p-90, 0x1.0000fffff8000p-1, 0x1.f003ffe0003e0p-59, 0x1.00008001fb800p+0, 0x1.f003ffe4003c0p-59},
{0x1.00000003ff000p-1, 0x1.ffff000000000p-90, -0x1.0000fffff8000p-1, -0x1.f003ffe0003e0p-59, -0x1.fff7f20000f80p-18, -0x1.ffee002000000p-74},
{-0x1.00000003ff000p-1, -0x1.ffff000000000p-90, 0x1.0000fffff8000p-1, 0x1.f003ffe0003e0p-59, 0x1.fff7f20000f80p-18, 0x1.ffee002000000p-74},
{-0x1.00000003ff000p-1, -0x1.ffff000000000p-90, -0x1.0000fffff8000p-1, -0x1.f003ffe0003e0p-59, -0x1.00008001fb800p+0, -0x1.f003ffe4003c0p-59},
{0x1.0000000000000p-1, 0x1.ffe001c0003c0p-60, 0x1.01fffffe00000p-1, 0x1.ffffffffffc00p-59, 0x1.00ffffff00000p+0, 0x1.7ff8006ffff00p-58},
{0x1.0000000000000p-1, 0x1.ffe001c0003c0p-60, -0x1.01fffffe00000p-1, -0x1.ffffffffffc00p-59, -0x1.fffffe0000004p-9, -0x1.ffe3fff440000p-72},
{-0x1.0000000000000p-1, -0x1.ffe001c0003c0p-60, 0x1.01fffffe00000p-1, 0x1.ffffffffffc00p-59, 0x1.fffffe0000004p-9, 0x1.ffe3fff440000p-72},
{-0x1.0000000000000p-1, -0x1.ffe001c0003c0p-60, -0x1.01fffffe00000p-1, -0x1.ffffffffffc00p-59, -0x1.00ffffff00000p+0, -0x1.7ff8006ffff00p-58},
{0x1.00003ff808000p-1, -0x1.ffe0000800000p-62, 0x1.000001ffff000p-1, 0x1.ff80000000000p-75, 0x1.000020fc03800p+0, -0x1.ffd0040800000p-62},
{0x1.00003ff808000p-1, -0x1.ffe0000800000p-62, -0x1.000001ffff000p-1, -0x1.ff80000000000p-75, 0x1.efc047ffff800p-20, 0x1.003f800000000p-74},
{-0x1.00003ff808000p-1, 0x1.ffe0000800000p-62, 0x1.000001ffff000p-1, 0x1.ff80000000000p-75, -0x1.efc047ffff800p-20, -0x1.003f800000000p-74},
{-0x1.00003ff808000p-1, 0x1.ffe0000800000p-62, -0x1.000001ffff000p-1, -0x1.ff80000000000p-75, -0x1.000020fc03800p+0, 0x1.ffd0040800000p-62},
{0x1.0000000000000p-1, 0x1.fc007f0ff0000p-63, 0x1.0000000007f80p-1, 0x1.ff0007f802000p-65, 0x1.0000000003fc0p+0, 0x1.3de04086f8400p-62},
{0x1.0000000000000p-1, 0x1.fc007f0ff0000p-63, -0x1.0000000007f80p-1, -0x1.ff0007f802000p-65, -0x1.fdfffe83bf82fp-39, 0x1.ef80000000000p-95},
{-0x1.0000000000000p-1, -0x1.fc007f0ff0000p-63, 0x1.0000000007f80p-1, 0x1.ff0007f802000p-65, 0x1.fdfffe83bf82fp-39, -0x1.ef80000000000p-95},
{-0x1.0000000000000p-1, -0x1.fc007f0ff0000p-63, -0x1.0000000007f80p-1, -0x1.ff0007f802000p-65, -0x1.0000000003fc0p+0, -0x1.3de04086f8400p-62},
{0x1.0020000000000p-1, -0x1.0000000000000p-78, 0x1.000001ffff802p-1, -0x1.ffffffc000040p-60, 0x1.001000ffffc01p+0, -0x1.00001fe000000p-59},
{0x1.0020000000000p-1, -0x1.0000000000000p-78, -0x1.000001ffff802p-1, 0x1.ffffffc000040p-60, 0x1.ffe00007fe040p-13, -0x1.00fffff000000p-78},
{-0x1.0020000000000p-1, 0x1.0000000000000p-78, 0x1.000001ffff802p-1, -0x1.ffffffc000040p-60, -0x1.ffe00007fe040p-13, 0x1.00fffff000000p-78},
{-0x1.0020000000000p-1, 0x1.0000000000000p-78, -0x1.000001ffff802p-1, 0x1.ffffffc000040p-60, -0x1.001000ffffc01p+0, 0x1.00001fe000000p-59},
{0x1.0000000000800p-1, -0x1.ffe0000400000p-63, 0x1.0000000000038p-1, 0x1.e100000000000p-75, 0x1.000000000041cp+0, -0x1.ffc1f00400000p-63},
{0x1.0000000000800p-1, -0x1.ffe0000400000p-63, -0x1.0000000000038p-1, -0x1.e100000000000p-75, 0x1.f1ffe0001effcp-43, 0x0.0000000000000p+0},
{-0x1.0000000000800p-1, 0x1.ffe0000400000p-63, 0x1.0000000000038p-1, 0x1.e100000000000p-75, -0x1.f1ffe0001effcp-43, 0x0.0000000000000p+0},
{-0x1.0000000000800p-1, 0x1.ffe0000400000p-63, -0x1.0000000000038p-1, -0x1.e100000000000p-75, -0x1.000000000041cp+0, 0x1.ffc1f00400000p-63},
{0x1.0000000000020p-1, -0x1.8007ffff80c04p-56, 0x1.0000000000000p-1, 0x1.ffffe00000c00p-64, 0x1.0000000000010p+0, -0x1.7e08001f80bf8p-56},
{0x1.0000000000020p-1, -0x1.8007ffff80c04p-56, -0x1.0000000000000p-1, -0x1.ffffe00000c00p-64, 0x1.fcfbf00040fe8p-49, -0x1.0000000000000p-104},
{-0x1.0000000000020p-1, 0x1.8007ffff80c04p-56, 0x1.0000000000000p-1, 0x1.ffffe00000c00p-64, -0x1.fcfbf00040fe8p-49, 0x1.0000000000000p-104},
{-0x1.0000000000020p-1, 0x1.8007ffff80c04p-56, -0x1.0000000000000p-1, -0x1.ffffe00000c00p-64, -0x1.0000000000010p+0, 0x1.7e08001f80bf8p-56},
{0x1.7fffffff80000p-1, 0x1.f000000000000p-78, 0x1.000007ffffffep-1, 0x1.ffffff8000000p-77, 0x1.400003ffbffffp+0, 0x1.7bffffc000000p-76},
{0x1.7fffffff80000p-1, 0x1.f000000000000p-78, -0x1.000007ffffffep-1, -0x1.ffffff8000000p-77, 0x1.ffffdffe00008p-3, -0x1.07ffff8000000p-77},
{-0x1.7fffffff80000p-1, -0x1.f000000000000p-78, 0x1.000007ffffffep-1, 0x1.ffffff8000000p-77, -0x1.ffffdffe00008p-3, 0x1.07ffff8000000p-77},
{-0x1.7fffffff80000p-1, -0x1.f000000000000p-78, -0x1.000007ffffffep-1, -0x1.ffffff8000000p-77, -0x1.400003ffbffffp+0, -0x1.7bffffc000000p-76},
{0x1.0000000010000p-1, -0x1.0000000000000p-87, 0x1.0000000000000p-1, 0x1.c000003ffc3fep-55, 0x1.0000000008000p+0, 0x1.c000003efc400p-55},
{0x1.0000000010000p-1, -0x1.0000000000000p-87, -0x1.0000000000000p-1, -0x1.c000003ffc3fep-55, 0x1.ffff1fffffdf8p-38, 0x1.e010000000000p-94},
{-0x1.0000000010000p-1, 0x1.0000000000000p-87, 0x1.0000000000000p-1, 0x1.c000003ffc3fep-55, -0x1.ffff1fffffdf8p-38, -0x1.e010000000000p-94},
{-0x1.0000000010000p-1, 0x1.0000000000000p-87, -0x1.0000000000000p-1, -0x1.c000003ffc3fep-55, -0x1.0000000008000p+0, -0x1.c000003efc400p-55},
{0x1.00000ffff0800p-1, -0x1.fff80000ffc20p-59, 0x1.0000000000002p-1, -0x1.0000000000000p-106, 0x1.000007fff8401p+0, -0x1.fff80000ffc40p-59},
{0x1.00000ffff0800p-1, -0x1.fff80000ffc20p-59, -0x1.0000000000002p-1, 0x1.0000000000000p-106, 0x1.fffe0ffbf0004p-22, -0x1.ff80000000000p-92},
{-0x1.00000ffff0800p-1, 0x1.fff80000ffc20p-59, 0x1.0000000000002p-1, -0x1.0000000000000p-106, -0x1.fffe0ffbf0004p-22, 0x1.ff80000000000p-92},
{-0x1.00000ffff0800p-1, 0x1.fff80000ffc20p-59, -0x1.0000000000002p-1, 0x1.0000000000000p-106, -0x1.000007fff8401p+0, 0x1.fff80000ffc40p-59},
{0x1.000ffffc00000p-1, 0x1.fffff00000000p-70, 0x1.0000000000040p-1, -0x1.0000000000000p-106, 0x1.0007fffe00020p+0, 0x1.fffff00000000p-70},
{0x1.000ffffc00000p-1, 0x1.fffff00000000p-70, -0x1.0000000000040p-1, 0x1.0000000000000p-106, 0x1.ffff7fff80000p-14, 0x1.fffff00010000p-70},
{-0x1.000ffffc00000p-1, -0x1.fffff00000000p-70, 0x1.0000000000040p-1, -0x1.0000000000000p-106, -0x1.ffff7fff80000p-14, -0x1.fffff00010000p-70},
{-0x1.000ffffc00000p-1, -0x1.fffff00000000p-70, -0x1.0000000000040p-1, 0x1.0000000000000p-106, -0x1.0007fffe00020p+0, -0x1.fffff00000000p-70},
{0x1.0004000000000p-1, -0x1.0000000000000p-69, 0x1.0000000200000p-1, -0x1.ffbffffff8020p-59, 0x1.0002000100000p+0, -0x1.fffffffff8000p-59},
{0x1.0004000000000p-1, -0x1.0000000000000p-69, -0x1.0000000200000p-1, 0x1.ffbffffff8020p-59, 0x1.ffff0000003ffp-16, -0x1.ff80000000000p-97},
{-0x1.0004000000000p-1, 0x1.0000000000000p-69, 0x1.0000000200000p-1, -0x1.ffbffffff8020p-59, -0x1.ffff0000003ffp-16, 0x1.ff80000000000p-97},
{-0x1.0004000000000p-1, 0x1.0000000000000p-69, -0x1.0000000200000p-1, 0x1.ffbffffff8020p-59, -0x1.0002000100000p+0, 0x1.fffffffff8000p-59},
{0x1.0000000000000p-1, 0x1.e0001e7fbff00p-62, 0x1.0000001fffffep-1, 0x0.0000000000000p+0, 0x1.0000000ffffffp+0, 0x1.e0001e7fc0000p-62},
{0x1.0000000000000p-1, 0x1.e0001e7fbff00p-62, -0x1.0000001fffffep-1, 0x0.0000000000000p+0, -0x1.fffffdff0ffffp-29, -0x1.8040100000000p-86},
{-0x1.0000000000000p-1, -0x1.e0001e7fbff00p-62, 0x1.0000001fffffep-1, 0x0.0000000000000p+0, 0x1.fffffdff0ffffp-29, 0x1.8040100000000p-86},
{-0x1.0000000000000p-1, -0x1.e0001e7fbff00p-62, -0x1.0000001fffffep-1, 0x0.0000000000000p+0, -0x1.0000000ffffffp+0, -0x1.e0001e7fc0000p-62},
{0x1.0000000000fffp-1, 0x1.fffff3e000000p-74, 0x1.001ffe0000000p-1, 0x1.fffffffffffc0p-60, 0x1.000fff0000800p+0, -0x1.f7ffe00000c20p-54},
{0x1.0000000000fffp-1, 0x1.fffff3e000000p-74, -0x1.001ffe0000000p-1, -0x1.fffffffffffc0p-60, -0x1.ffdffff001040p-13, 0x1.fffff3e100000p-74},
{-0x1.0000000000fffp-1, -0x1.fffff3e000000p-74, 0x1.001ffe0000000p-1, 0x1.fffffffffffc0p-60, 0x1.ffdffff001040p-13, -0x1.fffff3e100000p-74},
{-0x1.0000000000fffp-1, -0x1.fffff3e000000p-74, -0x1.001ffe0000000p-1, -0x1.fffffffffffc0p-60, -0x1.000fff0000800p+0, 0x1.f7ffe00000c20p-54},
{0x1.0000080000000p-1, -0x1.0000000000000p-106, 0x1.0000000080000p-1, -0x1.0000000000000p-56, 0x1.0000040040000p+0, -0x1.0000000000000p-56},
{0x1.0000080000000p-1, -0x1.0000000000000p-106, -0x1.0000000080000p-1, 0x1.0000000000000p-56, 0x1.ffe0000080000p-23, -0x1.0000000000000p-106},
{-0x1.0000080000000p-1, 0x1.0000000000000p-106, 0x1.0000000080000p-1, -0x1.0000000000000p-56, -0x1.ffe0000080000p-23, 0x1.0000000000000p-106},
{-0x1.0000080000000p-1, 0x1.0000000000000p-106, -0x1.0000000080000p-1, 0x1.0000000000000p-56, -0x1.0000040040000p+0, 0x1.0000000000000p-56},
{0x1.003ffffffffe0p-1, 0x1.ffffe03800000p-77, 0x1.0000000002000p-1, -0x1.0000000000000p-54, 0x1.0020000000ff0p+0, -0x1.fffff800007f0p-55},
{0x1.003ffffffffe0p-1, 0x1.ffffe03800000p-77, -0x1.0000000002000p-1, 0x1.0000000000000p-54, 0x1.ffffffeff0400p-12, 0x1.ffffe03800000p-77},
{-0x1.003ffffffffe0p-1, -0x1.ffffe03800000p-77, 0x1.0000000002000p-1, -0x1.0000000000000p-54, -0x1.ffffffeff0400p-12, -0x1.ffffe03800000p-77},
{-0x1.003ffffffffe0p-1, -0x1.ffffe03800000p-77, -0x1.0000000002000p-1, 0x1.0000000000000p-54, -0x1.0020000000ff0p+0, 0x1.fffff800007f0p-55},
{0x1.0000000000008p-1, -0x1.ff000007ffc00p-61, 0x1.0000000060000p-1, 0x1.ffffff8000000p-81, 0x1.0000000030004p+0, -0x1.feffe007ffc00p-61},
{0x1.0000000000008p-1, -0x1.ff000007ffc00p-61, -0x1.0000000060000p-1, -0x1.ffffff8000000p-81, -0x1.7ffe007fc0080p-35, -0x1.ffee000000000p-91},
{-0x1.0000000000008p-1, 0x1.ff000007ffc00p-61, 0x1.0000000060000p-1, 0x1.ffffff8000000p-81, 0x1.7ffe007fc0080p-35, 0x1.ffee000000000p-91},
{-0x1.0000000000008p-1, 0x1.ff000007ffc00p-61, -0x1.0000000060000p-1, -0x1.ffffff8000000p-81, -0x1.0000000030004p+0, 0x1.feffe007ffc00p-61},
{0x1.0000000000080p-1, -0x1.0000000000000p-106, 0x1.01ffffff80001p-1, -0x1.ff90000000000p-94, 0x1.00ffffffc0040p+0, 0x1.fffffffffe006p-54},
{0x1.0000000000080p-1, -0x1.0000000000000p-106, -0x1.01ffffff80001p-1, 0x1.ff90000000000p-94, -0x1.ffffff7ff8100p-9, 0x1.ff80000000000p-94},
{-0x1.0000000000080p-1, 0x1.0000000000000p-106, 0x1.01ffffff80001p-1, -0x1.ff90000000000p-94, 0x1.ffffff7ff8100p-9, -0x1.ff80000000000p-94},
{-0x1.0000000000080p-1, 0x1.0000000000000p-106, -0x1.01ffffff80001p-1, 0x1.ff90000000000p-94, -0x1.00ffffffc0040p+0, -0x1.fffffffffe006p-54},
{0x1.0000000000800p-1, -0x1.fffff80002000p-67, 0x1.0008000000000p-1, -0x1.0000000000000p-80, 0x1.0004000000400p+0, -0x1.0003fc0000000p-66},
{0x1.0000000000800p-1, -0x1.fffff80002000p-67, -0x1.0008000000000p-1, 0x1.0000000000000p-80, -0x1.ffffffe000002p-15, 0x1.00ffffc000000p-80},
{-0x1.0000000000800p-1, 0x1.fffff80002000p-67, 0x1.0008000000000p-1, -0x1.0000000000000p-80, 0x1.ffffffe000002p-15, -0x1.00ffffc000000p-80},
{-0x1.0000000000800p-1, 0x1.fffff80002000p-67, -0x1.0008000000000p-1, 0x1.0000000000000p-80, -0x1.0004000000400p+0, 0x1.0003fc0000000p-66},
{0x1.0000000000002p-1, -0x1.ffffffe200000p-75, 0x1.0003fffffffc0p-1, 0x1.ffffe00000400p-64, 0x1.0001fffffffe1p+0, 0x1.ffbfe00004000p-64},
{0x1.0000000000002p-1, -0x1.ffffffe200000p-75, -0x1.0003fffffffc0p-1, -0x1.ffffe00000400p-64, -0x1.fffffffdf0020p-16, -0x1.feffffe400000p-75},
{-0x1.0000000000002p-1, 0x1.ffffffe200000p-75, 0x1.0003fffffffc0p-1, 0x1.ffffe00000400p-64, 0x1.fffffffdf0020p-16, 0x1.feffffe400000p-75},
{-0x1.0000000000002p-1, 0x1.ffffffe200000p-75, -0x1.0003fffffffc0p-1, -0x1.ffffe00000400p-64, -0x1.0001fffffffe1p+0, -0x1.ffbfe00004000p-64},
{0x1.000003f000000p-1, 0x1.fffffffe00000p-62, 0x1.0000000000ff0p-1, 0x1.c03fff0e00000p-75, 0x1.000001f8007f8p+0, 0x1.000700fefc400p-61},
{0x1.000003f000000p-1, 0x1.fffffffe00000p-62, -0x1.0000000000ff0p-1, -0x1.c03fff0e00000p-75, 0x1.f7ff808007ffcp-24, 0x1.fdfe079000000p-78},
{-0x1.000003f000000p-1, -0x1.fffffffe00000p-62, 0x1.0000000000ff0p-1, 0x1.c03fff0e00000p-75, -0x1.f7ff808007ffcp-24, -0x1.fdfe079000000p-78},
{-0x1.000003f000000p-1, -0x1.fffffffe00000p-62, -0x1.0000000000ff0p-1, -0x1.c03fff0e00000p-75, -0x1.000001f8007f8p+0, -0x1.000700fefc400p-61},
{0x1.0000000018000p-1, 0x1.fffffe0000000p-83, 0x1.0000100000000p-1, -0x1.0000000000000p-106, 0x1.000008000c000p+0, 0x1.fffffc0000000p-83},
{0x1.0000000018000p-1, 0x1.fffffe0000000p-83, -0x1.0000100000000p-1, 0x1.0000000000000p-106, -0x1.fffd000000000p-22, 0x1.0000000000000p-82},
{-0x1.0000000018000p-1, -0x1.fffffe0000000p-83, 0x1.0000100000000p-1, -0x1.0000000000000p-106, 0x1.fffd000000000p-22, -0x1.0000000000000p-82},
{-0x1.0000000018000p-1, -0x1.fffffe0000000p-83, -0x1.0000100000000p-1, 0x1.0000000000000p-106, -0x1.000008000c000p+0, -0x1.fffffc0000000p-83},
{0x1.0000000000001p-1, -0x1.fffffc1000000p-68, 0x1.1ffffff01f000p-1, 0x1.ffffc000060fcp-56, 0x1.0ffffff80f801p+0, -0x1.80080fffeebc0p-54},
{0x1.0000000000001p-1, -0x1.fffffc1000000p-68, -0x1.1ffffff01f000p-1, -0x1.ffffc000060fcp-56, -0x1.ffffff01efff4p-5, -0x1.fbfffc70fc000p-68},
{-0x1.0000000000001p-1, 0x1.fffffc1000000p-68, 0x1.1ffffff01f000p-1, 0x1.ffffc000060fcp-56, 0x1.ffffff01efff4p-5, 0x1.fbfffc70fc000p-68},
{-0x1.0000000000001p-1, 0x1.fffffc1000000p-68, -0x1.1ffffff01f000p-1, -0x1.ffffc000060fcp-56, -0x1.0ffffff80f801p+0, 0x1.80080fffeebc0p-54},
{0x1.0000003800000p-1, 0x1.ffe000001fffcp-56, 0x1.0000007fffff0p-1, 0x1.fe1ffffc00000p-63, 0x1.0000005bffff8p+0, 0x1.01ee20000c000p-55},
{0x1.0000003800000p-1, 0x1.ffe000001fffcp-56, -0x1.0000007fffff0p-1, -0x1.fe1ffffc00000p-63, -0x1.1ffffbf020e20p-27, 0x1.3ffe000000000p-91},
{-0x1.0000003800000p-1, -0x1.ffe000001fffcp-56, 0x1.0000007fffff0p-1, 0x1.fe1ffffc00000p-63, 0x1.1ffffbf020e20p-27, -0x1.3ffe000000000p-91},
{-0x1.0000003800000p-1, -0x1.ffe000001fffcp-56, -0x1.0000007fffff0p-1, -0x1.fe1ffffc00000p-63, -0x1.0000005bffff8p+0, -0x1.01ee20000c000p-55},
{0x1.0000001fffc00p-1, 0x1.f00007ffe0000p-65, 0x1.0000000000180p-1, 0x1.fffffe0000800p-65, 0x1.0000000fffec0p+0, 0x1.f80002fff0000p-64},
{0x1.0000001fffc00p-1, 0x1.f00007ffe0000p-65, -0x1.0000000000180p-1, -0x1.fffffe0000800p-65, 0x1.fffa7fffff000p-29, 0x1.3ffbf00000000p-86},
{-0x1.0000001fffc00p-1, -0x1.f00007ffe0000p-65, 0x1.0000000000180p-1, 0x1.fffffe0000800p-65, -0x1.fffa7fffff000p-29, -0x1.3ffbf00000000p-86},
{-0x1.0000001fffc00p-1, -0x1.f00007ffe0000p-65, -0x1.0000000000180p-1, -0x1.fffffe0000800p-65, -0x1.0000000fffec0p+0, -0x1.f80002fff0000p-64},
{0x1.0000000008000p-1, -0x1.c0000007ffe04p-56, 0x1.00000fffffffep-1, 0x1.fc00000000000p-100, 0x1.0000080003fffp+0, -0x1.c0000007ffc08p-56},
{0x1.0000000008000p-1, -0x1.c0000007ffe04p-56, -0x1.00000fffffffep-1, -0x1.fc00000000000p-100, -0x1.fffefffc70000p-22, -0x1.0000000000000p-85},
{-0x1.0000000008000p-1, 0x1.c0000007ffe04p-56, 0x1.00000fffffffep-1, 0x1.fc00000000000p-100, 0x1.fffefffc70000p-22, 0x1.0000000000000p-85},
{-0x1.0000000008000p-1, 0x1.c0000007ffe04p-56, -0x1.00000fffffffep-1, -0x1.fc00000000000p-100, -0x1.0000080003fffp+0, 0x1.c0000007ffc08p-56},
{0x1.0000003e00004p-1, -0x1.fe00040000000p-71, 0x1.00007c0001e00p-1, 0x1.fffe13ffe0000p-65, 0x1.00003e1f00f02p+0, 0x1.f80613efe0000p-65},
{0x1.0000003e00004p-1, -0x1.fe00040000000p-71, -0x1.00007c0001e00p-1, -0x1.fffe13ffe0000p-65, -0x1.ef08077f00082p-19, 0x1.3d7e040000000p-78},
{-0x1.0000003e00004p-1, 0x1.fe00040000000p-71, 0x1.00007c0001e00p-1, 0x1.fffe13ffe0000p-65, 0x1.ef08077f00082p-19, -0x1.3d7e040000000p-78},
{-0x1.0000003e00004p-1, 0x1.fe00040000000p-71, -0x1.00007c0001e00p-1, -0x1.fffe13ffe0000p-65, -0x1.00003e1f00f02p+0, -0x1.f80613efe0000p-65},
{0x1.03fffffffc000p-1, 0x1.ffffffc000000p-76, 0x1.0003ffffff000p-1, 0x1.ffc0000000000p-96, 0x1.0201fffffd800p+0, 0x1.00000fe000000p-75},
{0x1.03fffffffc000p-1, 0x1.ffffffc000000p-76, -0x1.0003ffffff000p-1, -0x1.ffc0000000000p-96, 0x1.fdfffffe80000p-8, 0x1.ffffdfc400000p-76},
{-0x1.03fffffffc000p-1, -0x1.ffffffc000000p-76, 0x1.0003ffffff000p-1, 0x1.ffc0000000000p-96, -0x1.fdfffffe80000p-8, -0x1.ffffdfc400000p-76},
{-0x1.03fffffffc000p-1, -0x1.ffffffc000000p-76, -0x1.0003ffffff000p-1, -0x1.ffc0000000000p-96, -0x1.0201fffffd800p+0, -0x1.00000fe000000p-75},
{0x1.07ffffff80000p-1, 0x1.8007ffffc0000p-72, 0x1.0000000000004p-1, -0x1.fffc000200000p-70, 0x1.03ffffffc0002p+0, -0x1.9ffa000200000p-70},
{0x1.07ffffff80000p-1, 0x1.8007ffffc0000p-72, -0x1.0000000000004p-1, 0x1.fffc000200000p-70, 0x1.ffffffdffff00p-7, 0x1.2fff0000f8000p-69},
{-0x1.07ffffff80000p-1, -0x1.8007ffffc0000p-72, 0x1.0000000000004p-1, -0x1.fffc000200000p-70, -0x1.ffffffdffff00p-7, -0x1.2fff0000f8000p-69},
{-0x1.07ffffff80000p-1, -0x1.8007ffffc0000p-72, -0x1.0000000000004p-1, 0x1.fffc000200000p-70, -0x1.03ffffffc0002p+0, 0x1.9ffa000200000p-70},
{0x1.00003ffffff80p-1, 0x1.ffff0000ffe00p-63, 0x1.00003fffffff8p-1, 0x0.0000000000000p+0, 0x1.00003ffffffbcp+0, 0x1.ffff000100000p-63},
{0x1.00003ffffff80p-1, 0x1.ffff0000ffe00p-63, -0x1.00003fffffff8p-1, 0x0.0000000000000p+0, -0x1.dffe0000ffff0p-47, -0x1.0000000000000p-106},
{-0x1.00003ffffff80p-1, -0x1.ffff0000ffe00p-63, 0x1.00003fffffff8p-1, 0x0.0000000000000p+0, 0x1.dffe0000ffff0p-47, 0x1.0000000000000p-106},
{-0x1.00003ffffff80p-1, -0x1.ffff0000ffe00p-63, -0x1.00003fffffff8p-1, 0x0.0000000000000p+0, -0x1.00003ffffffbcp+0, -0x1.ffff000100000p-63},
{0x1.0000010000000p-1, -0x1.0000000000000p-106, 0x1.00f81ffffff81p-1, 0x1.fffff03c00000p-63, 0x1.007c107ffffc1p+0, -0x1.ff000007e2000p-54},
{0x1.0000010000000p-1, -0x1.0000000000000p-106, -0x1.00f81ffffff81p-1, -0x1.fffff03c00000p-63, -0x1.f03dfffff0201p-10, 0x1.f87ffc0000000p-84},
{-0x1.0000010000000p-1, 0x1.0000000000000p-106, 0x1.00f81ffffff81p-1, 0x1.fffff03c00000p-63, 0x1.f03dfffff0201p-10, -0x1.f87ffc0000000p-84},
{-0x1.0000010000000p-1, 0x1.0000000000000p-106, -0x1.00f81ffffff81p-1, -0x1.fffff03c00000p-63, -0x1.007c107ffffc1p+0, 0x1.ff000007e2000p-54},
{0x1.0000000000004p-1, -0x1.fff0008000000p-65, 0x1.0000000100000p-1, -0x1.fffffe0004000p-68, 0x1.0000000080002p+0, -0x1.1ff8002000000p-64},
{0x1.0000000000004p-1, -0x1.fff0008000000p-65, -0x1.0000000100000p-1, 0x1.fffffe0004000p-68, -0x1.ffff80037fe00p-34, -0x1.7fff000000000p-90},
{-0x1.0000000000004p-1, 0x1.fff0008000000p-65, 0x1.0000000100000p-1, -0x1.fffffe0004000p-68, 0x1.ffff80037fe00p-34, 0x1.7fff000000000p-90},
{-0x1.0000000000004p-1, 0x1.fff0008000000p-65, -0x1.0000000100000p-1, 0x1.fffffe0004000p-68, -0x1.0000000080002p+0, 0x1.1ff8002000000p-64},
{0x1.0000000000800p-1, -0x1.0000000000000p-79, 0x1.0000010000000p-1, -0x1.0000000000000p-104, 0x1.0000008000400p+0, -0x1.0000008000000p-79},
{0x1.0000000000800p-1, -0x1.0000000000000p-79, -0x1.0000010000000p-1, 0x1.0000000000000p-104, -0x1.ffff000000000p-26, -0x1.ffffff0000000p-80},
{-0x1.0000000000800p-1, 0x1.0000000000000p-79, 0x1.0000010000000p-1, -0x1.0000000000000p-104, 0x1.ffff000000000p-26, 0x1.ffffff0000000p-80},
{-0x1.0000000000800p-1, 0x1.0000000000000p-79, -0x1.0000010000000p-1, 0x1.0000000000000p-104, -0x1.0000008000400p+0, 0x1.0000008000000p-79},
{0x1.00000007fff00p-1, 0x1.fffc1ffff87f0p-57, 0x1.0000007c00000p-1, 0x1.fff0000000000p-72, 0x1.00000041fff80p+0, 0x1.00000feffc3f8p-56},
{0x1.00000007fff00p-1, 0x1.fffc1ffff87f0p-57, -0x1.0000007c00000p-1, -0x1.fff0000000000p-72, -0x1.d0003ff8001f8p-27, 0x1.ff87f00000000p-85},
{-0x1.00000007fff00p-1, -0x1.fffc1ffff87f0p-57, 0x1.0000007c00000p-1, 0x1.fff0000000000p-72, 0x1.d0003ff8001f8p-27, -0x1.ff87f00000000p-85},
{-0x1.00000007fff00p-1, -0x1.fffc1ffff87f0p-57, -0x1.0000007c00000p-1, -0x1.fff0000000000p-72, -0x1.00000041fff80p+0, -0x1.00000feffc3f8p-56},
{0x1.fffffffe00000p-1, 0x1.e000001fff800p-65, 0x1.00000001ffffep-1, 0x0.0000000000000p+0, 0x1.7ffffffffffffp+0, 0x1.e000002000000p-65},
{0x1.fffffffe00000p-1, 0x1.e000001fff800p-65, -0x1.00000001ffffep-1, 0x0.0000000000000p+0, 0x1.fffffff800004p-2, 0x1.e000001fff800p-65},
{-0x1.fffffffe00000p-1, -0x1.e000001fff800p-65, 0x1.00000001ffffep-1, 0x0.0000000000000p+0, -0x1.fffffff800004p-2, -0x1.e000001fff800p-65},
{-0x1.fffffffe00000p-1, -0x1.e000001fff800p-65, -0x1.00000001ffffep-1, 0x0.0000000000000p+0, -0x1.7ffffffffffffp+0, -0x1.e000002000000p-65},
{0x1.0000000000008p-1, -0x1.fffff00400000p-73, 0x1.00003ffffe000p-1, 0x1.ffffe00000000p-73, 0x1.00001fffff004p+0, -0x1.0040000000000p-93},
{0x1.0000000000008p-1, -0x1.fffff00400000p-73, -0x1.00003ffffe000p-1, -0x1.ffffe00000000p-73, -0x1.ffffeffc00002p-20, 0x1.7fe0000000000p-92},
{-0x1.0000000000008p-1, 0x1.fffff00400000p-73, 0x1.00003ffffe000p-1, 0x1.ffffe00000000p-73, 0x1.ffffeffc00002p-20, -0x1.7fe0000000000p-92},
{-0x1.0000000000008p-1, 0x1.fffff00400000p-73, -0x1.00003ffffe000p-1, -0x1.ffffe00000000p-73, -0x1.00001fffff004p+0, 0x1.0040000000000p-93},
{0x1.0000000000000p-1, 0x1.ffffffffffc00p-64, 0x1.000001ffe0000p-1, 0x1.ff87ff9f801c0p-60, 0x1.000000fff0000p+0, 0x1.0fc3ffcfc00c0p-59},
{0x1.0000000000000p-1, 0x1.ffffffffffc00p-64, -0x1.000001ffe0000p-1, -0x1.ff87ff9f801c0p-60, -0x1.ffe000003bf10p-25, 0x1.81ff800000000p-86},
{-0x1.0000000000000p-1, -0x1.ffffffffffc00p-64, 0x1.000001ffe0000p-1, 0x1.ff87ff9f801c0p-60, 0x1.ffe000003bf10p-25, -0x1.81ff800000000p-86},
{-0x1.0000000000000p-1, -0x1.ffffffffffc00p-64, -0x1.000001ffe0000p-1, -0x1.ff87ff9f801c0p-60, -0x1.000000fff0000p+0, -0x1.0fc3ffcfc00c0p-59},
{0x1.0000000000400p-1, -0x1.f800000002000p-62, 0x1.03fffffffff00p-1, 0x1.fffe7fffffffep-55, 0x1.0200000000180p+0, 0x1.fc0e7ffffffc0p-55},
{0x1.0000000000400p-1, -0x1.f800000002000p-62, -0x1.03fffffffff00p-1, -0x1.fffe7fffffffep-55, -0x1.ffffffffd8040p-8, -0x1.f740000001f00p-62},
{-0x1.0000000000400p-1, 0x1.f800000002000p-62, 0x1.03fffffffff00p-1, 0x1.fffe7fffffffep-55, 0x1.ffffffffd8040p-8, 0x1.f740000001f00p-62},
{-0x1.0000000000400p-1, 0x1.f800000002000p-62, -0x1.03fffffffff00p-1, -0x1.fffe7fffffffep-55, -0x1.0200000000180p+0, -0x1.fc0e7ffffffc0p-55},
{0x1.0000000000000p-1, 0x1.fffffffffff00p-57, 0x1.000007c000000p-1, 0x1.fff8000000000p-57, 0x1.000003e000000p+0, 0x1.fffbfffffff80p-56},
{0x1.0000000000000p-1, 0x1.fffffffffff00p-57, -0x1.000007c000000p-1, -0x1.fff8000000000p-57, -0x1.effffffffffe0p-23, -0x1.0000000000000p-101},
{-0x1.0000000000000p-1, -0x1.fffffffffff00p-57, 0x1.000007c000000p-1, 0x1.fff8000000000p-57, 0x1.effffffffffe0p-23, 0x1.0000000000000p-101},
{-0x1.0000000000000p-1, -0x1.fffffffffff00p-57, -0x1.000007c000000p-1, -0x1.fff8000000000p-57, -0x1.000003e000000p+0, -0x1.fffbfffffff80p-56},
{0x1.0000000000400p-1, -0x1.fffc0003fffd0p-58, 0x1.0000000000000p-1, 0x1.fffffff800000p-57, 0x1.0000000000200p+0, 0x1.0001fff600020p-57},
{0x1.0000000000400p-1, -0x1.fffc0003fffd0p-58, -0x1.0000000000000p-1, -0x1.fffffff800000p-57, 0x1.ffe8001000300p-44, 0x1.8000000000000p-105},
{-0x1.0000000000400p-1, 0x1.fffc0003fffd0p-58, 0x1.0000000000000p-1, 0x1.fffffff800000p-57, -0x1.ffe8001000300p-44, -0x1.8000000000000p-105},
{-0x1.0000000000400p-1, 0x1.fffc0003fffd0p-58, -0x1.0000000000000p-1, -0x1.fffffff800000p-57, -0x1.0000000000200p+0, -0x1.0001fff600020p-57},
{0x1.0000000002000p-1, -0x1.ffff040000000p-72, 0x1.03fffff800000p-1, 0x1.fffffe0000000p-69, 0x1.01fffffc01000p+0, 0x1.c0001d8000000p-69},
{0x1.0000000002000p-1, -0x1.ffff040000000p-72, -0x1.03fffff800000p-1, -0x1.fffffe0000000p-69, -0x1.fffffbff00000p-8, -0x1.1fffef4000000p-68},
{-0x1.0000000002000p-1, 0x1.ffff040000000p-72, 0x1.03fffff800000p-1, 0x1.fffffe0000000p-69, 0x1.fffffbff00000p-8, 0x1.1fffef4000000p-68},
{-0x1.0000000002000p-1, 0x1.ffff040000000p-72, -0x1.03fffff800000p-1, -0x1.fffffe0000000p-69, -0x1.01fffffc01000p+0, -0x1.c0001d8000000p-69},
{0x1.0000008000000p-1, -0x1.ffffff8200000p-58, 0x1.000fff8000000p-1, 0x1.fffffe0000000p-63, 0x1.0008000000000p+0, -0x1.efffff9200000p-58},
{0x1.0000008000000p-1, -0x1.ffffff8200000p-58, -0x1.000fff8000000p-1, -0x1.fffffe0000000p-63, -0x1.ffe0000000210p-14, 0x1.1c00000000000p-83},
{-0x1.0000008000000p-1, 0x1.ffffff8200000p-58, 0x1.000fff8000000p-1, 0x1.fffffe0000000p-63, 0x1.ffe0000000210p-14, -0x1.1c00000000000p-83},
{-0x1.0000008000000p-1, 0x1.ffffff8200000p-58, -0x1.000fff8000000p-1, -0x1.fffffe0000000p-63, -0x1.0008000000000p+0, 0x1.efffff9200000p-58},
{0x1.0000000000000p-1, 0x1.0001f9e000000p-61, 0x1.0000000000040p-1, -0x1.fffffc0080000p-73, 0x1.0000000000020p+0, 0x1.ffc3f3c080000p-62},
{0x1.0000000000000p-1, 0x1.0001f9e000000p-61, -0x1.0000000000040p-1, 0x1.fffffc0080000p-73, -0x1.fff7fef031020p-48, 0x1.0000000000000p-106},
{-0x1.0000000000000p-1, -0x1.0001f9e000000p-61, 0x1.0000000000040p-1, -0x1.fffffc0080000p-73, 0x1.fff7fef031020p-48, -0x1.0000000000000p-106},
{-0x1.0000000000000p-1, -0x1.0001f9e000000p-61, -0x1.0000000000040p-1, 0x1.fffffc0080000p-73, -0x1.0000000000020p+0, -0x1.ffc3f3c080000p-62},
{0x1.0000ff0007e00p-1, 0x1.ffc001ff00000p-68, 0x1.0000010000000p-1, -0x1.ffc007ffffe20p-59, 0x1.0000800003f00p+0, -0x1.fec027ff00600p-59},
{0x1.0000ff0007e00p-1, 0x1.ffc001ff00000p-68, -0x1.0000010000000p-1, 0x1.ffc007ffffe20p-59, 0x1.fc000fc001006p-18, -0x1.7ff009e000000p-79},
{-0x1.0000ff0007e00p-1, -0x1.ffc001ff00000p-68, 0x1.0000010000000p-1, -0x1.ffc007ffffe20p-59, -0x1.fc000fc001006p-18, 0x1.7ff009e000000p-79},
{-0x1.0000ff0007e00p-1, -0x1.ffc001ff00000p-68, -0x1.0000010000000p-1, 0x1.ffc007ffffe20p-59, -0x1.0000800003f00p+0, 0x1.fec027ff00600p-59},
{0x1.07ffffff1fe00p-1, 0x1.fc00000000000p-98, 0x1.0040000000000p-1, -0x1.c000000400000p-62, 0x1.041fffff8ff00p+0, -0x1.c0000003e0400p-62},
{0x1.07ffffff1fe00p-1, 0x1.fc00000000000p-98, -0x1.0040000000000p-1, 0x1.c000000400000p-62, 0x1.efffffc7f8000p-7, 0x1.c00000041fc00p-62},
{-0x1.07ffffff1fe00p-1, -0x1.fc00000000000p-98, 0x1.0040000000000p-1, -0x1.c000000400000p-62, -0x1.efffffc7f8000p-7, -0x1.c00000041fc00p-62},
{-0x1.07ffffff1fe00p-1, -0x1.fc00000000000p-98, -0x1.0040000000000p-1, 0x1.c000000400000p-62, -0x1.041fffff8ff00p+0, 0x1.c0000003e0400p-62},
{0x1.0080000000000p-1, -0x1.0000000000000p-106, 0x1.00000001fffc0p-1, 0x1.fffff80f80000p-73, 0x1.00400000fffe0p+0, 0x1.fffff80f00000p-73},
{0x1.0080000000000p-1, -0x1.0000000000000p-106, -0x1.00000001fffc0p-1, -0x1.fffff80f80000p-73, 0x1.fffff80010000p-11, -0x1.fffff81000000p-73},
{-0x1.0080000000000p-1, 0x1.0000000000000p-106, 0x1.00000001fffc0p-1, 0x1.fffff80f80000p-73, -0x1.fffff80010000p-11, 0x1.fffff81000000p-73},
{-0x1.0080000000000p-1, 0x1.0000000000000p-106, -0x1.00000001fffc0p-1, -0x1.fffff80f80000p-73, -0x1.00400000fffe0p+0, -0x1.fffff80f00000p-73},
{0x1.000000003ff00p-1, 0x0.0000000000000p+0, 0x1.0000000000000p-1, 0x1.f83007ffffe00p-61, 0x1.000000001ff80p+0, 0x1.f83007ffffe00p-61},
{0x1.000000003ff00p-1, 0x0.0000000000000p+0, -0x1.0000000000000p-1, -0x1.f83007ffffe00p-61, 0x1.ff7fff03e7fc0p-36, 0x1.0000000000000p-104},
{-0x1.000000003ff00p-1, 0x0.0000000000000p+0, 0x1.0000000000000p-1, 0x1.f83007ffffe00p-61, -0x1.ff7fff03e7fc0p-36, -0x1.0000000000000p-104},
{-0x1.000000003ff00p-1, 0x0.0000000000000p+0, -0x1.0000000000000p-1, -0x1.f83007ffffe00p-61, -0x1.000000001ff80p+0, -0x1.f83007ffffe00p-61},
{0x1.00003fffffe00p-1, 0x1.fffffc0000380p-61, 0x1.0000010000000p-1, -0x1.ffffff8000008p-57, 0x1.0000207ffff00p+0, -0x1.dfffffbffffd0p-57},
{0x1.00003fffffe00p-1, 0x1.fffffc0000380p-61, -0x1.0000010000000p-1, 0x1.ffffff8000008p-57, 0x1.f7ffff0011000p-20, -0x1.7ffff80000000p-82},
{-0x1.00003fffffe00p-1, -0x1.fffffc0000380p-61, 0x1.0000010000000p-1, -0x1.ffffff8000008p-57, -0x1.f7ffff0011000p-20, 0x1.7ffff80000000p-82},
{-0x1.00003fffffe00p-1, -0x1.fffffc0000380p-61, -0x1.0000010000000p-1, 0x1.ffffff8000008p-57, -0x1.0000207ffff00p+0, 0x1.dfffffbffffd0p-57},
{0x1.0000000080000p-1, -0x1.fffffc0010000p-70, 0x1.3fffff8000000p-1, 0x1.fffff800fe000p-67, 0x1.1fffffc040000p+0, 0x1.bffff880fc000p-67},
{0x1.0000000080000p-1, -0x1.fffffc0010000p-70, -0x1.3fffff8000000p-1, -0x1.fffff800fe000p-67, -0x1.fffffbfc00000p-4, -0x1.1ffffbc080000p-66},
{-0x1.0000000080000p-1, 0x1.fffffc0010000p-70, 0x1.3fffff8000000p-1, 0x1.fffff800fe000p-67, 0x1.fffffbfc00000p-4, 0x1.1ffffbc080000p-66},
{-0x1.0000000080000p-1, 0x1.fffffc0010000p-70, -0x1.3fffff8000000p-1, -0x1.fffff800fe000p-67, -0x1.1fffffc040000p+0, -0x1.bffff880fc000p-67},
{0x1.0003f001fc3f0p-1, 0x1.fffffc0000000p-57, 0x1.0000000000000p-1, 0x1.fc1fffc0000fep-55, 0x1.0001f800fe1f8p+0, 0x1.3e0fff6000080p-54},
{0x1.0003f001fc3f0p-1, 0x1.fffffc0000000p-57, -0x1.0000000000000p-1, -0x1.fc1fffc0000fep-55, 0x1.f800fe1f7d07cp-16, -0x1.80001fc000000p-80},
{-0x1.0003f001fc3f0p-1, -0x1.fffffc0000000p-57, 0x1.0000000000000p-1, 0x1.fc1fffc0000fep-55, -0x1.f800fe1f7d07cp-16, 0x1.80001fc000000p-80},
{-0x1.0003f001fc3f0p-1, -0x1.fffffc0000000p-57, -0x1.0000000000000p-1, -0x1.fc1fffc0000fep-55, -0x1.0001f800fe1f8p+0, -0x1.3e0fff6000080p-54},
{0x1.0000000000000p-1, 0x1.fffffffffffc0p-60, 0x1.0000000000000p-1, 0x1.ffffffffffe00p-63, 0x1.0000000000000p+0, 0x1.1ffffffffffc0p-59},
{0x1.0000000000000p-1, 0x1.fffffffffffc0p-60, -0x1.0000000000000p-1, -0x1.ffffffffffe00p-63, 0x1.c000000000000p-60, 0x0.0000000000000p+0},
{-0x1.0000000000000p-1, -0x1.fffffffffffc0p-60, 0x1.0000000000000p-1, 0x1.ffffffffffe00p-63, -0x1.c000000000000p-60, 0x0.0000000000000p+0},
{-0x1.0000000000000p-1, -0x1.fffffffffffc0p-60, -0x1.0000000000000p-1, -0x1.ffffffffffe00p-63, -0x1.0000000000000p+0, -0x1.1ffffffffffc0p-59},
{0x1.0000000100000p-1, -0x1.0000000000000p-80, 0x1.0000004000000p-1, -0x1.0000000000000p-81, 0x1.0000002080000p+0, -0x1.8000000000000p-80},
{0x1.0000000100000p-1, -0x1.0000000000000p-80, -0x1.0000004000000p-1, 0x1.0000000000000p-81, -0x1.f800000000000p-28, -0x1.0000000000000p-81},
{-0x1.0000000100000p-1, 0x1.0000000000000p-80, 0x1.0000004000000p-1, -0x1.0000000000000p-81, 0x1.f800000000000p-28, 0x1.0000000000000p-81},
{-0x1.0000000100000p-1, 0x1.0000000000000p-80, -0x1.0000004000000p-1, 0x1.0000000000000p-81, -0x1.0000002080000p+0, 0x1.8000000000000p-80},
{0x1.0000010000000p-1, -0x1.ffffe00001000p-57, 0x1.000ffffff007fp-1, 0x1.ffffc00000000p-82, 0x1.0008007ff803fp+0, 0x1.c000041fffdc0p-54},
{0x1.0000010000000p-1, -0x1.ffffe00001000p-57, -0x1.000ffffff007fp-1, -0x1.ffffc00000000p-82, -0x1.ffdffe00fe400p-14, 0x1.effff20000000p-77},
{-0x1.0000010000000p-1, 0x1.ffffe00001000p-57, 0x1.000ffffff007fp-1, 0x1.ffffc00000000p-82, 0x1.ffdffe00fe400p-14, -0x1.effff20000000p-77},
{-0x1.0000010000000p-1, 0x1.ffffe00001000p-57, -0x1.000ffffff007fp-1, -0x1.ffffc00000000p-82, -0x1.0008007ff803fp+0, -0x1.c000041fffdc0p-54},
{0x1.1000fffffc000p-1, 0x1.fffffc003f000p-57, 0x1.0000000001000p-1, -0x1.0000000000000p-106, 0x1.08007ffffe800p+0, 0x1.fffffc003f000p-57},
{0x1.1000fffffc000p-1, 0x1.fffffc003f000p-57, -0x1.0000000001000p-1, 0x1.0000000000000p-106, 0x1.000fffffb0002p-5, -0x1.ffe07fc000000p-80},
{-0x1.1000fffffc000p-1, -0x1.fffffc003f000p-57, 0x1.0000000001000p-1, -0x1.0000000000000p-106, -0x1.000fffffb0002p-5, 0x1.ffe07fc000000p-80},
{-0x1.1000fffffc000p-1, -0x1.fffffc003f000p-57, -0x1.0000000001000p-1, 0x1.0000000000000p-106, -0x1.08007ffffe800p+0, -0x1.fffffc003f000p-57},
{0x1.00000fff80000p-1, 0x1.fffe000000000p-86, 0x1.0000001f00000p-1, 0x1.fffffff000000p-78, 0x1.0000080f40000p+0, 0x1.00ffff0000000p-77},
{0x1.00000fff80000p-1, 0x1.fffe000000000p-86, -0x1.0000001f00000p-1, -0x1.fffffff000000p-78, 0x1.fc10000000000p-22, -0x1.fe0001f000000p-78},
{-0x1.00000fff80000p-1, -0x1.fffe000000000p-86, 0x1.0000001f00000p-1, 0x1.fffffff000000p-78, -0x1.fc10000000000p-22, 0x1.fe0001f000000p-78},
{-0x1.00000fff80000p-1, -0x1.fffe000000000p-86, -0x1.0000001f00000p-1, -0x1.fffffff000000p-78, -0x1.0000080f40000p+0, -0x1.00ffff0000000p-77},
{0x1.3fdfffff02000p-1, -0x1.0000000000000p-79, 0x1.0000000000000p-1, 0x1.ffffffffffe00p-62, 0x1.1fefffff81000p+0, 0x1.ffff7fffffe00p-62},
{0x1.3fdfffff02000p-1, -0x1.0000000000000p-79, -0x1.0000000000000p-1, -0x1.ffffffffffe00p-62, 0x1.fefffff810000p-4, -0x1.00003ffffff00p-61},
{-0x1.3fdfffff02000p-1, 0x1.0000000000000p-79, 0x1.0000000000000p-1, 0x1.ffffffffffe00p-62, -0x1.fefffff810000p-4, 0x1.00003ffffff00p-61},
{-0x1.3fdfffff02000p-1, 0x1.0000000000000p-79, -0x1.0000000000000p-1, -0x1.ffffffffffe00p-62, -0x1.1fefffff81000p+0, -0x1.ffff7fffffe00p-62},
{0x1.0000000000008p-1, -0x1.e0ffffd002000p-67, 0x1.0000000000400p-1, 0x1.ffffffff03ffep-55, 0x1.0000000000204p+0, 0x1.ffe1efff06ffcp-55},
{0x1.0000000000008p-1, -0x1.e0ffffd002000p-67, -0x1.0000000000400p-1, -0x1.ffffffff03ffep-55, -0x1.fc4003c1ffe02p-44, 0x0.0000000000000p+0},
{-0x1.0000000000008p-1, 0x1.e0ffffd002000p-67, 0x1.0000000000400p-1, 0x1.ffffffff03ffep-55, 0x1.fc4003c1ffe02p-44, 0x0.0000000000000p+0},
{-0x1.0000000000008p-1, 0x1.e0ffffd002000p-67, -0x1.0000000000400p-1, -0x1.ffffffff03ffep-55, -0x1.0000000000204p+0, -0x1.ffe1efff06ffcp-55},
{0x1.000000fffe020p-1, -0x1.f800100000000p-74, 0x1.0001fff80ffc0p-1, 0x1.ff00000000000p-98, 0x1.0001007c06ff0p+0, -0x1.f8000e0000000p-74},
{0x1.000000fffe020p-1, -0x1.f800100000000p-74, -0x1.0001fff80ffc0p-1, -0x1.ff00000000000p-98, -0x1.fef811fa00000p-17, -0x1.f80011ff00000p-74},
{-0x1.000000fffe020p-1, 0x1.f800100000000p-74, 0x1.0001fff80ffc0p-1, 0x1.ff00000000000p-98, 0x1.fef811fa00000p-17, 0x1.f80011ff00000p-74},
{-0x1.000000fffe020p-1, 0x1.f800100000000p-74, -0x1.0001fff80ffc0p-1, -0x1.ff00000000000p-98, -0x1.0001007c06ff0p+0, 0x1.f8000e0000000p-74},
{0x1.4000000000000p-1, -0x1.0000000000000p-106, 0x1.0000000000010p-1, 0x1.e000000000000p-69, 0x1.2000000000008p+0, 0x1.e000000000000p-69},
{0x1.4000000000000p-1, -0x1.0000000000000p-106, -0x1.0000000000010p-1, -0x1.e000000000000p-69, 0x1.fffffffffff80p-4, -0x1.e000000008000p-69},
{-0x1.4000000000000p-1, 0x1.0000000000000p-106, 0x1.0000000000010p-1, 0x1.e000000000000p-69, -0x1.fffffffffff80p-4, 0x1.e000000008000p-69},
{-0x1.4000000000000p-1, 0x1.0000000000000p-106, -0x1.0000000000010p-1, -0x1.e000000000000p-69, -0x1.2000000000008p+0, -0x1.e000000000000p-69},
{0x1.2000000000000p-1, -0x1.fffffff01ff20p-58, 0x1.0000000000000p-1, 0x1.fff8003ff0000p-58, 0x1.1000000000000p+0, -0x1.ffec0bfc80000p-72},
{0x1.2000000000000p-1, -0x1.fffffff01ff20p-58, -0x1.0000000000000p-1, -0x1.fff8003ff0000p-58, 0x1.ffffffffffffep-5, 0x1.fff3fc0380000p-72},
{-0x1.2000000000000p-1, 0x1.fffffff01ff20p-58, 0x1.0000000000000p-1, 0x1.fff8003ff0000p-58, -0x1.ffffffffffffep-5, -0x1.fff3fc0380000p-72},
{-0x1.2000000000000p-1, 0x1.fffffff01ff20p-58, -0x1.0000000000000p-1, -0x1.fff8003ff0000p-58, -0x1.1000000000000p+0, 0x1.ffec0bfc80000p-72},
{0x1.2000000000000p-1, -0x1.fc08000000000p-93, 0x1.0000000038004p-1, -0x1.fffffc4000000p-70, 0x1.100000001c002p+0, -0x1.0000001c00000p-69},
{0x1.2000000000000p-1, -0x1.fc08000000000p-93, -0x1.0000000038004p-1, 0x1.fffffc4000000p-70, 0x1.fffffffc7ffc0p-5, 0x1.fffff847f0000p-70},
{-0x1.2000000000000p-1, 0x1.fc08000000000p-93, 0x1.0000000038004p-1, -0x1.fffffc4000000p-70, -0x1.fffffffc7ffc0p-5, -0x1.fffff847f0000p-70},
{-0x1.2000000000000p-1, 0x1.fc08000000000p-93, -0x1.0000000038004p-1, 0x1.fffffc4000000p-70, -0x1.100000001c002p+0, 0x1.0000001c00000p-69},
{0x1.0000000000200p-1, 0x0.0000000000000p+0, 0x1.00001ffff0000p-1, 0x1.ffff80001f000p-58, 0x1.00000ffff8100p+0, 0x1.ffff80001f000p-58},
{0x1.0000000000200p-1, 0x0.0000000000000p+0, -0x1.00001ffff0000p-1, -0x1.ffff80001f000p-58, -0x1.fffefe0010000p-21, 0x1.ffff840000000p-76},
{-0x1.0000000000200p-1, 0x0.0000000000000p+0, 0x1.00001ffff0000p-1, 0x1.ffff80001f000p-58, 0x1.fffefe0010000p-21, -0x1.ffff840000000p-76},
{-0x1.0000000000200p-1, 0x0.0000000000000p+0, -0x1.00001ffff0000p-1, -0x1.ffff80001f000p-58, -0x1.00000ffff8100p+0, -0x1.ffff80001f000p-58},
{0x1.0000100000000p-1, -0x1.0000000000000p-106, 0x1.0038000800000p-1, -0x1.fffffe0000080p-61, 0x1.001c080400000p+0, -0x1.fffffe0000100p-61},
{0x1.0000100000000p-1, -0x1.0000000000000p-106, -0x1.0038000800000p-1, 0x1.fffffe0000080p-61, -0x1.bf803fffffff0p-12, -0x1.0000000000000p-84},
{-0x1.0000100000000p-1, 0x1.0000000000000p-106, 0x1.0038000800000p-1, -0x1.fffffe0000080p-61, 0x1.bf803fffffff0p-12, 0x1.0000000000000p-84},
{-0x1.0000100000000p-1, 0x1.0000000000000p-106, -0x1.0038000800000p-1, 0x1.fffffe0000080p-61, -0x1.001c080400000p+0, 0x1.fffffe0000100p-61},
{0x1.0000000000004p-1, -0x1.ffffe00400000p-74, 0x1.0000000000000p-1, 0x1.ffffffffffff8p-57, 0x1.0000000000002p+0, 0x1.ffff00000ffe0p-57},
{0x1.0000000000004p-1, -0x1.ffffe00400000p-74, -0x1.0000000000000p-1, -0x1.ffffffffffff8p-57, 0x1.effff800007ffp-52, 0x1.0000000000000p-106},
{-0x1.0000000000004p-1, 0x1.ffffe00400000p-74, 0x1.0000000000000p-1, 0x1.ffffffffffff8p-57, -0x1.effff800007ffp-52, -0x1.0000000000000p-106},
{-0x1.0000000000004p-1, 0x1.ffffe00400000p-74, -0x1.0000000000000p-1, -0x1.ffffffffffff8p-57, -0x1.0000000000002p+0, -0x1.ffff00000ffe0p-57},
{0x1.0000000000100p-1, -0x1.ffffe7e400000p-71, 0x1.1ffe000008000p-1, -0x1.fffc000004000p-68, 0x1.0fff000004080p+0, -0x1.1ffdfe7e40000p-67},
{0x1.0000000000100p-1, -0x1.ffffe7e400000p-71, -0x1.1ffe000008000p-1, 0x1.fffc000004000p-68, -0x1.ffe000007f000p-5, 0x1.bffc030384000p-68},
{-0x1.0000000000100p-1, 0x1.ffffe7e400000p-71, 0x1.1ffe000008000p-1, -0x1.fffc000004000p-68, 0x1.ffe000007f000p-5, -0x1.bffc030384000p-68},
{-0x1.0000000000100p-1, 0x1.ffffe7e400000p-71, -0x1.1ffe000008000p-1, 0x1.fffc000004000p-68, -0x1.0fff000004080p+0, 0x1.1ffdfe7e40000p-67},
{0x1.0000000000004p-1, -0x1.0000000000000p-106, 0x1.00000007ffffep-1, 0x1.ffffe000f0000p-62, 0x1.0000000400001p+0, 0x1.ffffe000f0000p-62},
{0x1.0000000000004p-1, -0x1.0000000000000p-106, -0x1.00000007ffffep-1, -0x1.ffffe000f0000p-62, -0x1.ffffe803ffffcp-31, -0x1.e020000000000p-95},
{-0x1.0000000000004p-1, 0x1.0000000000000p-106, 0x1.00000007ffffep-1, 0x1.ffffe000f0000p-62, 0x1.ffffe803ffffcp-31, 0x1.e020000000000p-95},
{-0x1.0000000000004p-1, 0x1.0000000000000p-106, -0x1.00000007ffffep-1, -0x1.ffffe000f0000p-62, -0x1.0000000400001p+0, -0x1.ffffe000f0000p-62},
{0x1.0000000000000p-1, 0x1.fffffff000000p-62, 0x1.0001fffc00001p-1, -0x1.0000000000000p-80, 0x1.0000fffe00001p+0, -0x1.fe00004010000p-54},
{0x1.0000000000000p-1, 0x1.fffffff000000p-62, -0x1.0001fffc00001p-1, 0x1.0000000000000p-80, -0x1.fffc00000ff00p-17, 0x1.ff80000000000p-81},
{-0x1.0000000000000p-1, -0x1.fffffff000000p-62, 0x1.0001fffc00001p-1, -0x1.0000000000000p-80, 0x1.fffc00000ff00p-17, -0x1.ff80000000000p-81},
{-0x1.0000000000000p-1, -0x1.fffffff000000p-62, -0x1.0001fffc00001p-1, 0x1.0000000000000p-80, -0x1.0000fffe00001p+0, 0x1.fe00004010000p-54},
{0x1.00007fdfffff0p-1, 0x1.fff000f803fe0p-59, 0x1.0080000000000p-1, -0x1.0000000000000p-93, 0x1.00403fefffff8p+0, 0x1.fff000f7c4000p-59},
{0x1.00007fdfffff0p-1, 0x1.fff000f803fe0p-59, -0x1.0080000000000p-1, 0x1.0000000000000p-93, -0x1.fe00800003fe0p-11, -0x1.ffe0f78040000p-72},
{-0x1.00007fdfffff0p-1, -0x1.fff000f803fe0p-59, 0x1.0080000000000p-1, -0x1.0000000000000p-93, 0x1.fe00800003fe0p-11, 0x1.ffe0f78040000p-72},
{-0x1.00007fdfffff0p-1, -0x1.fff000f803fe0p-59, -0x1.0080000000000p-1, 0x1.0000000000000p-93, -0x1.00403fefffff8p+0, -0x1.fff000f7c4000p-59},
{0x1.0000000000080p-1, 0x0.0000000000000p+0, 0x1.0000080000000p-1, -0x1.0000000000000p-86, 0x1.0000040000040p+0, -0x1.0000000000000p-86},
{0x1.0000000000080p-1, 0x0.0000000000000p+0, -0x1.0000080000000p-1, 0x1.0000000000000p-86, -0x1.fffffe0000000p-23, 0x1.0000000000000p-86},
{-0x1.0000000000080p-1, 0x0.0000000000000p+0, 0x1.0000080000000p-1, -0x1.0000000000000p-86, 0x1.fffffe0000000p-23, -0x1.0000000000000p-86},
{-0x1.0000000000080p-1, 0x0.0000000000000p+0, -0x1.0000080000000p-1, 0x1.0000000000000p-86, -0x1.0000040000040p+0, 0x1.0000000000000p-86},
{0x1.0400000000000p-1, -0x1.0000000000000p-106, 0x1.0000000000400p-1, -0x1.fffff00008000p-57, 0x1.0200000000200p+0, -0x1.fffff00008000p-57},
{0x1.0400000000000p-1, -0x1.0000000000000p-106, -0x1.0000000000400p-1, 0x1.fffff00008000p-57, 0x1.ffffffffe0010p-8, -0x1.ffff001000000p-78},
{-0x1.0400000000000p-1, 0x1.0000000000000p-106, 0x1.0000000000400p-1, -0x1.fffff00008000p-57, -0x1.ffffffffe0010p-8, 0x1.ffff001000000p-78},
{-0x1.0400000000000p-1, 0x1.0000000000000p-106, -0x1.0000000000400p-1, 0x1.fffff00008000p-57, -0x1.0200000000200p+0, 0x1.fffff00008000p-57},
{0x1.0000000000001p-1, -0x1.fc0000c004000p-66, 0x1.0000000000200p-1, -0x1.ffe0000200000p-69, 0x1.0000000000100p+0, 0x1.ffdc403ff3fbcp-54},
{0x1.0000000000001p-1, -0x1.fc0000c004000p-66, -0x1.0000000000200p-1, 0x1.ffe0000200000p-69, -0x1.ff000de020060p-45, 0x1.e000000000000p-101},
{-0x1.0000000000001p-1, 0x1.fc0000c004000p-66, 0x1.0000000000200p-1, -0x1.ffe0000200000p-69, 0x1.ff000de020060p-45, -0x1.e000000000000p-101},
{-0x1.0000000000001p-1, 0x1.fc0000c004000p-66, -0x1.0000000000200p-1, 0x1.ffe0000200000p-69, -0x1.0000000000100p+0, -0x1.ffdc403ff3fbcp-54},
{0x1.00000c0000004p-1, -0x1.ff80000800000p-69, 0x1.18003ffff8000p-1, 0x1.ffffff8000000p-67, 0x1.0c0025fffc002p+0, 0x1.801fff7e00000p-67},
{0x1.00000c0000004p-1, -0x1.ff80000800000p-69, -0x1.18003ffff8000p-1, -0x1.ffffff8000000p-67, -0x1.80033fff7ffc0p-5, -0x1.3fefffc100000p-66},
{-0x1.00000c0000004p-1, 0x1.ff80000800000p-69, 0x1.18003ffff8000p-1, 0x1.ffffff8000000p-67, 0x1.80033fff7ffc0p-5, 0x1.3fefffc100000p-66},
{-0x1.00000c0000004p-1, 0x1.ff80000800000p-69, -0x1.18003ffff8000p-1, -0x1.ffffff8000000p-67, -0x1.0c0025fffc002p+0, -0x1.801fff7e00000p-67},
{0x1.1ffff80000040p-1, -0x1.0000000000000p-91, 0x1.000000000fe00p-1, 0x1.ffff7e0003ff8p-57, 0x1.0ffffc0007f20p+0, 0x1.ffff7dffc4000p-57},
{0x1.1ffff80000040p-1, -0x1.0000000000000p-91, -0x1.000000000fe00p-1, -0x1.ffff7e0003ff8p-57, 0x1.ffff7fff023fep-5, 0x1.03ff780100000p-74},
{-0x1.1ffff80000040p-1, 0x1.0000000000000p-91, 0x1.000000000fe00p-1, 0x1.ffff7e0003ff8p-57, -0x1.ffff7fff023fep-5, -0x1.03ff780100000p-74},
{-0x1.1ffff80000040p-1, 0x1.0000000000000p-91, -0x1.000000000fe00p-1, -0x1.ffff7e0003ff8p-57, -0x1.0ffffc0007f20p+0, -0x1.ffff7dffc4000p-57},
{0x1.3fc0000040000p-1, -0x1.fffffff802000p-57, 0x1.fffffffffff00p-1, 0x0.0000000000000p+0, 0x1.9fe000001ff80p+0, -0x1.fffffff802000p-57},
{0x1.3fc0000040000p-1, -0x1.fffffff802000p-57, -0x1.fffffffffff00p-1, 0x0.0000000000000p+0, -0x1.807fffff7fe00p-2, -0x1.fffffff802000p-57},
{-0x1.3fc0000040000p-1, 0x1.fffffff802000p-57, 0x1.fffffffffff00p-1, 0x0.0000000000000p+0, 0x1.807fffff7fe00p-2, 0x1.fffffff802000p-57},
{-0x1.3fc0000040000p-1, 0x1.fffffff802000p-57, -0x1.fffffffffff00p-1, 0x0.0000000000000p+0, -0x1.9fe000001ff80p+0, 0x1.fffffff802000p-57},
{0x1.00000001fffffp-1, 0x1.00001f8000000p-72, 0x1.0000000000200p-1, -0x1.0000000000000p-106, 0x1.0000000100100p+0, -0x1.ffffbffff8200p-54},
{0x1.00000001fffffp-1, 0x1.00001f8000000p-72, -0x1.0000000000200p-1, 0x1.0000000000000p-106, 0x1.ffdff00002000p-33, 0x1.f804000000000p-92},
{-0x1.00000001fffffp-1, -0x1.00001f8000000p-72, 0x1.0000000000200p-1, -0x1.0000000000000p-106, -0x1.ffdff00002000p-33, -0x1.f804000000000p-92},
{-0x1.00000001fffffp-1, -0x1.00001f8000000p-72, -0x1.0000000000200p-1, 0x1.0000000000000p-106, -0x1.0000000100100p+0, 0x1.ffffbffff8200p-54},
{0x1.0000000010000p-1, -0x1.0000000000000p-78, 0x1.1fff010000000p-1, 0x1.ff8c0000007fep-55, 0x1.0fff808008000p+0, 0x1.ff8bfe0000800p-55},
{0x1.0000000010000p-1, -0x1.0000000000000p-78, -0x1.1fff010000000p-1, -0x1.ff8c0000007fep-55, -0x1.fff00fff00008p-5, 0x1.cff7fffe00800p-65},
{-0x1.0000000010000p-1, 0x1.0000000000000p-78, 0x1.1fff010000000p-1, 0x1.ff8c0000007fep-55, 0x1.fff00fff00008p-5, -0x1.cff7fffe00800p-65},
{-0x1.0000000010000p-1, 0x1.0000000000000p-78, -0x1.1fff010000000p-1, -0x1.ff8c0000007fep-55, -0x1.0fff808008000p+0, -0x1.ff8bfe0000800p-55},
{0x1.0000000000002p-1, -0x1.0000000000000p-88, 0x1.0000000000200p-1, -0x1.0000000000000p-86, 0x1.0000000000101p+0, -0x1.4000000000000p-86},
{0x1.0000000000002p-1, -0x1.0000000000000p-88, -0x1.0000000000200p-1, 0x1.0000000000000p-86, -0x1.fdffffffffa00p-45, 0x0.0000000000000p+0},
{-0x1.0000000000002p-1, 0x1.0000000000000p-88, 0x1.0000000000200p-1, -0x1.0000000000000p-86, 0x1.fdffffffffa00p-45, 0x0.0000000000000p+0},
{-0x1.0000000000002p-1, 0x1.0000000000000p-88, -0x1.0000000000200p-1, 0x1.0000000000000p-86, -0x1.0000000000101p+0, 0x1.4000000000000p-86},
{0x1.0000000000002p-1, -0x1.ffff800008000p-68, 0x1.000000003fffep-1, 0x1.f81fc007c0000p-72, 0x1.0000000020000p+0, -0x1.e07d83ff90000p-68},
{0x1.0000000000002p-1, -0x1.ffff800008000p-68, -0x1.000000003fffep-1, -0x1.f81fc007c0000p-72, -0x1.fffe00021f818p-36, 0x1.ffbe000000000p-91},
{-0x1.0000000000002p-1, 0x1.ffff800008000p-68, 0x1.000000003fffep-1, 0x1.f81fc007c0000p-72, 0x1.fffe00021f818p-36, -0x1.ffbe000000000p-91},
{-0x1.0000000000002p-1, 0x1.ffff800008000p-68, -0x1.000000003fffep-1, -0x1.f81fc007c0000p-72, -0x1.0000000020000p+0, 0x1.e07d83ff90000p-68},
{0x1.7ffffffff0000p-1, 0x1.ffff07fffc000p-65, 0x1.000000000ffe0p-1, 0x1.ff0000003c000p-64, 0x1.3fffffffffff0p+0, 0x1.7f7fc2001d000p-63},
{0x1.7ffffffff0000p-1, 0x1.ffff07fffc000p-65, -0x1.000000000ffe0p-1, -0x1.ff0000003c000p-64, 0x1.ffffffff80080p-3, -0x1.fe00f8007c000p-65},
{-0x1.7ffffffff0000p-1, -0x1.ffff07fffc000p-65, 0x1.000000000ffe0p-1, 0x1.ff0000003c000p-64, -0x1.ffffffff80080p-3, 0x1.fe00f8007c000p-65},
{-0x1.7ffffffff0000p-1, -0x1.ffff07fffc000p-65, -0x1.000000000ffe0p-1, -0x1.ff0000003c000p-64, -0x1.3fffffffffff0p+0, -0x1.7f7fc2001d000p-63},
{0x1.00007fff00080p-1, -0x1.ffffff0001000p-66, 0x1.00000000007c0p-1, 0x1.ffffff8000000p-68, 0x1.00003fff80420p+0, -0x1.7fffff2000000p-66},
{0x1.00007fff00080p-1, -0x1.ffffff0001000p-66, -0x1.00000000007c0p-1, -0x1.ffffff8000000p-68, 0x1.fffbfe2ffffb0p-19, 0x1.1fff000000000p-90},
{-0x1.00007fff00080p-1, 0x1.ffffff0001000p-66, 0x1.00000000007c0p-1, 0x1.ffffff8000000p-68, -0x1.fffbfe2ffffb0p-19, -0x1.1fff000000000p-90},
{-0x1.00007fff00080p-1, 0x1.ffffff0001000p-66, -0x1.00000000007c0p-1, -0x1.ffffff8000000p-68, -0x1.00003fff80420p+0, 0x1.7fffff2000000p-66},
{0x1.1fff81ff00000p-1, 0x1.e007ffff00000p-58, 0x1.1ffffffff8000p-1, 0x1.fe001ffffffe0p-57, 0x1.1fffc0ff7c000p+0, 0x1.77020fffbfff0p-56},
{0x1.1fff81ff00000p-1, 0x1.e007ffff00000p-58, -0x1.1ffffffff8000p-1, -0x1.fe001ffffffe0p-57, -0x1.f803e0000437fp-19, -0x1.0003fff000000p-76},
{-0x1.1fff81ff00000p-1, -0x1.e007ffff00000p-58, 0x1.1ffffffff8000p-1, 0x1.fe001ffffffe0p-57, 0x1.f803e0000437fp-19, 0x1.0003fff000000p-76},
{-0x1.1fff81ff00000p-1, -0x1.e007ffff00000p-58, -0x1.1ffffffff8000p-1, -0x1.fe001ffffffe0p-57, -0x1.1fffc0ff7c000p+0, -0x1.77020fffbfff0p-56},
{0x1.0000000000000p-1, 0x1.fffc03fe00700p-62, 0x1.0000000000001p-1, -0x1.fff8060000f34p-56, 0x1.0000000000000p+0, 0x1.8201fa83fdc3ap-54},
{0x1.0000000000000p-1, 0x1.fffc03fe00700p-62, -0x1.0000000000001p-1, 0x1.fff8060000f34p-56, -0x1.7e02027c01c2cp-54, 0x0.0000000000000p+0},
{-0x1.0000000000000p-1, -0x1.fffc03fe00700p-62, 0x1.0000000000001p-1, -0x1.fff8060000f34p-56, 0x1.7e02027c01c2cp-54, 0x0.0000000000000p+0},
{-0x1.0000000000000p-1, -0x1.fffc03fe00700p-62, -0x1.0000000000001p-1, 0x1.fff8060000f34p-56, -0x1.0000000000000p+0, -0x1.8201fa83fdc3ap-54},
{0x1.001ffffc00080p-1, -0x1.fe00000200000p-62, 0x1.0000000001000p-1, -0x1.fe3ffffff0080p-61, 0x1.000ffffe00840p+0, -0x1.7ea0000078000p-60},
{0x1.001ffffc00080p-1, -0x1.fe00000200000p-62, -0x1.0000000001000p-1, 0x1.fe3ffffff0080p-61, 0x1.ffffbff080010p-13, -0x1.8000021ff0000p-70},
{-0x1.001ffffc00080p-1, 0x1.fe00000200000p-62, 0x1.0000000001000p-1, -0x1.fe3ffffff0080p-61, -0x1.ffffbff080010p-13, 0x1.8000021ff0000p-70},
{-0x1.001ffffc00080p-1, 0x1.fe00000200000p-62, -0x1.0000000001000p-1, 0x1.fe3ffffff0080p-61, -0x1.000ffffe00840p+0, 0x1.7ea0000078000p-60},
{0x1.00001fffff800p-1, 0x1.ffffff801ffc0p-60, 0x1.0000000002000p-1, -0x1.0000000000000p-106, 0x1.0000100000c00p+0, 0x1.ffffff801ff80p-60},
{0x1.00001fffff800p-1, 0x1.ffffff801ffc0p-60, -0x1.0000000002000p-1, 0x1.0000000000000p-106, 0x1.ffffd80004000p-21, -0x1.ff80000000000p-86},
{-0x1.00001fffff800p-1, -0x1.ffffff801ffc0p-60, 0x1.0000000002000p-1, -0x1.0000000000000p-106, -0x1.ffffd80004000p-21, 0x1.ff80000000000p-86},
{-0x1.00001fffff800p-1, -0x1.ffffff801ffc0p-60, -0x1.0000000002000p-1, 0x1.0000000000000p-106, -0x1.0000100000c00p+0, -0x1.ffffff801ff80p-60},
{0x1.0000000000000p-1, 0x1.ffff83ffbfff0p-57, 0x1.0000000000000p-1, 0x1.ffffffffe0000p-57, 0x1.0000000000000p+0, 0x1.ffffc1ffcfff8p-56},
{0x1.0000000000000p-1, 0x1.ffff83ffbfff0p-57, -0x1.0000000000000p-1, -0x1.ffffffffe0000p-57, -0x1.f000800400000p-75, 0x0.0000000000000p+0},
{-0x1.0000000000000p-1, -0x1.ffff83ffbfff0p-57, 0x1.0000000000000p-1, 0x1.ffffffffe0000p-57, 0x1.f000800400000p-75, 0x0.0000000000000p+0},
{-0x1.0000000000000p-1, -0x1.ffff83ffbfff0p-57, -0x1.0000000000000p-1, -0x1.ffffffffe0000p-57, -0x1.0000000000000p+0, -0x1.ffffc1ffcfff8p-56},
{0x1.0000000000000p-1, 0x1.fffe7ff3dff00p-62, 0x1.000007f830000p-1, 0x1.fff0000ffe100p-62, 0x1.000003fc18000p+0, 0x1.fff74001ef000p-61},
{0x1.0000000000000p-1, 0x1.fffe7ff3dff00p-62, -0x1.000007f830000p-1, -0x1.fff0000ffe100p-62, -0x1.fe0bffffffffep-23, -0x1.801c1e2000000p-78},
{-0x1.0000000000000p-1, -0x1.fffe7ff3dff00p-62, 0x1.000007f830000p-1, 0x1.fff0000ffe100p-62, 0x1.fe0bffffffffep-23, 0x1.801c1e2000000p-78},
{-0x1.0000000000000p-1, -0x1.fffe7ff3dff00p-62, -0x1.000007f830000p-1, -0x1.fff0000ffe100p-62, -0x1.000003fc18000p+0, -0x1.fff74001ef000p-61},
{0x1.00000ffffe000p-1, 0x1.fffffffffc000p-68, 0x1.0000000000002p-1, -0x1.fffffe01f8004p-56, 0x1.000007ffff001p+0, -0x1.ffdffe01f8008p-56},
{0x1.00000ffffe000p-1, 0x1.fffffffffc000p-68, -0x1.0000000000002p-1, 0x1.fffffe01f8004p-56, 0x1.ffffbffc80080p-22, -0x1.fe08000000000p-80},
{-0x1.00000ffffe000p-1, -0x1.fffffffffc000p-68, 0x1.0000000000002p-1, -0x1.fffffe01f8004p-56, -0x1.ffffbffc80080p-22, 0x1.fe08000000000p-80},
{-0x1.00000ffffe000p-1, -0x1.fffffffffc000p-68, -0x1.0000000000002p-1, 0x1.fffffe01f8004p-56, -0x1.000007ffff001p+0, 0x1.ffdffe01f8008p-56},
{0x1.0000000000060p-1, 0x1.fffe3fff00000p-64, 0x1.0000000002000p-1, -0x1.ffe0002000000p-58, 0x1.0000000001030p+0, -0x1.f7e0072004000p-58},
{0x1.0000000000060p-1, 0x1.fffe3fff00000p-64, -0x1.0000000002000p-1, 0x1.ffe0002000000p-58, -0x1.f9fefc1003700p-41, -0x1.0000000000000p-96},
{-0x1.0000000000060p-1, -0x1.fffe3fff00000p-64, 0x1.0000000002000p-1, -0x1.ffe0002000000p-58, 0x1.f9fefc1003700p-41, 0x1.0000000000000p-96},
{-0x1.0000000000060p-1, -0x1.fffe3fff00000p-64, -0x1.0000000002000p-1, 0x1.ffe0002000000p-58, -0x1.0000000001030p+0, 0x1.f7e0072004000p-58},
{0x1.0003ffffffff8p-1, 0x1.ffffe07e00000p-75, 0x1.07ff000008000p-1, -0x1.e000020000000p-62, 0x1.0401800003ffcp+0, -0x1.dff00200fc000p-62},
{0x1.0003ffffffff8p-1, 0x1.ffffe07e00000p-75, -0x1.07ff000008000p-1, 0x1.e000020000000p-62, -0x1.fec0000200200p-7, 0x1.e01001ff03f00p-62},
{-0x1.0003ffffffff8p-1, -0x1.ffffe07e00000p-75, 0x1.07ff000008000p-1, -0x1.e000020000000p-62, 0x1.fec0000200200p-7, -0x1.e01001ff03f00p-62},
{-0x1.0003ffffffff8p-1, -0x1.ffffe07e00000p-75, -0x1.07ff000008000p-1, 0x1.e000020000000p-62, -0x1.0401800003ffcp+0, 0x1.dff00200fc000p-62},
};

const int numAccuracyTests = sizeof(accuracyTests) / sizeof(struct testVector);

#include <math.h>

double testAccuracy() {
	int i;
	DD a, b, c, r;
	
	double worstUlps = 0.5;
	
	for (i=0; i<numAccuracyTests; ++i) {
		a.hi = accuracyTests[i].xhi;
		a.lo = accuracyTests[i].xlo;
		b.hi = accuracyTests[i].yhi;
		b.lo = accuracyTests[i].ylo;
		r.hi = accuracyTests[i].rhi;
		r.lo = accuracyTests[i].rlo;
		
		DD c = { .ld = __gcc_qadd(a.ld, b.ld) };
		
		double error = __builtin_fabs(((r.hi - c.hi) + r.lo) - c.lo);
		
		if (error != 0.0) {
		
			int exponent = ilogb(r.hi);
			exponent = (exponent < -1022 ? -1022 : exponent);
			double ulpError = scalbn(error, 106 - exponent);
		
			if (ulpError > worstUlps) {
#ifdef PRINT_ACCURACY_INFORMATION
				printf("New worst rounding error for (%a,%a) + (%a,%a):\n", a.hi, a.lo, b.hi, b.lo);
				printf("\tExpected: (%a,%a)\n", r.hi, r.lo);
				printf("\tComputed: (%a,%a)\n", c.hi, c.lo);
				printf("\tError: %f ulps\n", ulpError);
#endif
				worstUlps = ulpError;
			}
		}
	}
	
	return worstUlps;
}
























