/*
   Hmw2txt.c

   This is a program to extract text and remove attributes in a Homeword
   for DeskMate file.  Syntax:

      hmw2txt <infile> <outfile>

   or:

      hmw2txt <infile>

   If only <infile> is specified, <outfile> is generated by replacing the
   extension of the input filename with .txt.  It is an error if both
   filenames are the same.
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#define MAXLINE 128
#define HDRLEN 253
#define TRUE 1
#define FALSE 0

char inname[MAXLINE + 1], outname[MAXLINE + 1], *tempname;
FILE *infile, *outfile, *tempfile;
char buffer[512];
unsigned char firstchar, secondchar;

void strupper( char *inbuf, int nchars )
{
    int i;

    for ( i = 0; i < nchars; i++ )
        inbuf[i] = toupper( inbuf[i] );
}

int ismargintab( char *inbuf )
{
    char *newline, *thischar;

    if ( (newline = strchr( inbuf, '\n' )) == NULL )
        newline = inbuf + strlen( inbuf );
    inbuf++;
    newline--;
    if ( newline <= inbuf )
        return FALSE;

    for ( thischar = inbuf; thischar < newline; thischar++ )
        if ( strchr( "-0123456789[]^", *thischar ) == NULL )
            return FALSE;

    return TRUE;
}

int main( int argc, char *argv[] )
{
    char *dotpos;
    int textfound;

    if ( argc < 2 )
    {
        fputs(
            "Hmw2txt, Homeword for DeskMate to text converter.  Syntax:\n\n", 
            stderr );
        fputs( "   hmw2txt <infile> <outfile>\n\n", stderr );
        fputs(
            "where <infile> is input Homeword file, <outfile> is output\n",
            stderr );
        fputs(
            "text file.  <infile> is required, <outfile> will be generated\n",
            stderr );
        fputs( "from <infile> if not specified. [1]\n", stderr );
        return 1;
    }
    else if ( argc < 3 )
    {
        strcpy( inname, argv[1] );
        strcpy( outname, inname );
        if ( (dotpos = strrchr( outname, '.' )) == NULL )
            strcat( outname, ".txt" );
        else
            strcpy( dotpos, ".txt" );
    }
    else
    {
        strcpy( inname, argv[1] );
        strcpy( outname, argv[2] );
    }
    strupper( inname, strlen( inname ) );
    strupper( outname, strlen( outname ) );
    if ( !strcmp( inname, outname ) )
    {
        fputs( "Input and output files are the same. [17]\n", stderr );
        return 17;
    }

    if ( (infile = fopen( inname, "rb" )) == NULL )
    {
        fputs( "Failed to open input file. [2]\n", stderr );
        return 2;
    }
    if ( (outfile = fopen( outname, "w" )) == NULL )
    {
        fputs( "Failed to open output file. [3]\n", stderr );
        return 3;
    }

    if ( fread( buffer, 1, HDRLEN, infile ) != HDRLEN )
    {
        fputs( "Premature EOF on input file. [4]\n", stderr );
        return 4;
    }
    buffer[HDRLEN] = '\0';
    strupper( buffer, HDRLEN );
    if ( (strstr( buffer, "HOMEWORD" ) == NULL) ||
            (strstr( buffer, "DESKMATE" ) == NULL) )
    {
        fputs( "Input file is not a Homeword for DeskMate file. [5]\n",
            stderr );
        return 5;
    }

    textfound = FALSE;
    if ( fread( &secondchar, 1, 1, infile ) != 1 )
    {
        fputs( "Premature EOF on input file. [4]\n", stderr );
        return 4;
    }
    while ( !feof( infile ) )
    {
        firstchar = secondchar;
        fread( &secondchar, 1, 1, infile );
        if ( (firstchar <= '\x07') && isprint( secondchar ) )
        {
            textfound = TRUE;
            break;
        }
    }
    if ( !textfound )
    {
        fputs( "Input file contains no text. [6]\n", stderr );
        return 6;
    }

    if ( (tempname = tempnam( NULL, "HMW2TXT" )) == NULL )
    {
        fputs( "Failed to generate temporary file name. [7]\n", stderr );
        return 7;
    }
    if ( (tempfile = fopen( tempname, "w" )) == NULL )
    {
        fputs( "Failed to create temporary file. [8]\n", stderr );
        return 8;
    }

    while ( !feof( infile ) )
    {
        if ( fputc( secondchar, tempfile ) == NULL )
        {
            fputs( "Error writing to temporary file. [9]\n", stderr );
            fclose( tempfile );
            remove( tempname );
            return 9;
        }
        if ( !fread( &firstchar, 1, 1, infile ) )
        {
            fputs( "Error reading from input file. [10]\n", stderr );
            fclose( tempfile );
            remove( tempname );
            return 10;
        }
        while ( firstchar >= (unsigned char) '\xf0' )
        {
            if ( !fread( &secondchar, 1, 1, infile ) )
            {
                fputs( "Error reading from input file. [10]\n", stderr );
                fclose( tempfile );
                remove( tempname );
                return 10;
            }
            if ( !fread( &firstchar, 1, 1, infile ) )
            {
                fputs( "Error reading from input file. [10]\n", stderr );
                fclose( tempfile );
                remove( tempname );
                return 10;
            }
        }
        if ( firstchar == '\x0a' /* line feed */ )
        {
            secondchar = '\n';
            continue;
        }
        if ( firstchar == '\x1a' /* EOF char */ )
        {
            secondchar = '\x1a';
            break;
        }
        if ( firstchar > '\x07' )
        {
            fputs( "Unexpected data in Homeword file. [11]\n", stderr );
            fclose( tempfile );
            remove( tempname );
            return 11;
        }
        if ( !fread( &secondchar, 1, 1, infile ) )
        {
            fputs( "Error reading from input file. [10]\n", stderr );
            fclose( tempfile );
            remove( tempname );
            return 10;
        }
    }
    if ( fputc( secondchar, stderr ) == NULL )
    {
        fputs( "Error writing to temporary file. [9]\n", stderr );
        fclose( tempfile );
        remove( tempname );
        return 9;
    }

    if ( fclose( tempfile ) )
    {
        fputs( "Error closing temporary file. [12]\n", stderr );
        remove( tempname );
        return 12;
    }
    if ( (tempfile = fopen( tempname, "r" )) == NULL )
    {
        fputs( "Error opening temporary file for reading. [13]\n", stderr );
        remove( tempname );
        return 13;
    }

    while ( !feof( tempfile ) )
    {
        if ( fgets( buffer, 512, tempfile ) == NULL )
        {
            if ( ferror( tempfile ) )
            {
                fputs( "Error reading from temporary file. [14]\n", stderr );
                fclose( tempfile );
                remove( tempname );
                return 14;
            }
            else
                continue;
        }
        if ( !ismargintab( buffer ) )
            if ( fputs( buffer, outfile ) == EOF )
            {
                fputs( "Error writing to output file. [15]\n", stderr );
                fclose( tempfile );
                remove( tempname );
                return 15;
            }
    }

    if ( fclose( outfile ) )
    {
        fputs( "Error closing output file. [16]\n", stderr );
        fclose( tempfile );
        remove( tempname );
        return 16;
    }   

    fclose( tempfile );
    remove( tempname );
    fclose( infile );
    fclose( outfile );
    return 0;
}
