// HexEditDoc.cpp : implementation of the CHexEditDoc class
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

#include "stdafx.h"
#include "HexEdit.h"

#include <stdio.h>                              // for rename() and remove()
#include <io.h>                                 // for _access()

#include "HexEditDoc.h"
#include "HexEditView.h"
#include "boyer.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CHexEditDoc

IMPLEMENT_DYNCREATE(CHexEditDoc, CDocument)

BEGIN_MESSAGE_MAP(CHexEditDoc, CDocument)
        //{{AFX_MSG_MAP(CHexEditDoc)
	ON_COMMAND(ID_DOCTEST, OnDocTest)
	//}}AFX_MSG_MAP
        ON_COMMAND(ID_FILE_CLOSE, OnFileClose)
        ON_COMMAND(ID_FILE_SAVE, OnFileSave)
        ON_COMMAND(ID_FILE_SAVE_AS, OnFileSaveAs)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CHexEditDoc construction/destruction

CHexEditDoc::CHexEditDoc()
 : start_event_(FALSE, TRUE), stopped_event_(FALSE, TRUE)
{
    length_ = 0L;

    pthread_ = NULL;

#ifndef NDEBUG
    // Make default capacity for undo_ vector small to force reallocation sooner.
    // This increases likelihood of catching bugs related to reallocation.
    undo_.reserve(2);
#else
    // Pre-allocate room for 128 elts for initial speed
    undo_.reserve(128);
#endif
}

CHexEditDoc::~CHexEditDoc()
{
}

/////////////////////////////////////////////////////////////////////////////
// CHexEditDoc diagnostics

#ifdef _DEBUG
void CHexEditDoc::AssertValid() const
{
        CDocument::AssertValid();
}

void CHexEditDoc::Dump(CDumpContext& dc) const
{
    if (file_.m_hFile != CFile::hFileNull)
    {
        dc << "\nFILE = " << file_.GetFilePath();
        if (readonly_) dc << " (READ ONLY)";
    }
    else
        dc << "\n(No file is open)";
    dc << "\nLENGTH = " << length_;

    if (last_view_ != NULL)
    {
        CString ss;
        last_view_->GetWindowText(ss);
        dc << "\nLast change by: " << ss;
    }

    dc << "\nUNDO INFO";
        std::vector <doc_undo>::const_iterator pu;
    for (pu = undo_.begin(); pu != undo_.end(); ++pu)
    {
        dc << "\n  ";
        switch ((*pu).utype)
        {
        case mod_insert : dc << "INSERT      "; break;
        case mod_replace: dc << "REPLACE     "; break;
        case mod_repback: dc << "REPLACE BACK"; break;
        case mod_delforw: dc << "DEL FORWARD "; break;
        case mod_delback: dc << "DEL BACK    "; break;
        default:    dc << "UNKNOWN TYPE"; break;
        }
        dc << " @" << long((*pu).address);
        dc << " len:" << long((*pu).len);
    }

    dc << "\nLOCATION INFO";
        std::list <doc_loc>::const_iterator pl;
    for (pl = loc_.begin(); pl != loc_.end(); ++pl)
    {
        dc << "\n  ";
        switch ((*pl).location)
        {
        case loc_mem : dc << "MEMORY"; break;
        case loc_file: dc << "FILE  "; break;
        default:       dc << "??????"; break;
        }
        dc << " len = " << long((*pl).len);
    }

    dc << "\n";

    CDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CHexEditDoc commands

// Called indirectly as part of File/New command
BOOL CHexEditDoc::OnNewDocument()
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    if (!CDocument::OnNewDocument())
    {
        aa->mac_error_ = 20;
        return FALSE;
    }

    DeleteContents();
    SetModifiedFlag(FALSE);
    readonly_ = FALSE;
    length_ = 0L;
    last_view_ = NULL;

    if (aa->bg_search_)
        CreateThread();

    return TRUE;
}

// Called indirectly as part of File/Open command
BOOL CHexEditDoc::OnOpenDocument(LPCTSTR lpszPathName) 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    DeleteContents();
    SetModifiedFlag(FALSE);

    if (!open_file(lpszPathName))
        return FALSE;               // open_file has already set mac_error_ = 10

    length_ = file_.GetLength();
    last_view_ = NULL;

    // Init locations list with all of original file as only loc record
    ASSERT(pthread_ == NULL);       // Must modify loc_ before creating thread (else docdata_ needs to be locked)
    loc_.push_back(doc_loc(0L, file_.GetLength()));

    if (aa->bg_search_)
    {
        CreateThread();
        if (aa->pboyer_ != NULL)        // If a search has already been done do bg search on newly opened file
            StartSearch();
    }

    return TRUE;
}

// Called when document closed probably as part of File/Close command
void CHexEditDoc::OnCloseDocument() 
{
    POSITION pos = GetFirstViewPosition();
    if (pos != NULL)
        ((CHexEditView *)GetNextView(pos))->StoreOptions();

    DeleteContents();
    ASSERT(pthread_ == NULL);

    CDocument::OnCloseDocument();
}

// Handles File/Close command
void CHexEditDoc::OnFileClose()
{
    CDocument::OnFileClose();
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_close);

    // BG search finished message may be lost if modeless dlg running
    ((CHexEditApp *)AfxGetApp())->CheckBGSearchFinished();
}

BOOL CHexEditDoc::SaveModified() 
{
    BOOL retval = CDocument::SaveModified();

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!retval) aa->mac_error_ = 2;
    return retval;
}

// Save command
void CHexEditDoc::OnFileSave()
{
    CDocument::OnFileSave();
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_save);
}

// Save As command
void CHexEditDoc::OnFileSaveAs()
{
    CDocument::OnFileSaveAs();
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_saveas);
}

// Called as part of Save and Save As commands
BOOL CHexEditDoc::DoSave(LPCTSTR lpszPathName, BOOL bReplace)
{
    BOOL retval = CDocument::DoSave(lpszPathName, bReplace);

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!retval && aa->mac_error_ < 2) aa->mac_error_ = 2;
    return retval;
}

// Called as part of Save and Save As commands
BOOL CHexEditDoc::OnSaveDocument(LPCTSTR lpszPathName) 
{
    last_view_ = NULL;
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    if (aa->playing_)
    {
        if (::HMessageBox("Saving file to disk will lose all undo information\r"
                          "Continue?", MB_OKCANCEL) != IDOK)
        {
            aa->mac_error_ = 10;
            return TRUE;
        }
        else
            aa->mac_error_ = 1;
    }

    // If backup file not required and no insertions/deletions
    if (!aa->backup() && only_over() && lpszPathName == file_.GetFilePath())
    {
        // Just write changes back to file in place
        // (avoid worries about disk space, creating temp files etc)
        WriteInPlace();
//      file_.Seek(0L, CFile::begin);
    }
    else
    {
        CFile temp;                             // File data is saved to
        CString temp_name(lpszPathName);        // Name of saved file
        CString backup_name;                    // Name of backup file
        int path_len;                           // Length of path part of file name

        // Since we may change the file being accessed and the location data (loc_) we need to stop
        // the bg search from accessing it during this time.  (Auto Unlock when it goes out of scope)
        CSingleLock sl(&docdata_, TRUE);

        // Create temp file name
        if ( (path_len = temp_name.ReverseFind('\\')) != -1 ||
             (path_len = temp_name.ReverseFind(':')) != -1 )
            temp_name = temp_name.Left(path_len+1);
        else
            temp_name.Empty();
        temp_name += "~HxEd";

        for (int ii = 0; ii < 1000; ++ii)
        {
            CString tt;
            tt.Format("%03d", ii);
            if (_access(temp_name + tt + ".TMP", 0) < 0)
            {
                // File does not exist so use this file name
                temp_name += tt;
                temp_name += ".TMP";
                break;
            }
        }
        if (ii >= 1000)
        {
            aa->mac_error_ = 20;
            return FALSE;
        }

        // Save current file to temp file
        if (!WriteData(temp_name, 0, length_))
            return FALSE;                       // already done: mac_error_ = 10

        // If we have a file open then close it
        if (file_.m_hFile != CFile::hFileNull)
        {
            file_.Close();
            if (pthread_ != NULL && file2_.m_hFile != CFile::hFileNull)
                file2_.Close();
        }

        // If file we are writing exists (Save or SaveAs overwriting another file)
        if (_access(lpszPathName, 0) != -1)
        {
            if (aa->backup())
            {
                int ext_len;
                backup_name = lpszPathName;

                // Make old file the backup file
                if ((ext_len = backup_name.ReverseFind('.')) > path_len)
                    backup_name = backup_name.Left(ext_len);

                if (_stricmp(lpszPathName, (const char *)(backup_name + ".BAK")) == 0)
                {
                    // We're editing .BAK so save old as .BAC
                    backup_name += ".BAC";
                    CString mess;
                    mess.Format("Creating backup file\r \"%s\"",
                        (const char *)backup_name);
                    if (::HMessageBox(mess, MB_OKCANCEL) == IDCANCEL)
                    {
                        // User doesn't want to wipe out .BAC
                        (void)remove(temp_name);
                        (void)open_file(lpszPathName);
                        aa->mac_error_ = 2;
                        return FALSE;
                    }
                }
                else
                    backup_name += ".BAK";

                // Remove previous backup file if present
                if (_access(backup_name, 0) != -1)
                    if (remove(backup_name) == -1)
                    {
                        (void)remove(temp_name);
                        (void)open_file(lpszPathName);

                        CString mess;
                        mess.Format("Could not remove previous backup file\r \"%s\"",
                            (const char *)backup_name);
                        ::HMessageBox(mess);
                        aa->mac_error_ = 10;
                        return FALSE;
                    }

                // Rename the old file to the backup file name
                if (rename(lpszPathName, backup_name) == -1)
                {
                    (void)remove(temp_name);
                    (void)open_file(lpszPathName);              // Reopen orig. file

                    CString mess;
                    mess.Format("Could not create backup file\r \"%s\"",
                        (const char *)backup_name);
                    ::HMessageBox(mess);
                    aa->mac_error_ = 10;
                    return FALSE;
                }
            }
            else
            {
                // Delete the file.  Note if this was due to user selecting File-SaveAs
                // to an existing file then the user should have already been prompted
                // if he wants to overwrite by the File-Save common dialog.
                if (remove(lpszPathName) == -1)
                {
                    (void)remove(temp_name);
                    (void)open_file(lpszPathName);              // Reopen orig. file

                    CString mess;
                    mess.Format("Old file \"%s\" could not be removed", lpszPathName);
                    ::HMessageBox(mess);
                    aa->mac_error_ = 10;
                    return FALSE;
                }
            }
        }

        // Rename the temp file to the real file name
        if (rename(temp_name, lpszPathName) == -1)
        {
            // This should not happen (since temp_name must exist and lpszPathName
            // was successfully removed if it existed) but allow for it anyway

            // Do we have a backup of the previous file?
            if (!backup_name.IsEmpty())
            {
                // Reopen the old file (keep undo info etc) - this allows the
                // user to possibly save the file as something else
                if (rename(backup_name, lpszPathName) != -1)
                    (void)open_file(lpszPathName);
                else
                    (void)open_file(backup_name);       // Open with backup name
                (void)remove(temp_name);
            }
            else
            {
                // As a last resort open the new file (with wrong name)
                (void)open_file(temp_name);

                // But now undo info, length_ etc are incompatible
                length_ = file_.GetLength();
                // Remove all undo info and just use all of new file as only loc record
                CRemoveHint rh(undo_.size() - 1);
                UpdateAllViews(NULL, 0, &rh);
                undo_.clear();
                loc_.clear();
                loc_.push_back(doc_loc(0L, length_));
                SetModifiedFlag(FALSE);
            }

            CString mess;
            mess.Format("File \"%s\" could not be renamed\r"
                        "  to \"%s\"", temp_name, lpszPathName);
            ::HMessageBox(mess);
            aa->mac_error_ = 10;
            return FALSE;
        }

        // Open the new file as the document file (file_).
        if (!open_file(lpszPathName))
            return FALSE;                       // already done: mac_error_ = 10
    }

    length_ = file_.GetLength();

    // Notify views to remove undo info up to the last doc undo
    CRemoveHint rh(undo_.size() - 1);
    UpdateAllViews(NULL, 0, &rh);

    // Remove all undo info and just use all of new file as only loc record
    undo_.clear();
    loc_.clear();
    loc_.push_back(doc_loc(0L, length_));

    SetModifiedFlag(FALSE);

    // BG search finished message may be lost if modeless dlg running
    ((CHexEditApp *)AfxGetApp())->CheckBGSearchFinished();

    return TRUE;
}

BOOL CHexEditDoc::open_file(LPCTSTR lpszPathName)
{
    CFileException fe;

    if (file_.Open(lpszPathName,
                CFile::modeReadWrite|CFile::shareDenyWrite|CFile::typeBinary))
    {
        // Opened OK for writing
        readonly_ = FALSE;
    }
    else if (file_.Open(lpszPathName,
                CFile::modeRead|CFile::shareDenyWrite|CFile::typeBinary, &fe) )
    {
        // Open for write (above) failed but open for read was OK
        CString mess;
        mess.Format("%s is in use or is read only).\r\r"
            "It is opened for read only (no changes possible)", lpszPathName);
        ::HMessageBox(mess);
        readonly_ = TRUE;
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        aa->mac_error_ = 1;
    }
    else if (file_.Open(lpszPathName,
                CFile::modeRead|CFile::shareDenyNone|CFile::typeBinary, &fe) )
    {
        // Open for write (above) failed but open for read was OK
        CString mess;
        mess.Format("%s is in use elsewhere for writing.\r\r"
            "It is opened for read only (no changes possible).\r"
            "Note that file contents may change without warning.", lpszPathName);
        ::HMessageBox(mess);
        readonly_ = TRUE;
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        aa->mac_error_ = 2;
    }
    else
    {
        // Display info about why the open failed
        CString mess(lpszPathName);
        CFileStatus fs;

        switch (fe.m_cause)
        {
        case CFileException::fileNotFound:
            mess += "\rdoes not exist";
            break;
        case CFileException::badPath:
            mess += "\ris an invalid file name or the drive/path does not exist";
            break;
        case CFileException::tooManyOpenFiles:
            mess += "\r- too many files already open";
            break;
        case CFileException::accessDenied:
            if (!CFile::GetStatus(lpszPathName, fs))
                mess += "\rdoes not exist";
            else
            {
                if (fs.m_attribute & CFile::directory)
                    mess += "\ris a directory";
                else if (fs.m_attribute & (CFile::volume|CFile::hidden|CFile::system))
                    mess += "\ris a special file";
                else
                    mess += "\rcannot be used (reason unknown)";
            }
            break;
        case CFileException::sharingViolation:
        case CFileException::lockViolation:
            mess += "\ris in use";
            break;
        case CFileException::hardIO:
            mess += "\r- hardware error";
            break;
        default:
            mess += "\rcould not be opened (reason unknown)";
            break;
        }
        ::HMessageBox(mess);

        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        aa->mac_error_ = 10;
        return FALSE;
    }

    // If doing background searches and the newly opened file is not the same
    // as file2_ then close file2_ and open it as the new file.
    if (pthread_ != NULL && 
        (file2_.m_hFile == CFile::hFileNull ||
         file_.GetFilePath() != file2_.GetFilePath()) )
    {
        if (file2_.m_hFile != CFile::hFileNull)
            file2_.Close();

        if (!file2_.Open(file_.GetFilePath(),
                    CFile::modeRead|CFile::shareDenyNone|CFile::typeBinary) )
        {
            TRACE1("File2 open failed for %p\n", this);
            return FALSE;
        }
    }

    return TRUE;
}

void CHexEditDoc::DeleteContents() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    // Close file if it was opened successfully
    if (file_.m_hFile != CFile::hFileNull)
        file_.Close();
        
    if (aa->bg_search_ && pthread_ != NULL)
        KillThread();

    undo_.clear();
    loc_.clear();               // Done after thread killed so no docdata_ lock needed

    CDocument::DeleteContents();
}

void CHexEditDoc::SetModifiedFlag(BOOL bMod /*=TRUE*/)
{
    if (IsModified() != bMod)
    {
        // Modified status has changed (on or off)
        if (bMod)
            SetTitle(GetTitle() + " *");
        else
        {
            CString title = GetTitle();
            int len = title.GetLength();
            if (len > 1 && title[len-1] == '*')
                SetTitle(title.Left(len-2));
        }
        UpdateFrameCounts();
        CDocument::SetModifiedFlag(bMod);
    }
}
