; ROMDRIVE.ASM
;
; This is a device driver for the ROM disk on the Tandy 1000-series.  Several
; people have stated that they lost access to the ROM drive (and consequently
; the ROM-based DeskMate program) after upgrading to DOS version 5 or above.
; This installable driver is intended to restore access to the ROM drive when
; placed in the CONFIG.SYS file.
;
; This driver accesses the ROM disk at the BIOS level through Int 13h, using
; the undocumented drive number of the ROM drive (drive 8).  Since the drive
; number is undocumented, I cannot be sure that this driver will work on all
; machines.  The driver will abort installation if unable to locate the drive.
;
; Modification added May 23, 1997:  I'm now storing the ROM drive number in
; the BIOS data segment at 40:C2h, as provided in the _1000 TL Technical
; Reference Manual_, software section, p. 59.
;

	ORG	0

;
; Data.
;
; Device header.
;
DEVICEHEADER	DD	-1		; link to next driver
		DW	0		; device attribute:  block device, no 
					;   IOCTL
		DW	OFFSET STRATEGY		; strategy entry point
		DW	OFFSET INTERRUPT  	; interrupt entry point
		DB	1		; 1 unit (logical drive)
		DB	7 DUP (0)	; reserved by DOS
		;
		; Pointer to request header.
		;
REQUESTPTR	DD	0
		;
		; BIOS parameter block.  Initialized from boot sector on
		; ROM drive.
		;
BPB		LABEL	BYTE
SECTORSIZE	DW	0		; sector size in bytes
CLUSTERSIZE	DB	0		; sectors per cluster
NUMRESERVED	DW	0		; number of reserved sectors
NUMFATS		DB	0		; number of file allocation tables
ROOTDIRENTRIES	DW	0		; maxiumum files in the root directory
TOTALSECTORS	DW	0		; total number of sectors on disk
MEDIAID		DB	0		; media ID byte
FATSIZE		DW	0		; number of sectors per FAT
TRACKLEN	DW	0		; number of sectors per track
NUMHEADS	DW	0		; number of drive heads
NUMHIDDEN	DW	0		; number of hidden sectors
		;
		; Device driver error codes.
		;
WRITEPROTECT	EQU	00h		; write-protect violation
BADCOMMAND	EQU	03h		; unknown/unsupported function code
BADSECTOR	EQU	08h		; invalid sector number
GENERALFAIL	EQU	0Ch		; general failure (programming error)
		;
		; Jump table for driver functions.
		;
FUNCTIONJUMPS	LABEL	WORD
		DW	OFFSET INIT		; Init
		DW	OFFSET MEDIACHECK	; Media Check
		DW	OFFSET BUILDBPB		; Build BIOS Parameter Block
		DW	OFFSET BADFUNCTION	; IOCTL Read (unsupported)
		DW	OFFSET READ		; Read
		DW	OFFSET NULLFUNCTION	; Nondestructive read
						;   (meaningless)
		DW	OFFSET NULLFUNCTION	; Input Status (meaningless)
		DW	OFFSET NULLFUNCTION	; Flush Input Buffers
						;   (meaningless)
		DW	OFFSET WRITE		; Write (returns error)
		DW	OFFSET WRITE		; Write with Verify (also error)
		DW	OFFSET NULLFUNCTION	; Output Status (meaningless)
		DW	OFFSET BADFUNCTION	; IOCTL Write (unsupported)
		;
		; BPB pointer "array" (1 element) for Init.
		;
BPBARRAY	DW	OFFSET BPB
		;
		; Request header fields, static part.
		;
STRUC
		DB	?	; header length, ignored
		DB	?	; unit number, ignored
  COMMANDCODE	DB	?	; driver function code
  STATUS	DW	?	; status returned
ENDS

;
; Code - resident part.
;
; Strategy routine.
;
STRATEGY:
	MOV	WORD PTR CS:REQUESTPTR,BX
	MOV	WORD PTR CS:REQUESTPTR+2,ES
	RETF
;
; Interrupt routine.
;
; Preserve caller's registers on stack.
;
INTERRUPT:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
	PUSH	BP
	PUSH	DS
	PUSH	ES
	;
	; String copy moves upward.
	;
	CLD
	;
	; DS addresses local data.
	;
	MOV	AX,CS
	MOV	DS,AX
	;
	; ES:BX addresses request header.
	;
	LES	BX,REQUESTPTR
	;
	; Get and verify command code.
	;
	MOV	AL,ES:[BX].COMMANDCODE
	CMP	AL,12
	JBE	COMMANDOK
	;
	; Error:  command out of range.
	;
	MOV	AL,BADCOMMAND
	JMP	ERROR
	;
	; Command code in range.  Jump to function routine.
	;
COMMANDOK:
	XOR	AH,AH
	SHL	AX,1
	MOV	DI,AX
	JMP	[DI+FUNCTIONJUMPS]
	;
	; Functions should jump to here in case of error.  ES:BX must address
	; the request header, AL must contain an error code.
	;
ERROR:
	MOV	AH,80h
	JMP	DONE
	;
	; Functions should jump to here if successful.  ES:BX must address the
	; request header.
	;
SUCCESS:
	XOR	AX,AX
	;
	; Save status in header, restore caller's registers, and return.
	;
DONE:
	OR	AX,0100h		; set done bit
	MOV	ES:[BX].STATUS,AX	
	POP	ES
	POP	DS
	POP	BP
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RETF

;
; Driver functions.
;
; Media Check.  Returns medium not changed.
;
STRUC					; variable request header part
 		DB	0Eh DUP (?)
  CHANGECODE	DB	?
ENDS
MEDIACHECK:
	MOV	ES:[BX].CHANGECODE,1
	JMP	SUCCESS
;
; Build BIOS Parameter Block.
;
STRUC					; variable request header part
		DB	12h DUP (?)
  BPBOFFSET	DW	?		; offset of BIOS parameter block
  BPBSEGMENT	DW	?		; segment of BIOS parameter block
ENDS
BUILDBPB:
	MOV	ES:[BX].BPBOFFSET,OFFSET BPB
	MOV	ES:[BX].BPBSEGMENT,CS
	JMP	SUCCESS
;
; "Bad function," i.e., unsupported command.  Return error code.
;
BADFUNCTION:
	MOV	AL,BADCOMMAND
	JMP	ERROR
;
; "Null function," i.e., function used by character devices only and 
; meaningless for this driver.  Return success.
;
NULLFUNCTION:
	JMP	SUCCESS
;
; Write.  Return write-protect error (ROM drive is read-only).
;
STRUC					; variable request header part
		DB	12h DUP (?)
  NSECTORS	DW	?		; number of sectors transferred (0)
ENDS
WRITE:
	MOV	ES:[BX].NSECTORS,0
	MOV	AL,WRITEPROTECT
	JMP	ERROR
;
; Read.  Read data from ROM disk.
;
STRUC					; variable request header part
		DB	0Eh DUP (?)
  DATAPTR	DD	?		; pointer to caller's buffer
  NSECTORS	DW	?		; number of sectors requested/
					;   transferred
  STARTSECTOR	DW	?		; starting sector of request
ENDS
	;
	; Check for valid range of sectors in request.
	;
READ:
	MOV	AX,ES:[BX].STARTSECTOR
	CMP	AX,TOTALSECTORS
	JAE	READ_BADSECTOR
	ADD	AX,ES:[BX].NSECTORS
	CMP	AX,TOTALSECTORS
	JA	READ_BADSECTOR
	;
	; Valid range of sectors.  Save ES, BX for return.
	;
	PUSH	ES
	PUSH	BX
	;
	; AX = logical sector, CX = loop count, ES:DI -> buffer
	;
	MOV	AX,ES:[BX].STARTSECTOR
	MOV	CX,ES:[BX].NSECTORS
	LES	DI,ES:[BX].DATAPTR
	JCXZ	READ_LOOPEND		; request for zero sectors
	;
	; Loop over sectors in the request.
	;
READ_LOOP:
	PUSH	CX
	PUSH	AX
	CALL	READ_CONVERT		; convert geometry
	CALL	READ_CALLBIOS		; read a sector
	JC	READ_PROGERR		; jump should never be made
	;
	; Move buffer pointer.
	;
	XOR	AX,AX
	MOV	CL,4
	ADD	DI,SECTORSIZE
	RCR	AX,CL
	MOV	BX,ES
	ADD	BX,AX
	MOV	ES,BX
	;
	; Do next sector (if more).
	;
	POP	AX
	INC	AX
	POP	CX
	LOOP	READ_LOOP
	;
	; Restore ES, BX and return.
	;
READ_LOOPEND:
	POP	BX
	POP	ES
	JMP	SUCCESS
	;
	; Error:  invalid sector number.
	;
READ_BADSECTOR:
	MOV	AL,BADSECTOR
	MOV	ES:[BX].NSECTORS,0
	JMP	ERROR
	;
	; Error:  BIOS indicated error.  This should never happen.  Indicates
	; programming error.
	;
READ_PROGERR:
	POP	AX
	POP	CX
	POP	BX
	POP	ES
	MOV	AX,ES:[BX].NSECTORS
	SUB	AX,CX
	MOV	ES:[BX].NSECTORS,AX
	MOV	AL,GENERALFAIL
	JMP	ERROR
	;
	; Subroutine, converts logical sector in AX, returns:
	;   AX = sector
	;   BX = head
	;   CX = cylinder
	;
READ_CONVERT:
	PUSH	DX
	XOR	DX,DX
	DIV	TRACKLEN
	INC	DX		; increment, push remainder
	PUSH	DX
	XOR	DX,DX
	DIV	NUMHEADS
	MOV	BX,DX		; BX = remainder (head)
	MOV	CX,AX		; CX = quotient (cylinder)
	POP	AX		; AX = sector
	POP	DX
	RET
	;
	; Subroutine, calls the BIOS to read a sector.  Requires:
	;   AX = sector
	;   BX = head
	;   CX = cylinder
	;   ES:DI -> read buffer
	; Returns carry set if error.
	;
READ_CALLBIOS:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	DI
	MOV	DH,BL		; DH = head
	MOV	BX,DI		; ES:BX -> buffer
	XCHG	CH,CL		; CH  = low 8 bits of cylinder
	ROR	CL,1		; bits 6,7 of CL = high 2 bits of cylinder
	ROR	CL,1
	AND	CL,0C0h
	AND	AL,3Fh		; bits 0-5 of CL = sector
	OR	CL,AL
	MOV	AX,0201h	; code:  read one sector
	MOV	DL,8		; drive 8 = ROM drive (undocumented)
	INT	13h		; call BIOS to read sector
	POP	DI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; End of resident part, begin transient initialization code.
;
; Init routine.
;
STRUC					; variable request header part
		DB	0Dh DUP (?)
  UNITS		DB	?		; number of units (drives), set to 1
  FREEMEMOFFS	DW	?		; pointer to free memory at end of 
  FREEMEMSEG	DW	?		;   driver
  BPBPTROFFS	DW	?		; pointer to BPB "array"
  BPBPTRSEG	DW	?
  DRIVENUM	DB	?		; number of drive
ENDS
	;
	; Save ES, BX.
	;
INIT:
	PUSH	ES
	PUSH	BX
	;
	; ES:BX -> local sector buffer
	;
	MOV	AX,CS
	MOV	ES,AX
	MOV	BX,OFFSET LOCALBUFFER
	;
	; Read boot sector from ROM drive into buffer, jump if error.
	;
	CALL	INIT_READBOOT
	JC	INIT_ERROR
	;
	; Copy BPB.
	;
	PUSH	DS
	PUSH	DS
	PUSH	ES
	POP	DS
	POP	ES
	LEA	SI,[BX+0Bh]
	MOV	DI,OFFSET BPB
	MOV	CX,13h
	REP	MOVSB
	POP	DS
	;
	; Set request header fields.
  	;
	POP	BX
	POP	ES
	MOV	ES:[BX].UNITS,1
	MOV	ES:[BX].FREEMEMOFFS,OFFSET INIT
	MOV	ES:[BX].FREEMEMSEG,CS
	MOV	ES:[BX].BPBPTROFFS,OFFSET BPBARRAY
	MOV	ES:[BX].BPBPTRSEG,CS
	;
	; Save the drive number in the BIOS data segment.
	;
	PUSH	DS
	MOV	AX,40h
	MOV	DS,AX
	MOV	AL,ES:[BX].DRIVENUM
	INC	AL
	MOV	[0C2h],AL
	POP	DS
	;
	; Display drive letter and exit.
	;
	ADD	AL,'A'-1
	MOV	DRIVELETTER,AL
	MOV	DX,OFFSET INITMSG
	MOV	AH,9
	INT	21h
	JMP	SUCCESS
	;
	; Error:  can't read from drive.  Restore ES, BX.
	;
INIT_ERROR:
	POP	BX
	POP	ES
	;
	; Display error message.
	;
	MOV	DX,OFFSET INITERRMSG
	MOV	AH,9
	INT	21h
	;
	; Set request header fields to abort installation and return.
	;
	MOV	ES:[BX].UNITS,0
	MOV	ES:[BX].FREEMEMOFFS,OFFSET DEVICEHEADER
	MOV	ES:[BX].FREEMEMSEG,CS
	JMP	SUCCESS
	;
	; Subroutine, reads the ROM drive boot sector into the buffer addressed
	; by ES:BX.  Sets carry if error.
	;
INIT_READBOOT:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	MOV	AX,0201h
	MOV	CX,1
	MOV	DX,8
	INT	13h
	POP	DX
	POP	CX
	POP	AX
	RET

;
; Initialization data.
;
INITMSG		DB	0Dh,0Ah,"ROM disk driver installed, drive "
DRIVELETTER	DB	0
		DB	":",0Dh,0Ah,"$"
INITERRMSG	DB	0Dh,0Ah,"Can't find ROM drive, driver not installed."
		DB	0Dh,0Ah,"$"
LOCALBUFFER	LABEL	BYTE
