/*
    Modstuf.i

    This is the second part of the C "half" of the player module.  Here we 
    do file I/O, which is easier to do in C than assembler, and not that 
    time-critical.  This file has been broken into two parts to get around 
    the editor's 64k limit.
*/

/*************************************************************************/
/*                                                                       */
/* iss3m() routine.  Returns 1 if the file is an .s3m module, 0 if not   */
/* (or if there is a file error).                                        */
/*                                                                       */
/*************************************************************************/

static int iss3m( FILE *modfile )
{
    char filetag[4];        /* filetype tag */
    unsigned version;       /* version of creator program */

    /* Seek to the file tag. */
    if ( fseek( modfile, 0x2cL, SEEK_SET ) )
        return( 0 );

    /* Read it in. */
    if ( fread( filetag, sizeof( char ), 4, modfile ) != 4 )
        return( 0 );

    /* It needs to be "SCRM" for this to be .s3m. */
    if ( strncmp( filetag, "SCRM", 4 ) )
        return( 0 );

    /* Seek to the filetype character. */
    if ( fseek( modfile, 0x1dL, SEEK_SET ) )
        return( 0 );

    /* Read in the filetype character.  It needs to be 16. */
    if ( fgetc( modfile ) != 16 )
        return( 0 );

    /* Seek to the creator program code. */
    if ( fseek( modfile, 0x28L, SEEK_SET ) )
        return( 0 );

    /* Read in the creator version. */
    if ( fread( &version, sizeof( unsigned ), 1, modfile ) != 1 )
        return( 0 );

    /* The creator version needs to be 1300h or later (ScreamTracker 3.00 
       or later). */
    if ( version < 0x1300 )
        return( 0 );

    /* .s3m file confirmed. */
    return( 1 );
}


/*************************************************************************/
/*                                                                       */
/* s3m2modeff() routine.  This function converts the command and         */
/* infobyte from a ScreamTracker 3 file to internal format.  Returns     */
/* nothing.                                                              */
/*                                                                       */
/*************************************************************************/

static void s3m2modeff( struct noterec *note )
{
        /* Effect translation table, converts an .s3m effect number 0-26 to 
           an internal effect number 0-28.  -1 means no effect (invalid or 
           unimplemented .s3m command). */
    static signed char effxlat[] = {
        -1, 16, 11, 13, 17, 18, 19,  3,  4, 20,
        21, 22, 23, -1, -1,  9, -1, 24,  7, 28,
        25, 26, 27, -1, -1, -1, -1
    };
    signed char neweff;         /* new effect number */

    /* If the effect number is out of range, the effect is invalid.  Set 
       the note to no effect. */
    if ( note->noteeffnum > 26 )
    {
        note->noteeffnum = note->noteeffparm = 0;
        return;
    }

    /* Convert the major effect number.  If invalid or unimplemented, set 
       the note to no effect. */
    if ( (neweff = effxlat[note->noteeffnum]) == -1 )
    {
        note->noteeffnum = note->noteeffparm = 0;
        return;
    }

    /* That's all we have to do. */
    note->noteeffnum = neweff;
}


/*************************************************************************/
/*                                                                       */
/* reads3mleft() routine.  This function reads in a mono .s3m sample, or */
/* the left channel of a stereo .s3m sample.  The sample is converted as */
/* necessary to *unsigned* 8-bit during loading.  Assumes that the file  */
/* pointer is at the beginning of the sample on entry.  Returns nothing; */
/* in case of file errors, reading simply stops.                         */
/*     bflags is a bitmap indicating the sample type:  bit 1 = stereo,   */
/* bit 2 = 16-bit, bit 3 = unsigned.                                     */
/*                                                                       */
/*************************************************************************/

static void reads3mleft( FILE *modfile, struct samprec *samprecord,
    char bflags )
{
    unsigned dseg;          /* default data segment value */
    unsigned wordsleft;     /* number of words remaining to read */
    unsigned thistime;      /* number to read this time */
    unsigned long linaddr;  /* linear address in sound data buffer */
    unsigned insegment;     /* words in 64k segment of sound buffer */

    /* Get DS register value. */
    _asm { mov dseg,ds }

    /* Two main cases:  sample in EMS, and sample in conventional RAM.  
       First we handle the sample in EMS RAM. */
    if ( samprecord->sampemsflag )
    {
        /* Read sound data into expanded RAM buffer, starting with the 
           first 64k. */
        wordsleft = samprecord->samplen;
        mapems1st( samprecord->sampemshandle );
        insegment = 0;
        while ( wordsleft && insegment < 32768 )
        {
            /* If 16-bit, read multiple of 4 bytes. */
            if ( bflags & 4 )
            {
                thistime =
                    (wordsleft > IOBUFSIZE/4 ? IOBUFSIZE/4 : wordsleft);
                if ( 32768 - insegment < thistime )
                    thistime = 32768 - insegment;
                if ( fread( iobuf, 4, thistime, modfile ) != thistime )
                    wordsleft = thistime;

                /* Convert to 8-bit. */
                _asm {
                    push si
                    push di
                    push es
                    pushf
                    mov ax,ds
                    mov es,ax
                    lea si,byte ptr iobuf
                    mov di,si
                    mov cx,thistime
                    shl cx,1
                    cld
                reads3mleft_loop16a:
                    lodsw
                    mov al,ah
                    stosb
                    loop reads3mleft_loop16a
                    popf
                    pop es
                    pop di
                    pop si
                }
            }

            /* If 8-bit, read multiple of 2 bytes. */
            else
            {
                thistime =
                    (wordsleft > IOBUFSIZE/2 ? IOBUFSIZE/2 : wordsleft);
                if ( 32768 - insegment < thistime )
                    thistime = 32768 - insegment;
                if ( fread( iobuf, 2, thistime, modfile ) != thistime )
                    wordsleft = thistime;
            }

            /* If signed, convert to unsigned. */
            if ( !(bflags & 8) )
            {
                _asm {
                    push si
                    push di
                    push es
                    pushf
                    mov ax,ds
                    mov es,ax
                    lea si,byte ptr iobuf
                    mov di,si
                    mov cx,thistime
                    shl cx,1
                    cld
                reads3mleft_loopsa:
                    lodsb
                    add al,80h
                    stosb
                    loop reads3mleft_loopsa
                    popf
                    pop es
                    pop di
                    pop si
                }
            }

            /* Copy the data into the sample buffer. */
            movedata( dseg, (unsigned) iobuf,
                emsframeseg, insegment << 1, thistime << 1 );
            insegment += thistime;
            wordsleft -= thistime;
        }

        /* If 64k or more in the sample, map in the second 64k and read in 
           the rest. */
        if ( wordsleft )
        {
            mapems2nd( samprecord->sampemshandle );
            insegment = 0;
            while ( wordsleft )
            {
                /* If 16-bit, read multiple of 4 bytes. */
                if ( bflags & 4 )
                {
                    thistime =
                        (wordsleft > IOBUFSIZE/4 ? IOBUFSIZE/4 : wordsleft);
                    if ( fread( iobuf, 4, thistime, modfile ) != thistime )
                        wordsleft = thistime;

                    /* Convert to 8-bit. */
                    _asm {
                        push si
                        push di
                        push es
                        pushf
                        mov ax,ds
                        mov es,ax
                        lea si,byte ptr iobuf
                        mov di,si
                        mov cx,thistime
                        shl cx,1
                        cld
                    reads3mleft_loop16b:
                        lodsw
                        mov al,ah
                        stosb
                        loop reads3mleft_loop16b
                        popf
                        pop es
                        pop di
                        pop si
                    }
                }

                /* If 8-bit, read multiple of 2 bytes. */
                else
                {
                    thistime =
                        (wordsleft > IOBUFSIZE/2 ? IOBUFSIZE/2 : wordsleft);
                    if ( fread( iobuf, 2, thistime, modfile ) != thistime )
                        wordsleft = thistime;
                }

                /* If signed, convert to unsigned. */
                if ( !(bflags & 8) )
                {
                    _asm {
                        push si
                        push di
                        push es
                        pushf
                        mov ax,ds
                        mov es,ax
                        lea si,byte ptr iobuf
                        mov di,si
                        mov cx,thistime
                        shl cx,1
                        cld
                    reads3mleft_loopsb:
                        lodsb
                        add al,80h
                        stosb
                        loop reads3mleft_loopsb
                        popf
                        pop es
                        pop di
                        pop si
                    }
                }

                /* Copy the data into the sample buffer. */
                movedata( dseg, (unsigned) iobuf,
                    emsframeseg, insegment << 1, thistime << 1 );
                insegment += thistime;
                wordsleft -= thistime;
            } /* end while ( wordsleft ) */
        } /* end if ( wordsleft ) */
    } /* end if ( samprecord->sampemsflag ) */

    /* Second main case:  sample in conventional RAM. */
    else
    {
        /* Read sound data into conventional RAM buffer. */
        wordsleft = samprecord->samplen;
        linaddr = (unsigned long) samprecord->sampseg << 4;
        while ( wordsleft )
        {
            /* If 16-bit, read multiple of 4 bytes. */
            if ( bflags & 4 )
            {
                thistime =
                    (wordsleft > IOBUFSIZE/4 ? IOBUFSIZE/4 : wordsleft);
                if ( fread( iobuf, 4, thistime, modfile ) != thistime )
                    wordsleft = thistime;

                /* Convert to 8-bit. */
                _asm {
                    push si
                    push di
                    push es
                    pushf
                    mov ax,ds
                    mov es,ax
                    lea si,byte ptr iobuf
                    mov di,si
                    mov cx,thistime
                    shl cx,1
                    cld
                reads3mleft_loop16c:
                    lodsw
                    mov al,ah
                    stosb
                    loop reads3mleft_loop16c
                    popf
                    pop es
                    pop di
                    pop si
                }
            }

            /* If 8-bit, read multiple of 2 bytes. */
            else
            {
                thistime =
                    (wordsleft > IOBUFSIZE/2 ? IOBUFSIZE/2 : wordsleft);
                if ( fread( iobuf, 2, thistime, modfile ) != thistime )
                    wordsleft = thistime;
            }

            /* If signed, convert to unsigned. */
            if ( !(bflags & 8) )
            {
                _asm {
                    push si
                    push di
                    push es
                    pushf
                    mov ax,ds
                    mov es,ax
                    lea si,byte ptr iobuf
                    mov di,si
                    mov cx,thistime
                    shl cx,1
                    cld
                reads3mleft_loopsc:
                    lodsb
                    add al,80h
                    stosb
                    loop reads3mleft_loopsc
                    popf
                    pop es
                    pop di
                    pop si
                }
            }

            /* Copy the data into the sample buffer. */
            movedata( dseg, (unsigned) iobuf,
                linaddr >> 4, linaddr & 0xf, thistime << 1 );
            linaddr += thistime << 1;
            wordsleft -= thistime;
        }
    }
}


/*************************************************************************/
/*                                                                       */
/* reads3mright() routine.  This function reads in the right channel of  */
/* a stereo .s3m sample, mixing it into the left channel data.  It is    */
/* assumed that the left channel has already been read in in *unsigned*  */
/* 8-bit format.  It is also assumed that the file pointer is at the     */
/* beginning of the right channel data on entry.  Returns nothing (file  */
/* errors simply stop the loading process).                              */
/*     bflags is a bitmap indicating the sample type:  bit 1 = stereo,   */
/* bit 2 = 16-bit, bit 3 = unsigned.                                     */
/*                                                                       */
/*************************************************************************/

static void reads3mright( FILE *modfile, struct samprec *samprecord,
    char bflags )
{
    unsigned wordsleft;     /* number of words remaining to read */
    unsigned thistime;      /* number to read this time */
    unsigned long linaddr;  /* linear address in sound data buffer */
    unsigned insegment;     /* words in 64k segment of sound buffer */

    /* Two main cases:  sample in EMS, and sample in conventional RAM.  
       First we handle the sample in EMS RAM. */
    if ( samprecord->sampemsflag )
    {
        /* Read sound data into expanded RAM buffer, starting with the 
           first 64k. */
        wordsleft = samprecord->samplen;
        mapems1st( samprecord->sampemshandle );
        insegment = 0;
        while ( wordsleft && insegment < 32768 )
        {
            /* If 16-bit, read multiple of 4 bytes. */
            if ( bflags & 4 )
            {
                thistime =
                    (wordsleft > IOBUFSIZE/4 ? IOBUFSIZE/4 : wordsleft);
                if ( 32768 - insegment < thistime )
                    thistime = 32768 - insegment;
                if ( fread( iobuf, 4, thistime, modfile ) != thistime )
                    wordsleft = thistime;

                /* Convert to 8-bit. */
                _asm {
                    push si
                    push di
                    push es
                    pushf
                    mov ax,ds
                    mov es,ax
                    lea si,byte ptr iobuf
                    mov di,si
                    mov cx,thistime
                    shl cx,1
                    cld
                reads3mright_loop16a:
                    lodsw
                    mov al,ah
                    stosb
                    loop reads3mright_loop16a
                    popf
                    pop es
                    pop di
                    pop si
                }
            }

            /* If 8-bit, read multiple of 2 bytes. */
            else
            {
                thistime =
                    (wordsleft > IOBUFSIZE/2 ? IOBUFSIZE/2 : wordsleft);
                if ( 32768 - insegment < thistime )
                    thistime = 32768 - insegment;
                if ( fread( iobuf, 2, thistime, modfile ) != thistime )
                    wordsleft = thistime;
            }

            /* If signed, convert to unsigned. */
            if ( !(bflags & 8) )
            {
                _asm {
                    push si
                    push di
                    push es
                    pushf
                    mov ax,ds
                    mov es,ax
                    lea si,byte ptr iobuf
                    mov di,si
                    mov cx,thistime
                    shl cx,1
                    cld
                reads3mright_loopsa:
                    lodsb
                    add al,80h
                    stosb
                    loop reads3mright_loopsa
                    popf
                    pop es
                    pop di
                    pop si
                }
            }

            /* Mix the data into the sample buffer. */
            _asm {
                push si
                push di
                push es
                pushf
                lea si,byte ptr iobuf
                mov di,insegment
                shl di,1
                mov cx,thistime
                shl cx,1
                mov es,emsframeseg
                cld
            reads3mright_loopma:
                lodsb
                add al,es:[di]
                rcr al,1
                stosb
                loop reads3mright_loopma
                popf
                pop es
                pop di
                pop si
            }

            /* Adjust counts. */
            insegment += thistime;
            wordsleft -= thistime;
        }

        /* If 64k or more in the sample, map in the second 64k and read in 
           the rest. */
        if ( wordsleft )
        {
            mapems2nd( samprecord->sampemshandle );
            insegment = 0;
            while ( wordsleft )
            {
                /* If 16-bit, read multiple of 4 bytes. */
                if ( bflags & 4 )
                {
                    thistime =
                        (wordsleft > IOBUFSIZE/4 ? IOBUFSIZE/4 : wordsleft);
                    if ( fread( iobuf, 4, thistime, modfile ) != thistime )
                        wordsleft = thistime;

                    /* Convert to 8-bit. */
                    _asm {
                        push si
                        push di
                        push es
                        pushf
                        mov ax,ds
                        mov es,ax
                        lea si,byte ptr iobuf
                        mov di,si
                        mov cx,thistime
                        shl cx,1
                        cld
                    reads3mright_loop16b:
                        lodsw
                        mov al,ah
                        stosb
                        loop reads3mright_loop16b
                        popf
                        pop es
                        pop di
                        pop si
                    }
                }

                /* If 8-bit, read multiple of 2 bytes. */
                else
                {
                    thistime =
                        (wordsleft > IOBUFSIZE/2 ? IOBUFSIZE/2 : wordsleft);
                    if ( fread( iobuf, 2, thistime, modfile ) != thistime )
                        wordsleft = thistime;
                }

                /* If signed, convert to unsigned. */
                if ( !(bflags & 8) )
                {
                    _asm {
                        push si
                        push di
                        push es
                        pushf
                        mov ax,ds
                        mov es,ax
                        lea si,byte ptr iobuf
                        mov di,si
                        mov cx,thistime
                        shl cx,1
                        cld
                    reads3mright_loopsb:
                        lodsb
                        add al,80h
                        stosb
                        loop reads3mright_loopsb
                        popf
                        pop es
                        pop di
                        pop si
                    }
                }

                /* Mix the data into the sample buffer. */
                _asm {
                    push si
                    push di
                    push es
                    pushf
                    lea si,byte ptr iobuf
                    mov di,insegment
                    shl di,1
                    mov cx,thistime
                    shl cx,1
                    mov es,emsframeseg
                    cld
                reads3mright_loopmb:
                    lodsb
                    add al,es:[di]
                    rcr al,1
                    stosb
                    loop reads3mright_loopmb
                    popf
                    pop es
                    pop di
                    pop si
                }

                /* Adjust counts. */
                insegment += thistime;
                wordsleft -= thistime;
            } /* end while ( wordsleft ) */
        } /* end if ( wordsleft ) */
    } /* end if ( samprecord->sampemsflag ) */

    /* Second main case:  sample in conventional RAM. */
    else
    {
        /* Read sound data into conventional RAM buffer. */
        wordsleft = samprecord->samplen;
        linaddr = (unsigned long) samprecord->sampseg << 4;
        while ( wordsleft )
        {
            /* If 16-bit, read multiple of 4 bytes. */
            if ( bflags & 4 )
            {
                thistime =
                    (wordsleft > IOBUFSIZE/4 ? IOBUFSIZE/4 : wordsleft);
                if ( fread( iobuf, 4, thistime, modfile ) != thistime )
                    wordsleft = thistime;

                /* Convert to 8-bit. */
                _asm {
                    push si
                    push di
                    push es
                    pushf
                    mov ax,ds
                    mov es,ax
                    lea si,byte ptr iobuf
                    mov di,si
                    mov cx,thistime
                    shl cx,1
                    cld
                reads3mright_loop16c:
                    lodsw
                    mov al,ah
                    stosb
                    loop reads3mright_loop16c
                    popf
                    pop es
                    pop di
                    pop si
                }
            }

            /* If 8-bit, read multiple of 2 bytes. */
            else
            {
                thistime =
                    (wordsleft > IOBUFSIZE/2 ? IOBUFSIZE/2 : wordsleft);
                if ( fread( iobuf, 2, thistime, modfile ) != thistime )
                    wordsleft = thistime;
            }

            /* If signed, convert to unsigned. */
            if ( !(bflags & 8) )
            {
                _asm {
                    push si
                    push di
                    push es
                    pushf
                    mov ax,ds
                    mov es,ax
                    lea si,byte ptr iobuf
                    mov di,si
                    mov cx,thistime
                    shl cx,1
                    cld
                reads3mright_loopsc:
                    lodsb
                    add al,80h
                    stosb
                    loop reads3mright_loopsc
                    popf
                    pop es
                    pop di
                    pop si
                }
            }

            /* Mix the data into the sample buffer. */
            _asm {
                push si
                push di
                push es
                pushf
                lea si,byte ptr iobuf
                mov ax,word ptr linaddr
                mov di,ax
                and di,0fh
                and ax,0fff0h
                add ax,word ptr linaddr+2
                ror ax,1
                ror ax,1
                ror ax,1
                ror ax,1
                mov es,ax
                mov cx,thistime
                shl cx,1
                cld
            reads3mright_loopmc:
                lodsb
                add al,es:[di]
                rcr al,1
                stosb
                loop reads3mright_loopmc
                popf
                pop es
                pop di
                pop si
            }

            /* Adjust counts. */
            linaddr += thistime << 1;
            wordsleft -= thistime;
        }
    }
}


/*************************************************************************/
/*                                                                       */
/* tosigned() routine.  Converts an unsigned sample to signed.           */
/*                                                                       */
/*************************************************************************/

static void tosigned( struct samprec *samprecord )
{
    unsigned infirst;       /* number of samples in first 64k */
    unsigned insecond;      /* number of samples in second 64k */
    unsigned firstseg;      /* segment address of first 64k */
    unsigned secondseg;     /* segment address of second 64k */

    /* Determine how much is in the first and second 64k parts, and where 
       they are. */
    if ( samprecord->samplen >= 32768 )
    {
        infirst = 0; /* = 65536 */
        insecond = (samprecord->samplen - 32768) << 1;
    }
    else
        infirst = samprecord->samplen << 1;
    if ( samprecord->sampemsflag )
        firstseg = secondseg = emsframeseg;
    else
    {
        firstseg = samprecord->sampseg;
        secondseg = firstseg + 0x1000;
    }

    /* If the sample is in EMS, map in the first part. */
    if ( samprecord->sampemsflag )
        mapems1st( samprecord->sampemshandle );

    /* Convert the first part. */
    _asm {
        push si
        push di
        push es
        push ds
        pushf
        xor si,si
        mov di,si
        mov cx,infirst
        mov ax,firstseg
        mov ds,ax
        mov es,ax
        cld
    tosigned_loopua:
        lodsb
        sub al,80h
        stosb
        loop tosigned_loopua
        popf
        pop ds
        pop es
        pop di
        pop si
    }

    /* If 64k or more in the sample, convert the second part. */
    if ( samprecord->samplen > 32768 )
    {
        /* If the sample is in EMS, map in the second part. */
        if ( samprecord->sampemsflag )
            mapems2nd( samprecord->sampemshandle );

        /* Convert the second half. */
        _asm {
            push si
            push di
            push es
            push ds
            pushf
            xor si,si
            mov di,si
            mov cx,insecond
            mov ax,secondseg
            mov ds,ax
            mov es,ax
            cld
        tosigned_loopub:
            lodsb
            sub al,80h
            stosb
            loop tosigned_loopub
            popf
            pop ds
            pop es
            pop di
            pop si
        }
    }
}


/*************************************************************************/
/*                                                                       */
/* loads3mfile() routine.  This function takes an open music module file */
/* in ScreamTracker 3 format, loads it into memory, and closes the file. */
/* Assumes that an .s3m has been detected (i.e., by iss3m()).  If        */
/* successful, this routine returns 0.  If unsuccessful, it returns -1   */
/* and sets errmsg to an error string (max 80 characters, 1 line, null-  */
/* terminated) to be displayed to the user.                              */
/*                                                                       */
/*************************************************************************/

static int loads3mfile( FILE *modfile, char **errmsg )
{
    int sampsunsigned;          /* 1 if samples are unsigned */
    unsigned char chanmap[32];  /* channel settings */
        /* Channel translation table, used to convert .s3m channel numbers 
           to mod channel numbers. */
    static unsigned char chanxlat[] = {
        0, 3, 4, 7, 8, 11, 12, 15, 1, 2, 5, 6, 9, 10, 13, 14
    };
    int i, j;                   /* for looping */
    int nright, nleft;          /* number of right and left channels */
    char chanused[16];          /* 1 if channel number used */
        /* Pattern table from the file (use global I/O buffer). */
    unsigned char *filpattable = (unsigned char *) iobuf;
    unsigned instptrs[99];      /* parapointers to instrument records */
    unsigned patptrs[100];      /* parapointers to pattern records */
    int patnum;                 /* current pattern number */
    int found;                  /* 1 if pattern is actually played */
    int bytesleft;              /* length of remaining packed pattern data */
        /* Pattern array pointer. */
    struct noterec far *noteptr;
    struct noterec tempnote;    /* temporary note record */
    int row;                    /* row in the pattern */
    int chan;                   /* channel in the row */
    int firstbyte;              /* first byte of compressed note data */
    int octavenote;             /* octave and note */
        /* Conversion table for discarding unused channels. */
    unsigned char chanconvert[16];
    int newnchannels;           /* number of channels actually in use */
    int newbytesperrow;         /* number of bytes per row in new patterns */
    int newpatbufsize;          /* size of new pattern in paragraphs */
    int sampnum;                /* number of sample */
        /* Sample frequency structures, used to optimize loading. */
    struct freqstruct sampfreqs[100];
    struct s3msamp samprecord;  /* sample record */
    unsigned long sampstart;    /* start of sample data in file */
    unsigned long sampstart2;   /* start of right channel data, if stereo */

    /* Initialize all samples to be copies of sample 0, the silence sample, 
       and mark all pattern segments unallocated. */
    for ( i=1; i<100; i++ )
        samplerecs[i] = samplerecs[0];
    for ( i=0; i<256; i++ )
        patternsegs[i] = 0;

   /* Read in the global volume. */
    *errmsg = seekfailmsg;
    if ( fseek( modfile, 0x30L, SEEK_SET ) )
        return( errclose( modfile ) );
    *errmsg = readfailmsg;
    globalvoldefl = globalvol = fgetc( modfile );
    if ( globalvol > 64 )
        globalvoldefl = globalvol = 64;

    /* Read in the initial speed. */
    if ( (ticksdefl = ticksperrow = fgetc( modfile )) == EOF )
        return( errclose( modfile ) );
    if ( !ticksperrow )
        ticksdefl = ticksperrow = 6;

    /* Read in the initial tempo. */
    bpmdefl = beatspermin = fgetc( modfile );
    if ( !beatspermin )
        bpmdefl = beatspermin = 125;

    /* We ignore the master volume, "ultra click removal," channel pan 
       settings, and various flags.  We also ignore special custom data and 
       extended header data.  Determine whether samples are signed or 
       unsigned. */
    *errmsg = seekfailmsg;
    if ( fseek( modfile, 0x2aL, SEEK_SET ) )
        return( errclose( modfile ) );
    *errmsg = readfailmsg;
    if ( fread( &sampsunsigned, sizeof( int ), 1, modfile ) != 1 )
        return( errclose( modfile ) );
    *errmsg = badsamtypemsg;
    if ( sampsunsigned <= 0 || sampsunsigned > 2 )
        return( errclose( modfile ) );
    sampsunsigned--;

    /* Read in the number of orders (called "pattern table size" for mod 
       compatibility). */
    *errmsg = seekfailmsg;
    if ( fseek( modfile, 0x20L, SEEK_SET ) )
        return( errclose( modfile ) );
    *errmsg = readfailmsg;
    if ( fread( &pattablesize, sizeof( unsigned ), 1, modfile ) != 1 )
        return( errclose( modfile ) );
    *errmsg = badnordersmsg;
    if ( !pattablesize || pattablesize > 255 || pattablesize & 1 )
        return( errclose( modfile ) );

    /* Read in the number of instruments. */
    *errmsg = readfailmsg;
    if ( fread( &ninstruments, sizeof( unsigned ), 1, modfile ) != 1 )
        return( errclose( modfile ) );
    *errmsg = badninstrsmsg;
    if ( !ninstruments || ninstruments > 99 )
        return( errclose( modfile ) );

    /* Read in the number of patterns. */
    *errmsg = readfailmsg;
    if ( fread( &nfilepatterns, sizeof( unsigned ), 1, modfile ) != 1 )
        return( errclose( modfile ) );
    *errmsg = badnpatsmsg;
    if ( !nfilepatterns || nfilepatterns > 100 )
        return( errclose( modfile ) );

    /* Read in the channel settings. */
    *errmsg = seekfailmsg;
    if ( fseek( modfile, 0x40L, SEEK_SET ) )
        return( errclose( modfile ) );
    *errmsg = readfailmsg;
    if ( fread( chanmap, sizeof( unsigned char ), 32, modfile ) != 32 )
        return( errclose( modfile ) );
    *errmsg = badchansetmsg;
    for ( i=0; i<32; i++ )
    {
        if ( chanmap[i] < 128 && chanmap[i] > 31 )
            return( errclose( modfile ) );
        if ( chanmap[i] < 32 )
        {
            for ( j=0; j<i; j++ )
                if ( chanmap[j] == chanmap[i] )
                    chanmap[i] = 255;
        }
    }

    /* Determine how many right and left channels there are, and 
       consequently how many channels will be in the pattern we convert to.   
       Make it so that there are no "holes" in the channel list. */
    nright = nleft = 0;
    for ( i=0; i<16; i++ )
        chanused[i] = 0;
    for ( i=0; i<32; i++ )
        if ( chanmap[i] < 16 )
            chanused[chanmap[i]] = 1;
    for ( i=0; i<32; i++ )
    {
        if ( chanmap[i] < 8 )
        {
            for ( j = 0; j < chanmap[i]; j++ )
                if ( !chanused[j] )
                {
                    chanused[j] = 1;
                    chanused[chanmap[i]] = 0;
                    chanmap[i] = j;
                    break;
                }
            nleft++;
        }
        else if ( chanmap[i] < 16 )
        {
            for ( j = 8; j < chanmap[i]; j++ )
                if ( !chanused[j] )
                {
                    chanused[j] = 1;
                    chanused[chanmap[i]] = 0;
                    chanmap[i] = j;
                    break;
                }
            nright++;
        }
    }
    *errmsg = alladlibmsg;
    if ( !nright && !nleft )
        return( errclose( modfile ) );
    nchannels = nleft << 1;
    if ( (nright << 1) > nchannels )
        nchannels = nright << 1;
    if ( nchannels < 4 )
        nchannels = 4;

    /* Set the number of bytes per row in the (internal) pattern table and 
       the size of a pattern buffer in paragraphs.  .s3m files do not use 
       the song end jump position. */
    bytesperrow = nchannels * sizeof( struct noterec );
    patbufsize = 4*bytesperrow;
    songendjump = 256;

    /* Convert channel settings so that if we mix to stereo, channels will 
       be on the correct side.  Set unused or Adlib channels to 255 (we 
       won't play Adlib stuff). */
    for ( i=0; i<32; i++ )
    {
        if ( chanmap[i] >= 16 )
            chanmap[i] = 255;
        else
            chanmap[i] = chanxlat[chanmap[i]];
    }

    /* Read in the pattern table. */
    *errmsg = readfailmsg;
    if ( fread( filpattable, sizeof( unsigned char ), pattablesize, modfile )
            != pattablesize )
        return( errclose( modfile ) );
    for ( i=0; i<pattablesize; i++ )
    {
        /* If the pattern number is invalid, just mark it to be skipped. */
        if ( filpattable[i] >= nfilepatterns && filpattable[i] < 254 )
            patterntable[i] = 254;
        patterntable[i] = filpattable[i];
    }

    /* Read in the parapointers to the instrument records and the 
       parapointers to the patterns. */
    *errmsg = readfailmsg;
    if ( fread( instptrs, sizeof( unsigned ), ninstruments, modfile )
            != ninstruments )
        return( errclose( modfile ) );
    if ( fread( patptrs, sizeof( unsigned ), nfilepatterns, modfile )
            != nfilepatterns )
        return( errclose( modfile ) );

    /* Read in the patterns. */
    for ( patnum = 0; patnum < nfilepatterns; patnum++ )
    {
        /* If the "save memory" flag is set, check to see if the pattern is 
           actually used.  If not, don't load it. */
        if ( savemem )
        {
            found = 0;
            for ( i=0; i<pattablesize; i++ )
                if ( patterntable[i] == patnum )
                {
                    found = 1;
                    break;
                }
            if ( !found )
                continue;
        }

        /* Allocate memory for the pattern. */
        *errmsg = nomemmsg;
        if ( !(patternsegs[patnum] = getblock( patbufsize )) )
            return( errclose( modfile ) );

        /* Get a pointer to the pattern data and initialize the pattern to 
           all empty notes. */
        FP_SEG( noteptr ) = patternsegs[patnum];
        FP_OFF( noteptr ) = 0;
        tempnote.notesampnum = tempnote.noteperiod = tempnote.noteeffparm =
            tempnote.noteeffnum = 0;
        tempnote.notevol = 255;
        for ( row = 0; row < 64; row++ )
            for ( chan = 0; chan < nchannels; chan++ )
                noteptr[row*nchannels + chan] = tempnote;

        /* Seek to the pattern data. */
        *errmsg = seekfailmsg;
        if ( fseek( modfile, (long) patptrs[patnum] << 4, SEEK_SET ) )
            return( errclose( modfile ) );

        /* Read in the length of the packed pattern data. */
        *errmsg = readfailmsg;
        if ( fread( &bytesleft, sizeof( unsigned ), 1, modfile ) != 1 )
            return( errclose( modfile ) );

        /* Read in and convert the pattern data. */
        row = 0;
        while ( bytesleft > 0 )
        {
            /* Read in the first byte of the compressed note. */
            if ( (firstbyte = fgetc( modfile )) == EOF )
                return( errclose( modfile ) );
            --bytesleft;

            /* New row if the first byte is 0.  New pattern if the new row 
               is 64. */
            if ( !firstbyte )
            {
                if ( ++row >= 64 )
                    break;
                continue;
            }

            /* The 5 low-order bits are the channel number.  Note that we 
               might not need this data if the channel is disabled, but we 
               still have to read it from the file to get to the next 
               channel. */
            chan = firstbyte & 31;

            /* If bit 5 is set, read in a note and instrument number. */
            if ( firstbyte & 32 )
            {
                if ( (octavenote = fgetc( modfile )) == EOF )
                    return( errclose( modfile ) );
                tempnote.noteperiod = note2period( octavenote );
                tempnote.notesampnum = fgetc( modfile );
                bytesleft -= 2;
            }
            else
                tempnote.noteperiod = tempnote.notesampnum = 0;

            /* If bit 6 is set, read in a volume. */
            if ( firstbyte & 64 )
            {
                tempnote.notevol = fgetc( modfile );
                --bytesleft;
            }
            else
                tempnote.notevol = 255;

            /* If bit 7 is set, read in a command and infobyte. */
            if ( firstbyte & 128 )
            {
                tempnote.noteeffnum = fgetc( modfile );
                tempnote.noteeffparm = fgetc( modfile );
                s3m2modeff( &tempnote );
                bytesleft -= 2;
            }
            else
                tempnote.noteeffparm = tempnote.noteeffnum = 0;

            /* Convert the .s3m channel number to an internal channel 
               number.  If this is an unused (or Adlib) channel, skip it. 
               */
            if ( (chan = chanmap[chan]) > 16 )
                continue;

            /* Save the note. */
            noteptr[row*nchannels + chan] = tempnote;
        }
    }

    /* If "save memory" is enabled, scan the patterns for unused channels, 
       and delete them if possible. */
    if ( savemem )
    {
        /* Loop over the channels. */
        for ( chan = 0; chan < nchannels; chan++ )
        {
            /* Initially mark the channel unused. */
            chanused[chan] = 0;

            /* Loop over the patterns that are actually played. */
            for ( i=0; i<pattablesize && !chanused[chan]; i++ )
            {
                /* If the pattern table entry is 254 or 255, skip it - 254 
                   means skip the entry, 255 means stop the song. */
                if ( patterntable[i] >= 254 )
                    continue;

                /* Get a C pointer to the pattern buffer. */
                FP_SEG( noteptr ) = patternsegs[patterntable[i]];
                FP_OFF( noteptr ) = chan * sizeof( struct noterec );

                /* Loop over the notes in the pattern. */
                for ( j=0; j<64 && !chanused[chan]; j++ )
                {
                    /* If the sample number, period, effect number, or 
                       effect parameter is nonzero, the channel is actually 
                       in use. */
                    if ( noteptr->notesampnum || noteptr->noteperiod
                            || noteptr->noteeffparm || noteptr->noteeffnum )
                        chanused[chan] = 1;
                    noteptr += nchannels;
                }
            }
        }

        /* Loop over the channels again, checking to see how many right and 
           left channels there really are. */
        nright = nleft = 0;
        for ( chan = 0; chan < nchannels; chan++ )
        {
            if ( chanleft[chan] )
            {
                if ( chanused[chan] )
                {
                    chanconvert[chan] = chan;
                    nleft++;
                }
                else
                    chanconvert[chan] = 255;
            }
            else
            {
                if ( chanused[chan] )
                {
                    chanconvert[chan] = chan;
                    nright++;
                }
                else
                    chanconvert[chan] = 255;
            }
        }

        /* Determine the number of used channels. */
        newnchannels = nleft << 1;
        if ( (nright << 1) > newnchannels )
            newnchannels = nright << 1;
        if ( newnchannels < 4 )
            newnchannels = 4;

        /* If the number of channels actually used is less than the number 
           given in the file header, get rid of the unused channels. */
        if ( newnchannels < nchannels )
        {
            /* Determine where the channels should go. */
            for ( chan = 0; chan < nchannels; chan++ )
                for ( i=0; i<newnchannels; i++ )
                    if ( chanleft[i] == chanleft[chan]
                        && chanconvert[i] == 255
                        && chanconvert[chan] != 255 )
                    {
                        chanconvert[i] = chanconvert[chan];
                        chanconvert[chan] = 255;
                        break;
                    }

            /* Determine the number of bytes per row in the compressed 
               pattern and the number of paragraphs of data in the 
               compressed pattern. */
            newbytesperrow = newnchannels * sizeof( struct noterec );
            newpatbufsize = 4*newbytesperrow;

            /* Loop over the patterns.  Note that we *must* go in the order 
               the patterns were loaded here. */
            for ( i=0; i<nfilepatterns; i++ )
            {
                /* If the pattern was not loaded, skip it. :) */
                if ( !patternsegs[i] )
                    continue;

                /* Get a C pointer to the pattern buffer. */
                FP_SEG( noteptr ) = patternsegs[i];
                FP_OFF( noteptr ) = 0;

                /* Compress unused channels out of the pattern. */
                tempnote.notesampnum = tempnote.noteperiod =
                    tempnote.noteeffparm = tempnote.noteeffnum = 0;
                tempnote.notevol = 255;
                for ( row = 0; row < 64; row++ )
                    for ( chan = 0; chan < newnchannels; chan++ )
                    {
                        if ( chanconvert[chan] == 255 )
                            noteptr[row*newnchannels + chan] = tempnote;
                        else
                            noteptr[row*newnchannels + chan] =
                                noteptr[row*nchannels + chanconvert[chan]];
                    }

                /* Release the (now) unused memory at the end of the 
                   pattern. */
                if ( freeblock( patternsegs[i] + newpatbufsize,
                        patbufsize - newpatbufsize ) )
                    panic( allocerrmsg, 3 );

                /* Move the block down in memory as far as possible. */
                patternsegs[i] = movedown( patternsegs[i], newpatbufsize );
            }

            /* Reset the number of channels, number of bytes per row in a 
               pattern, and number of paragraphs of memory required for a 
               pattern. */
            nchannels = newnchannels;
            bytesperrow = newbytesperrow;
            patbufsize = newpatbufsize;
        }
    }

    /* Construct the sample loading order - by default, the order in which 
       they appear in the file.  Set occurrence counts to 0. */
    for ( i=1; i<=ninstruments; i++ )
    {
        sampfreqs[i].sampnum = i;
        sampfreqs[i].freq = 0L;
    }

    /* If we're to optimize loading, scan the patterns to see which samples 
       are most commonly used, and adjust the loading order so that the 
       most common ones will be loaded first (hence be more likely to be 
       loaded into conventional RAM). */
    if ( optimize )
    {
        /* Loop over the channels. */
        for ( chan = 0; chan < nchannels; chan++ )
        {
            /* Initial sample on every channel is 0. */
            sampnum = 0;

            /* Loop over the patterns that are actually played. */
            for ( i=0; i<pattablesize; i++ )
            {
                /* If the pattern table entry is 254 or 255, skip it - 254 
                   means skip the entry, 255 means stop the song. */
                if ( patterntable[i] >= 254 )
                    continue;

                /* Get a C pointer to the pattern buffer. */
                FP_SEG( noteptr ) = patternsegs[patterntable[i]];
                FP_OFF( noteptr ) = chan * sizeof( struct noterec );

                /* Loop over the notes in the pattern, accumulating sample 
                   counts. */
                for ( j=0; j<64; j++ )
                {
                    if ( noteptr->notesampnum &&
                            noteptr->notesampnum <= ninstruments )
                        sampnum = noteptr->notesampnum;
                    sampfreqs[sampnum].freq++;
                    noteptr += nchannels;
                }
            }
        }

        /* Sort the occurrence structures in increasing order. */
        qsort( &sampfreqs[1], ninstruments, sizeof( struct freqstruct ),
            freqcompare );
    }

    /* Read in the samples. */
    for ( i=1; i<=ninstruments; i++ )
    {
        /* If loading optimization and "save memory" are both enabled, and 
           the sample is never used, skip loading it. */
        if ( optimize && savemem && !sampfreqs[i].freq )
            continue;

        /* Get the sample number from the order list. */
        sampnum = sampfreqs[i].sampnum;

        /* Seek to the instrument record. */
        *errmsg = seekfailmsg;
        if ( fseek( modfile, (long) instptrs[sampnum-1] << 4, SEEK_SET ) )
            return( errclose( modfile ) );

        /* Read in the instrument record. */
        *errmsg = readfailmsg;
        if ( fread( &samprecord, sizeof( struct s3msamp ), 1, modfile ) != 1 )
            return( errclose( modfile ) );

        /* The instrument record might be an invalid.  That might be the 
           case, for example, if the instrument record is only there to add 
           a comment to the .s3m file.  In that case, just go to the next 
           instrument (the instrument has already been set to the default 
           silence sample).  We also skip Adlib samples, samples with zero 
           length, samples with an out-of-range C4SPD value, and packed 
           (ADPCM) samples. */
        if ( strncmp( samprecord.tag, "SCRS", 4 ) || samprecord.samptype != 1
                || samprecord.samplen == 0 || samprecord.c4spd == 0
                || samprecord.c4spd > 65535 || samprecord.pack )
            continue;

        /* Get a pointer to the actual sample data and the start of the 
           right channel data, if stereo. */
        sampstart = (((unsigned long) samprecord.paratop << 16)
            + samprecord.paraptr) << 4;
        sampstart2 = sampstart + samprecord.samplen;

        /* Adjust the sample parameters so they make sense.  ScreamTracker 
           up to 3.20 does not actually support samples longer than 64000 
           bytes.  We support samples up to 128k-1.  If the sample is 
           looped, verify that the loop start and end are in range as well.  
           We need the parameters to be even as well since they need to be 
           converted to words. */
        if ( samprecord.bflags & 4 )
        {
            /* Sample is 16-bit, convert lengths to # 8-bit converted 
               samples. */
            samprecord.samplen >>= 1;
            samprecord.loopbeg >>= 1;
            samprecord.loopend >>= 1;
        }
        if ( samprecord.samplen > 131070L )
            samprecord.samplen = 131070L;
        samprecord.samplen &= ~1L;
        if ( samprecord.bflags & 1 )
        {
            /* Make the loop begin and end even. */
            samprecord.loopbeg &= ~1L;
            samprecord.loopend &= ~1L;

            /* If the start of the loop is past the end of the sample, make 
               the sample unlooped. */
            if ( samprecord.loopbeg > samprecord.samplen )
                samprecord.bflags &= ~1;

            /* If the end of the loop is past the end of the sample, set 
               the end of the loop to the end of the sample. */
            else if ( samprecord.loopend > samprecord.samplen )
                samprecord.loopend = samprecord.samplen;
        }
        if ( samprecord.bflags & 1 )
        {
            /* If the start of the loop is past the end of the loop, make 
               the sample unlooped. */
            if ( samprecord.loopbeg >= samprecord.loopend )
                samprecord.bflags &= ~1;

            /* If the loop is 2 bytes or less in length, make the sample 
               unlooped. */
            else if ( samprecord.loopend - samprecord.loopbeg <= 2L )
                samprecord.bflags &= ~1;

            /* If the sample extends past the end of the loop, cut it off 
               at the end of the loop. */
            else
                samprecord.samplen = samprecord.loopend;
        }

        /* If after those adjustments we wound up with a sample less than 4 
           bytes in length, skip it.  Otherwise, fill in the internal 
           sample structure. */
        if ( samprecord.samplen < 4L )
            continue;
        samplerecs[sampnum].samplen = samprecord.samplen >> 1;
        if ( samprecord.bflags & 1 )
        {
            samplerecs[sampnum].samplstart = samprecord.loopbeg >> 1;
            samplerecs[sampnum].sampllen =
                (samprecord.loopend - samprecord.loopbeg) >> 1;
        }
        else
            samplerecs[sampnum].samplstart = samplerecs[sampnum].sampllen = 0;
        samplerecs[sampnum].sampc4spd = samprecord.c4spd;
        if ( samprecord.vol > 64 )
            samplerecs[sampnum].sampvol = 64;
        else
            samplerecs[sampnum].sampvol = samprecord.vol;

        /* Add the "unsigned sample" flag to the other flags, for 
           convenience. */
        samprecord.bflags = (samprecord.bflags & 7) + (sampsunsigned << 3);

        /* Allocate a buffer for the sample.  First try to get conventional 
           RAM, then EMS RAM. */
        *errmsg = nomemmsg;
        if ( !(samplerecs[sampnum].sampseg
            = getblock( (samplerecs[sampnum].samplen + 7L) / 8 )) )
        {
            /* Didn't get conventional RAM, try EMS. */
            if ( !(samplerecs[sampnum].sampemshandle
                    = getemsblock( (samplerecs[sampnum].samplen + 7L) / 8 )) )
                return( errclose( modfile ) );

            /* We got some EMS RAM. */
            samplerecs[sampnum].sampseg = emsframeseg;
            samplerecs[sampnum].sampemsflag = 1;
        }

        /* Seek to the sample data. */
        *errmsg = seekfailmsg;
        if ( fseek( modfile, sampstart, SEEK_SET ) )
            return( errclose( modfile ) );

        /* Read in the sample data.  We ignore file errors here.  Some .s3m 
           files are a few bytes too short; we load them anyway. */
        reads3mleft( modfile, &samplerecs[sampnum], samprecord.bflags );

        /* If the sample is stereo, read in the right channel data and mix 
           it into the sample.  Note that a stereo .s3m sample is not 
           interlaced - the right channel data follows the left channel 
           data in the file. */
        if ( samprecord.bflags & 2 )
        {
            /* Seek to the right channel data. */
            if ( fseek( modfile, sampstart2, SEEK_SET ) )
                return( errclose( modfile ) );

            /* Read and mix in the right channel data.  Again we ignore 
               file errors. */
            reads3mright( modfile, &samplerecs[sampnum], samprecord.bflags );
        }

        /* Convert the sample to signed. */
        tosigned( &samplerecs[sampnum] );
    }

    /* Close the file. */
    fclose( modfile );

    /* Internal check:  verify the state of the memory manager.  Program 
       dies if the check fails. */
    if ( checkmem() )
        panic( allocerrmsg, 3 );

    /* Success. */
    return( 0 );
}


/*************************************************************************/
/*                                                                       */
/* loadfile() routine.  This function takes the name of a music module   */
/* file, determines its type, opens the file, and calls the appropriate  */
/* subroutine to load the file.  If successful, it returns 0.  If        */
/* unsuccessful, it returns -1 and sets errmsg to an error string (max   */
/* 80 characters, 1 line, null-terminated) to be displayed to the user.  */
/*                                                                       */
/*************************************************************************/

int loadfile( char *filename, char **errmsg )
{
    FILE *modfile;          /* module file to be loaded */

    /* Verify the drive letter, if any, in the filename. */
    *errmsg = baddrivemsg;
    if ( chkdrive( filename ) )
        return( -1 );

    /* Open the mod file. */
    *errmsg = openfailmsg;
    if ( (modfile = fopen( filename, "rb" )) == NULL )
        return( -1 );

    /* Check whether it's an .s3m file, and call that loader if it is. */
    if ( iss3m( modfile ) )
        return( loads3mfile( modfile, errmsg ) );

    /* Not an .s3m file - call the mod loader. */
    else
        return( loadmodfile( modfile, errmsg ) );
}
