/*
    Modstuf.c

    This is the first part of the C "half" of the player module.  Here we 
    do file I/O, which is easier to do in C than assembler, and not that 
    time-critical.  This file has been broken into two parts to get around 
    the editor's 64k limit.
*/

#include <stdio.h>      /* standard I/O functions header file */
#include <string.h>     /* standard string functions header file */
#include <dos.h>        /* standard DOS functions header file */
#include <stdlib.h>     /* standard library functions header file */
#include "memmgr.h"     /* memory management routines */
#include "modstuf.h"    /* music module loading/playing routines */
#include "dacstuf.h"    /* Tandy DAC routines */

    /* Routine to abort with a message on nonrecoverable errors. */
extern void panic( char *msg, int retcod );

/*************************************************************************/
/*                                                                       */
/* Macros.                                                               */
/*                                                                       */
/*************************************************************************/

    /* Size of the file I/O buffer.  This is not a random number - 8192 is 
       the size of a 32-channel pattern in the mod file. */
#define IOBUFSIZE 8192


/*************************************************************************/
/*                                                                       */
/* Types.                                                                */
/*                                                                       */
/*************************************************************************/

    /* Sample frequency count record. */
struct freqstruct {
    int sampnum;            /* sample number */
    unsigned long freq;     /* occurrence count in patterns */
};

    /* .s3m sample header record. */
struct s3msamp {
    char samptype;      /* 1 = sample, 2 = Adlib melody, 3 = Adlib drum */
    char dosname[12];   /* name of file */
    unsigned char paratop;  /* top 8 bits of parapointer to sample data (?) */
    unsigned paraptr;   /* low 16 bits of parapointer to sample data */
    unsigned long samplen;  /* sample length */
    unsigned long loopbeg;  /* loop start */
    unsigned long loopend;  /* loop end */
    unsigned char vol;  /* volume for sample, 0-64 */
    char pad4;          /* (unused) */
    char pack;          /* 1 if sample is packed in ADPCM format */
        /* Bitmapped flags:  bit 0 = looped sample; bit 1 = stereo sample; 
           bit 2 = 16-bit sample. */
    char bflags;
    unsigned long c4spd;    /* C4SPD value (finetune) */
    char pad6[4];       /* (unused) */
    unsigned intgp;     /* address of sample in GUS memory */
    unsigned int152;    /* SB loop expansion flags */
    unsigned long lastused; /* last used position for SB */
    char name[28];      /* name of sample */
    char tag[4];        /* "SCRS" */
};


/*************************************************************************/
/*                                                                       */
/* Global variables.                                                     */
/*                                                                       */
/*************************************************************************/

    /* "Save memory" flag.  If 1, unused patterns are not loaded. */
char savemem = 1;
    /* "Optimize loading" flag.  If 1, the patterns are scanned to 
       determine which samples are most common, and they are loaded first. 
       */
char optimize = 1;

static unsigned char iobuf[IOBUFSIZE];  /* file I/O buffer */

    /* Channel flags.  1 if channel is on the left. */
static char chanleft[32] = {
    1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1,
    1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1
};

    /* Error message strings. */
static char baddrivemsg[] = "Invalid drive.";
static char openfailmsg[] = "Open failed on song file.";
static char seekfailmsg[] = "Seek failed on song file.";
static char readfailmsg[] = "Read failed on song file.";
static char patsizemsg[] =
    "Invalid number of entries in the pattern table.";
static char nomemmsg[] = "Insufficient memory.";
static char openfailmsg2[] = "Open failed on output file.";
static char writefailmsg2[] = "Write failed on output file.";
static char closefailmsg2[] = "Close failed on output file.";
static char badsammsg[] = "Invalid sample number.";
static char badmodnamemsg[] = "Bad module name for sample dump.";
static char allocerrmsg[] = "Internal memory allocation error.";
static char badsamtypemsg[] = "Bad sample type in .s3m file.";
static char badnordersmsg[] = "Bad number of orders in .s3m file.";
static char badninstrsmsg[] = "Bad number of instruments in .s3m file.";
static char badnpatsmsg[] = "Bad number of patterns in .s3m file.";
static char badchansetmsg[] = "Bad channel settings in .s3m file.";
static char alladlibmsg[] = "No non-Adlib data in .s3m file.";


/*************************************************************************/
/*                                                                       */
/* Function prototypes.                                                  */
/*                                                                       */
/*************************************************************************/

static int errclose( FILE *fileptr );
static void interlace( struct filnotrec *filenotes );
static int errclose2( FILE *outfile, char *outpath );
static int errclose3( FILE *outfile, char *outpath, char *outpath2 );
static int freqcompare( struct freqstruct *first, struct freqstruct *second );
static int loadmodfile( FILE *modfile, char **errmsg );
static int iss3m( FILE *modfile );
static void s3m2modeff( struct noterec *note );
static void reads3mleft( FILE *modfile, struct samprec *samprecord,
    char bflags );
static void reads3mright( FILE *modfile, struct samprec *samprecord,
    char bflags );
static void tosigned( struct samprec *samprecord );
static int loads3mfile( FILE *modfile, char **errmsg );


/*************************************************************************/
/*                                                                       */
/* errclose() function.  This is a convenience routine used on errors    */
/* with loadmodfile().  It releases memory associated with the partially */
/* loaded mod file, closes the file, and returns -1.                     */
/*                                                                       */
/*************************************************************************/

static int errclose( FILE *fileptr )
{
    int i;              /* for looping */

    /* Close the file. */
    fclose( fileptr );

    /* Release memory allocated to sound data buffers before we panicked. 
       */
    for ( i=1; i<100; i++ )
        if ( samplerecs[i].sampseg
            && samplerecs[i].sampseg != samplerecs[0].sampseg )
        {
            if ( samplerecs[i].sampemsflag )
                freeemsblock( samplerecs[i].sampemshandle );
            else
                freeblock( samplerecs[i].sampseg,
                    (samplerecs[i].samplen + 7L) / 8 );
            samplerecs[i].sampseg = 0;
        }

    /* Release memory allocated to pattern buffers before we panicked. */
    if ( patbufsize )
    {
        for ( i=0; i<256; i++ )
            if ( patternsegs[i] )
            {
                freeblock( patternsegs[i], patbufsize );
                patternsegs[i] = 0;
            }
        patbufsize = 0;
    }

    return( -1 );
}


/*************************************************************************/
/*                                                                       */
/* interlace() routine.  This routine takes a buffer containing 2 4-     */
/* channel patterns and interlaces them, making an 8-channel pattern.    */
/* For FLT8 modules.                                                     */
/*                                                                       */
/*************************************************************************/

static void interlace( struct filnotrec *filenotes )
{
        /* We deal with data in 4-note "rows" here. */
    struct filrow {
        struct filnotrec thenotes[4];
    } *filerows, startdata;
    int startrow, thisrow, nextrow;  /* rows being moved */
    char doneflags[128];             /* flag is 1 if row was done */
    int i;                           /* for looping */

    /* filerows is the array of 4-note records we're shuffling around. */
    filerows = (struct filrow *) filenotes;

    /* The first and last 4-note records are already in the right places, 
       but the rest are not. */
    for ( i=1; i<127; i++ )
        doneflags[i] = 0;
    doneflags[0] = doneflags[127] = 1;

    /* Reshuffle rows until they are all in the right places. */
    for ( startrow = 1; startrow < 127; startrow++ )
    {
        /* If this row already has the right note data in it, go to the 
           next one. */
        if ( doneflags[startrow] )
            continue;

        /* Shuffle rows around.  For each row, we figure out what row 
           should be there and put it there.  We're guaranteed to get back 
           to where we started because the same row doesn't get put in two 
           different places. */
        thisrow = startrow;
        startdata = filerows[thisrow];
        do
        {
            if ( thisrow & 1 )
                nextrow = (thisrow-1)/2 + 64;
            else
                nextrow = thisrow/2;
            filerows[thisrow] = filerows[nextrow];
            doneflags[thisrow] = 1;
            thisrow = nextrow;
        } while ( thisrow != startrow );
    }
}


/*************************************************************************/
/*                                                                       */
/* freqcompare() routine.  This function is to be used with qsort() when */
/* loading optimization is on.  It takes pointers to to sample frequency */
/* structures and returns -1 if the first sample is more frequent than   */
/* the second, 0 if they are the same, 1 if the second is more frequent  */
/* than the first (i.e., we're sorting the samples in order of increas-  */
/* in frequency).                                                        */
/*                                                                       */
/*************************************************************************/

static int freqcompare( struct freqstruct *first, struct freqstruct *second )
{
    if ( first->freq > second->freq )
        return( -1 );
    else if ( first->freq < second->freq )
        return( 1 );
    else
        return( 0 );
}


/*************************************************************************/
/*                                                                       */
/* loadmodfile() routine.  This function takes an open music module file */
/* in Protracker (or unknown) format, loads it into memory, and closes   */
/* the file.  If successful, it returns 0.  If unsuccessful, it returns  */
/* -1 and sets errmsg to an error string (max 80 characters, 1 line,     */
/* null-terminated) to be displayed to the user.                         */
/*                                                                       */
/*************************************************************************/

static int loadmodfile( FILE *modfile, char **errmsg )
{
    int i,j;                /* for looping, of course */
    char filetag[4];        /* filetype tag */
    int flt8flag;           /* 1 if an FLT8 mod */
    unsigned char tempchar; /* for reading 1 byte from the file */
        /* Pattern table from the file (use global I/O buffer). */
    unsigned char *filpattable = (unsigned char *) iobuf;
    long patstart;          /* start of pattern data in the file */
    unsigned filpatsize;    /* size of a pattern in the file */
        /* Sample structure from the file. */
    struct filsamrec filesample;
        /* Starting location in the file of the sound data for each sample.  
           Yes, we waste 4 bytes here ;). */
    long samplestarts[32];
        /* Pointer in the pattern buffer. */
    struct noterec far *noteptr;
    int found;              /* 1 if pattern found in pattern table */
    int sampnum;            /* number of sample */
    int chan;               /* number of channel */
        /* Sample frequency structures, used to optimize loading. */
    struct freqstruct sampfreqs[32];
    struct SREGS segregs;   /* segment register values, we need DS */
    unsigned wordsleft;     /* number of words remaining to read */
    unsigned thistime;      /* number to read this time */
    unsigned long linaddr;  /* linear address in sound data buffer */
    unsigned insegment;     /* words in 64k segment of sound buffer */
        /* Conversion table for discarding unused channels. */
    unsigned char chanconvert[32];
    int newnchannels;           /* number of channels actually in use */
    int newbytesperrow;         /* number of bytes per row in new patterns */
    int newpatbufsize;          /* size of new pattern in paragraphs */
    int nright, nleft;          /* number of right and left channels */
    char chanused[32];          /* 1 if channel number used */
    int row;                    /* row in the pattern */
    struct noterec tempnote;    /* temporary note record */

    /* Initialize all samples to be copies of sample 0, the silence sample. 
       */
    for ( i=1; i<100; i++ )
        samplerecs[i] = samplerecs[0];

    /* Seek to the filetype tag at offset 1080 and determine the mod file 
       type.  Set the number of bytes per row in the pattern table 
       according to the number of channels found. */
    *errmsg = seekfailmsg;
    if ( fseek( modfile, 1080L, SEEK_SET ) )
        return( errclose( modfile ) );
    *errmsg = readfailmsg;
    if ( fread( filetag, sizeof( char ), 4, modfile ) != 4 )
        return( errclose( modfile ) );
    ninstruments = 31;
    flt8flag = 0;
    if ( !strncmp( filetag, "M.K.", 4 ) )
        nchannels = 4;
    else if ( !strncmp( filetag, "M!K!", 4 ) )
        nchannels = 4;
    else if ( !strncmp( filetag, "FLT4", 4 ) )
        nchannels = 4;
    else if ( !strncmp( filetag, "4CHN", 4 ) )
        nchannels = 4;
    else if ( !strncmp( filetag, "6CHN", 4 ) )
        nchannels = 6;
    else if ( !strncmp( filetag, "FLT8", 4 ) )
    {
        nchannels = 8;
        flt8flag = 1;
    }
    else if ( !strncmp( filetag, "8CHN", 4 ) )
        nchannels = 8;
    else if ( !strncmp( filetag, "OCTA", 4 ) )
        nchannels = 8;
    else if ( !strncmp( filetag, "CD81", 4 ) )
        nchannels = 8;
    else if ( !strncmp( filetag, "16CH", 4 ) )
        nchannels = 16;
    else if ( !strncmp( filetag, "32CH", 4 ) )
        nchannels = 32;
    else
    {
        ninstruments = 15;
        nchannels = 4;
    }
    bytesperrow = nchannels * sizeof( struct noterec );

    /* Mod files do not contain a global default volume, initial ticks per 
       row, or initial beats per minute, so just set them to something 
       sensible. */
    globalvoldefl = globalvol = 64;
    ticksdefl = ticksperrow = 6;
    bpmdefl = beatspermin = 125;

    /* Get the number of entries in the pattern table and the song end jump 
       position. */
    *errmsg = seekfailmsg;
    if ( fseek( modfile, 20L + ninstruments*sizeof( struct filsamrec ),
            SEEK_SET ) )
        return( errclose( modfile ) );
    *errmsg = readfailmsg;
    if ( fread( &tempchar, sizeof( char ), 1, modfile ) != 1 )
        return( errclose( modfile ) );
    *errmsg = patsizemsg;
    if ( tempchar == 0 || tempchar > 128 )
        return( errclose( modfile ) );
    pattablesize = tempchar;
    *errmsg = readfailmsg;
    if ( fread( &tempchar, sizeof( char ), 1, modfile ) != 1 )
        return( errclose( modfile ) );
    if ( tempchar >= pattablesize || tempchar >= 127 )
        songendjump = 256;
    else
        songendjump = tempchar;

    /* Read in the pattern table. */
    if ( fread( filpattable, sizeof( char ), 128, modfile ) != 128 )
        return( errclose( modfile ) );

    /* Nasty hack #1:  StarTrekker creates 8-channel mods where the 
       patterns are pairs of 2 4-channel patterns.  Hence, with this 
       format, all the pattern numbers need to be divided by 2. */
    if ( flt8flag )
        for ( i=0; i<128; i++ )
            filpattable[i] /= 2;

    /* Convert the pattern table entries to 16 bits and save the table.  
       Determine the number of patterns stored in the file. */
    nfilepatterns = 0;
    for ( i=0; i<128; i++ )
        if ( (patterntable[i] = filpattable[i]) > nfilepatterns )
            nfilepatterns = patterntable[i];
    nfilepatterns++;

    /* Determine the size of a pattern buffer in memory in paragraphs and 
       the size of a pattern in the file in bytes, and initialize the table 
       of segment addresses of pattern buffers to all zeros. */
    patbufsize = 4*bytesperrow;
    filpatsize = 64*nchannels*sizeof( struct filnotrec );
    for ( i=0; i<256; i++ )
        patternsegs[i] = 0;

    /* Determine the start of pattern data in the file. */
    patstart = 20L + ninstruments*sizeof( struct filsamrec ) + 2 + 128;
    if ( ninstruments == 31 )
        patstart += 4;

    /* Read in sample data structures. */
    *errmsg = seekfailmsg;
    if ( fseek( modfile, 20L, SEEK_SET ) )
        return( errclose( modfile ) );
    *errmsg = readfailmsg;
    for ( i=1; i<=ninstruments; i++ )
    {
        if ( fread( &filesample, sizeof( struct filsamrec ), 1, modfile )
                != 1 )
            return( errclose( modfile ) );
        samplerecs[i].samplen =
            ((filesample.filsamlen << 8) & 0xff00) +
            ((filesample.filsamlen >> 8) & 0x00ff);
        if ( filesample.filsamfine > 15 )
            filesample.filsamfine = 0;
        samplerecs[i].sampc4spd = c4spds[filesample.filsamfine ^ 8];
        samplerecs[i].sampvol =
            (filesample.filsamvol > 64 ? 64 : filesample.filsamvol);
        samplerecs[i].samplstart =
            ((filesample.filsamrstart << 8) & 0xff00) +
            ((filesample.filsamrstart >> 8) & 0x00ff);
        samplerecs[i].sampllen =
            ((filesample.filsamrlen << 8) & 0xff00) +
            ((filesample.filsamrlen >> 8) & 0x00ff);
    }

    /* Nasty hack #2:  Grave Composer for IBM creates 8-channel mods with 
       an "M.K." signature.  We need to check the length of the file to 
       determine whether we have one of those or just a normal 4-channel 
       M.K. mod. */
    if ( !strncmp( filetag, "M.K.", 4 ) )
    {
        long eightlen;      /* length of file if 8 channels */
        long currentpos;    /* current position in the file */
        long reallen;       /* real length of the file */

        /* Determine how long the file would be if it had 8 channels. */
        eightlen = 1084 + nfilepatterns * 2048L;
        for ( i=1; i<=31; i++ )
            eightlen += (long) samplerecs[i].samplen * 2;

        /* Determine how long the file actually is. */
        *errmsg = seekfailmsg;
        if ( (currentpos = ftell( modfile )) == -1L )
            return( errclose( modfile ) );
        if ( fseek( modfile, 0L, SEEK_END ) )
            return( errclose( modfile ) );
        if ( (reallen = ftell( modfile )) == -1L )
            return( errclose( modfile ) );
        if ( fseek( modfile, currentpos, SEEK_SET ) )
            return( errclose( modfile ) );

        /* If the length of the file matches an 8-channel mod, change the 
           variables to match. */
        if ( reallen >= eightlen )
        {
            nchannels = 8;
            bytesperrow = nchannels * sizeof( struct noterec );
            patbufsize = 4*bytesperrow;
            filpatsize = 64*nchannels*sizeof( struct filnotrec );
        }
    }

    /* Determine where the sound data for each sample begins in the file.  
       We're going to skip the first 2 bytes of each sample, which are 
       always 0 and not really part of the sample. */
    samplestarts[1] = patstart + ((long) nfilepatterns)*filpatsize + 2;
    for ( i=2; i<=ninstruments; i++ )
        samplestarts[i] = samplestarts[i-1]
            + ((long) samplerecs[i-1].samplen << 1);

    /* Go back and fix up the sample data structures so they make sense.  
       In the process, we figure out how much of the sample we *really* 
       want. */
    for ( i=1; i<=ninstruments; i++ )
    {
        /* If the length of the sample and/or the start of the loop is not 
           zero, decrement it.  We're tossing out the zero word at the 
           beginning of the sample. */
        if ( samplerecs[i].samplen )
            samplerecs[i].samplen--;
        if ( samplerecs[i].samplstart )
            samplerecs[i].samplstart--;

        /* If the start of the loop is past the end of the sample, set the 
           start of the loop and loop length to zero. */
        if ( samplerecs[i].samplstart >= samplerecs[i].samplen )
        {
            samplerecs[i].samplstart = 0;
            samplerecs[i].sampllen = 0;
        }

        /* If the end of the loop is past the end of the sample, set the 
           loop length to make the end of the loop equal to the length of 
           the sample. */
        if ( (long) samplerecs[i].samplstart + samplerecs[i].sampllen
                > (long) samplerecs[i].samplen )
            samplerecs[i].sampllen =
                samplerecs[i].samplen - samplerecs[i].samplstart;

        /* If the length of the loop is 1 or less, set the start of the 
           loop and loop length to zero. */
        if ( samplerecs[i].sampllen <= 1 )
        {
            samplerecs[i].samplstart = 0;
            samplerecs[i].sampllen = 0;
        }

        /* If the length of the loop is not zero, indicating that the 
           sample is looped, and the sample extends past the end of the 
           loop, adjust the length of the sample to discard the part after 
           the loop.  We won't play it anyway. */
        if ( samplerecs[i].sampllen
                && samplerecs[i].samplen
                > samplerecs[i].samplstart + samplerecs[i].sampllen )
        samplerecs[i].samplen =
            samplerecs[i].samplstart + samplerecs[i].sampllen;
    }

    /* Read in the patterns. */
    for ( i=0; i<nfilepatterns; i++ )
    {
        /* If the "save memory" flag is set, check to see if the pattern is 
           actually used.  If not, don't load it. */
        if ( savemem )
        {
            found = 0;
            for ( j=0; j<pattablesize; j++ )
                if ( patterntable[j] == i )
                {
                    found = 1;
                    break;
                }
            if ( !found )
                continue;
        }

        /* Allocate memory for the pattern. */
        *errmsg = nomemmsg;
        if ( !(patternsegs[i] = getblock( patbufsize )) )
            return( errclose( modfile ) );

        /* Seek to the pattern data. */
        *errmsg = seekfailmsg;
        if ( fseek( modfile, patstart + i*filpatsize, SEEK_SET ) )
            return( errclose( modfile ) );

        /* Read in the pattern data. */
        *errmsg = readfailmsg;
        if ( fread( iobuf, 1, filpatsize, modfile ) != filpatsize )
            return( errclose( modfile ) );

        /* Nasty hack #3:  StarTrekker creates 8-channel mods where the 
           patterns are pairs of 2 4-channel patterns.  We need to 
           interlace the pattern data to correspond to a normal 8-channel 
           mod. */
        if ( flt8flag )
            interlace( (struct filnotrec *) iobuf );

        /* Convert the pattern to internal format, copying to pattern 
           buffer. */
        FP_SEG( noteptr ) = patternsegs[i];
        FP_OFF( noteptr ) = 0;
        fil2memnotes( noteptr, (struct filnotrec *) iobuf, nchannels*64 );
    }

    /* If "save memory" is enabled, scan the patterns for unused channels, 
       and delete them if possible. */
    if ( savemem )
    {
        /* Loop over the channels. */
        for ( chan = 0; chan < nchannels; chan++ )
        {
            /* Initially mark the channel unused. */
            chanused[chan] = 0;

            /* Loop over the patterns that are actually played. */
            for ( i=0; i<pattablesize && !chanused[chan]; i++ )
            {
                /* If the pattern table entry is 254 or 255, skip it - 254 
                   means skip the entry, 255 means stop the song. */
                if ( patterntable[i] >= 254 )
                    continue;

                /* Get a C pointer to the pattern buffer. */
                FP_SEG( noteptr ) = patternsegs[patterntable[i]];
                FP_OFF( noteptr ) = chan * sizeof( struct noterec );

                /* Loop over the notes in the pattern. */
                for ( j=0; j<64 && !chanused[chan]; j++ )
                {
                    /* If the sample number, period, effect number, or 
                       effect parameter is nonzero, the channel is actually 
                       in use. */
                    if ( noteptr->notesampnum || noteptr->noteperiod
                            || noteptr->noteeffparm || noteptr->noteeffnum )
                        chanused[chan] = 1;
                    noteptr += nchannels;
                }
            }
        }

        /* Loop over the channels again, checking to see how many right and 
           left channels there really are. */
        nright = nleft = 0;
        for ( chan = 0; chan < nchannels; chan++ )
        {
            if ( chanleft[chan] )
            {
                if ( chanused[chan] )
                {
                    chanconvert[chan] = chan;
                    nleft++;
                }
                else
                    chanconvert[chan] = 255;
            }
            else
            {
                if ( chanused[chan] )
                {
                    chanconvert[chan] = chan;
                    nright++;
                }
                else
                    chanconvert[chan] = 255;
            }
        }

        /* Determine the number of used channels. */
        newnchannels = nleft << 1;
        if ( (nright << 1) > newnchannels )
            newnchannels = nright << 1;
        if ( newnchannels < 4 )
            newnchannels = 4;

        /* If the number of channels actually used is less than the number 
           given in the file header, get rid of the unused channels. */
        if ( newnchannels < nchannels )
        {
            /* Determine where the channels should go. */
            for ( chan = 0; chan < nchannels; chan++ )
                for ( i=0; i<newnchannels; i++ )
                    if ( chanleft[i] == chanleft[chan]
                        && chanconvert[i] == 255
                        && chanconvert[chan] != 255 )
                    {
                        chanconvert[i] = chanconvert[chan];
                        chanconvert[chan] = 255;
                        break;
                    }

            /* Determine the number of bytes per row in the compressed 
               pattern and the number of paragraphs of data in the 
               compressed pattern. */
            newbytesperrow = newnchannels * sizeof( struct noterec );
            newpatbufsize = 4*newbytesperrow;

            /* Loop over the patterns.  Note that we *must* go in the order 
               the patterns were loaded here. */
            for ( i=0; i<nfilepatterns; i++ )
            {
                /* If the pattern was not loaded, skip it. :) */
                if ( !patternsegs[i] )
                    continue;

                /* Get a C pointer to the pattern buffer. */
                FP_SEG( noteptr ) = patternsegs[i];
                FP_OFF( noteptr ) = 0;

                /* Compress unused channels out of the pattern. */
                tempnote.notesampnum = tempnote.noteperiod =
                    tempnote.noteeffparm = tempnote.noteeffnum = 0;
                tempnote.notevol = 255;
                for ( row = 0; row < 64; row++ )
                    for ( chan = 0; chan < newnchannels; chan++ )
                    {
                        if ( chanconvert[chan] == 255 )
                            noteptr[row*newnchannels + chan] = tempnote;
                        else
                            noteptr[row*newnchannels + chan] =
                                noteptr[row*nchannels + chanconvert[chan]];
                    }

                /* Release the (now) unused memory at the end of the 
                   pattern. */
                if ( freeblock( patternsegs[i] + newpatbufsize,
                        patbufsize - newpatbufsize ) )
                    panic( allocerrmsg, 3 );

                /* Move the block down in memory as far as possible. */
                patternsegs[i] = movedown( patternsegs[i], newpatbufsize );
            }

            /* Reset the number of channels, number of bytes per row in a 
               pattern, and number of paragraphs of memory required for a 
               pattern. */
            nchannels = newnchannels;
            bytesperrow = newbytesperrow;
            patbufsize = newpatbufsize;
        }
    }

    /* Construct the sample loading order - by default, the order in which 
       they appear in the file.  Set occurrence counts to 0. */
    for ( i=1; i<=ninstruments; i++ )
    {
        sampfreqs[i].sampnum = i;
        sampfreqs[i].freq = 0L;
    }

    /* If we're to optimize loading, scan the patterns to see which samples 
       are most commonly used, and adjust the loading order so that the 
       most common ones will be loaded first (hence be more likely to be 
       loaded into conventional RAM). */
    if ( optimize )
    {
        /* Loop over the channels. */
        for ( chan = 0; chan < nchannels; chan++ )
        {
            /* Initial sample on every channel is 0. */
            sampnum = 0;

            /* Loop over the patterns that are actually played. */
            for ( i=0; i<pattablesize; i++ )
            {
                /* Get a C pointer to the pattern buffer. */
                FP_SEG( noteptr ) = patternsegs[patterntable[i]];
                FP_OFF( noteptr ) = chan * sizeof( struct noterec );

                /* Loop over the notes in the pattern, accumulating sample 
                   counts. */
                for ( j=0; j<64; j++ )
                {
                    if ( noteptr->notesampnum &&
                            noteptr->notesampnum <= ninstruments )
                        sampnum = noteptr->notesampnum;
                    sampfreqs[sampnum].freq++;
                    noteptr += nchannels;
                }
            }
        }

        /* Sort the occurrence structures in increasing order. */
        qsort( &sampfreqs[1], ninstruments, sizeof( struct freqstruct ),
            freqcompare );
    }

    /* Allocate buffers for the sound data for each sample, and read the 
       data into those buffers. */
    segread( &segregs );
    for ( i=1; i<=ninstruments; i++ )
    {
        /* If loading optimization and "save memory" are both enabled, and 
           the sample is never used, skip loading it. */
        if ( optimize && savemem && !sampfreqs[i].freq )
            continue;

        /* Get the sample number from the order list. */
        sampnum = sampfreqs[i].sampnum;

        /* If the sample is empty, set it back to the silence sample. */
        if ( !samplerecs[sampnum].samplen )
        {
            samplerecs[sampnum] = samplerecs[0];
            continue;
        }

        /* Allocate a buffer for the sample.  First try to get conventional 
           RAM, then EMS RAM. */
        *errmsg = nomemmsg;
        if ( !(samplerecs[sampnum].sampseg
            = getblock( (samplerecs[sampnum].samplen + 7L) / 8 )) )
        {
            /* Didn't get conventional RAM, try EMS. */
            if ( !(samplerecs[sampnum].sampemshandle
                    = getemsblock( (samplerecs[sampnum].samplen + 7L) / 8 )) )
                return( errclose( modfile ) );

            /* We got some EMS RAM. */
            samplerecs[sampnum].sampseg = emsframeseg;
            samplerecs[sampnum].sampemsflag = 1;
        }

        /* Seek to the start of the sound data for the sample. */
        *errmsg = seekfailmsg;
        if ( fseek( modfile, samplestarts[sampnum], SEEK_SET ) )
            return( errclose( modfile ) );

        /* Read the sound data into the buffer.  There are two cases:  
           conventional and EMS.  In case of file errors, we will just stop 
           loading the sample. */
        if ( samplerecs[sampnum].sampemsflag )
        {
            /* Read sound data into expanded RAM buffer, starting with the 
               first 64k. */
            wordsleft = samplerecs[sampnum].samplen;
            mapems1st( samplerecs[sampnum].sampemshandle );
            insegment = 0;
            while ( wordsleft && insegment < 32768 )
            {
                thistime =
                    (wordsleft > IOBUFSIZE/2 ? IOBUFSIZE/2 : wordsleft);
                if ( 32768 - insegment < thistime )
                    thistime = 32768 - insegment;
                if ( fread( iobuf, 2, thistime, modfile ) != thistime )
                    wordsleft = thistime;
                movedata( segregs.ds, (unsigned) iobuf,
                    emsframeseg, insegment << 1, thistime << 1 );
                insegment += thistime;
                wordsleft -= thistime;
            }

            /* If less than or equal to 64k in the sample, go to the next 
               sample. */
            if ( !wordsleft )
                continue;

            /* Map in the second 64k and read in the rest of the sample. */
            mapems2nd( samplerecs[sampnum].sampemshandle );
            insegment = 0;
            while ( wordsleft )
            {
                thistime =
                    (wordsleft > IOBUFSIZE/2 ? IOBUFSIZE/2 : wordsleft);
                if ( fread( iobuf, 2, thistime, modfile ) != thistime )
                    wordsleft = thistime;
                movedata( segregs.ds, (unsigned) iobuf,
                    emsframeseg, insegment << 1, thistime << 1 );
                insegment += thistime;
                wordsleft -= thistime;
            }
        }
        else
        {
            /* Read sound data into conventional RAM buffer. */
            wordsleft = samplerecs[sampnum].samplen;
            linaddr = (unsigned long) samplerecs[sampnum].sampseg << 4;
            while ( wordsleft )
            {
                thistime =
                    (wordsleft > IOBUFSIZE/2 ? IOBUFSIZE/2 : wordsleft);
                if ( fread( iobuf, 2, thistime, modfile ) != thistime )
                    wordsleft = thistime;
                movedata( segregs.ds, (unsigned) iobuf,
                    linaddr >> 4, linaddr & 0xf, thistime << 1 );
                linaddr += thistime << 1;
                wordsleft -= thistime;
            }
        }
    }

    /* Close the file. */
    fclose( modfile );

    /* Internal check:  verify the state of the memory manager.  Program 
       dies if the check fails. */
    if ( checkmem() )
        panic( allocerrmsg, 3 );

    /* Success. */
    return( 0 );
}


/*************************************************************************/
/*                                                                       */
/* errclose2() function.  This is a convenience routine used on errors   */
/* with dumpmod().  It closes the output file and deletes it, then       */
/* returns -1.                                                           */
/*                                                                       */
/*************************************************************************/

static int errclose2( FILE *outfile, char *outpath )
{
    fclose( outfile );
    remove( outpath );
    return( -1 );
}


/*************************************************************************/
/*                                                                       */
/* dumpmod() function.  This routine writes information about the        */
/* currently-loaded music module to file outpath in human-readable form. */
/* Returns 0 if successful.  If unsuccessful, errmsg is an error message */
/* to be displayed to the user, and -1 is returned.                      */
/*                                                                       */
/*************************************************************************/

int dumpmod( char *outpath, char **errmsg )
{
    FILE *outfile;                  /* output file */
    int i,j,k;                      /* for looping */
    struct noterec far *patptr;     /* pointer to note in pattern */
    char *patstrptr;                /* traversing pointer in output string */
    int linelen;                    /* length of an output line */
    int found;                      /* true if finetune in c4spds table */

    /* Open the output file. */
    *errmsg = openfailmsg2;
    if ( (outfile = fopen( outpath, "wt" )) == NULL )
        return( -1 );

    /* Print the number of channels and number of samples. */
    *errmsg = writefailmsg2;
    if ( fprintf( outfile, "Module data:\n  %u channels\n  %u samples\n",
            nchannels, ninstruments ) == EOF )
        return( errclose2( outfile, outpath ) );
    if ( fprintf( outfile, "  initial ticks/division %d\n",
            ticksdefl ) == EOF )
        return( errclose2( outfile, outpath ) );
    if ( fprintf( outfile, "  initial bpm %d\n", bpmdefl ) == EOF )
        return( errclose2( outfile, outpath ) );
    if ( fprintf( outfile, "  default global volume %d\n\n",
            globalvoldefl ) == EOF )
        return( errclose2( outfile, outpath ) );

    /* Print sample information. */
    if ( fputs( "Sample data ---------------------------------\n\n",
            outfile ) == EOF )
        return( errclose2( outfile, outpath ) );
    for ( i=0; i<=ninstruments; i++ )
    {
        if ( fprintf( outfile, "Sample %d\n", i ) == EOF )
            return( errclose2( outfile, outpath ) );
        if ( fprintf( outfile, "  length %u\n", samplerecs[i].samplen )
                == EOF )
            return( errclose2( outfile, outpath ) );
        if ( fprintf( outfile, "  loop start %u\n",
                samplerecs[i].samplstart ) == EOF )
            return( errclose2( outfile, outpath ) );
        if ( fprintf( outfile, "  loop length %u\n",
                samplerecs[i].sampllen ) == EOF )
            return( errclose2( outfile, outpath ) );
        if ( fprintf( outfile, "  C4SPD:  %u ",
                samplerecs[i].sampc4spd ) == EOF )
            return( errclose2( outfile, outpath ) );
        found = 0;
        for ( j=0; j<16; j++ )
            if ( c4spds[j] == samplerecs[i].sampc4spd )
            {
                found = 1;
                break;
            }
        if ( found )
        {
            if ( fprintf( outfile, "(finetune %d)\n", j-8 ) == EOF )
                return( errclose2( outfile, outpath ) );
        }
        else
        {
            if ( fputs( "(no such finetune)\n", outfile ) == EOF )
                return( errclose2( outfile, outpath ) );
        }
        if ( fprintf( outfile, "  volume %d\n",
                samplerecs[i].sampvol ) == EOF )
            return( errclose2( outfile, outpath ) );
        if ( samplerecs[i].sampemsflag )
        {
            if ( fprintf( outfile, "  stored in EMS with handle %04Xh\n\n",
                    samplerecs[i].sampemshandle ) == EOF )
                return( errclose2( outfile, outpath ) );
        }
        else
        {
            if ( fprintf( outfile, "  stored at segment %04Xh\n\n",
                    samplerecs[i].sampseg ) == EOF )
                return( errclose2( outfile, outpath ) );
        }
    }

    /* Print pattern table information. */
    if ( fputs( "Pattern data --------------------------------\n\n",
            outfile ) == EOF )
        return( errclose2( outfile, outpath ) );
    if ( fprintf( outfile, "  pattern table has %u entries\n",
            pattablesize ) == EOF )
        return( errclose2( outfile, outpath ) );
    if ( fprintf( outfile, "  there were %u patterns in the file\n",
            nfilepatterns ) == EOF )
        return( errclose2( outfile, outpath ) );
    if ( fprintf( outfile, "  a pattern buffer is %u paragraphs\n",
            patbufsize ) == EOF )
        return( errclose2( outfile, outpath ) );
    if ( fprintf( outfile, "  song end jump position is %u\n",
            songendjump ) == EOF )
        return( errclose2( outfile, outpath ) );
    if ( fprintf( outfile, "  %u bytes per row in the pattern table\n\n",
            bytesperrow ) == EOF )
        return( errclose2( outfile, outpath ) );

    /* Print out pattern table. */
    if ( fputs( "Pattern table:\n", outfile ) == EOF )
        return( errclose2( outfile, outpath ) );
    for ( i=0; i<pattablesize; i++ )
    {
        if ( !(i & 7) )
            if ( fputc( '\n', outfile ) == EOF )
                return( errclose2( outfile, outpath ) );
        if ( fprintf( outfile, "  %02X:%02X", i, patterntable[i] ) == EOF )
            return( errclose2( outfile, outpath ) );
    }
    if ( fputs( "\n\n", outfile ) == EOF )
        return( errclose2( outfile, outpath ) );

    /* Print out the patterns themselves. */
    strcpy( iobuf, "                   " );
    for ( k=2; k<nchannels; k++ )
        strcat( iobuf, " |              " );
    strcat( iobuf, " |              \n" );
    linelen = strlen( iobuf );
    for ( i=0; i<nfilepatterns; i++ )
    {
        if ( fprintf( outfile, "Pattern %02X ", i ) == EOF )
            return( errclose2( outfile, outpath ) );
        if ( !patternsegs[i] )
        {
            if ( fputs( "not loaded\n\n", outfile ) == EOF )
                return( errclose2( outfile, outpath ) );
            continue;
        }
        if ( fprintf( outfile, "(stored at segment %04Xh):\n\n  ",
                patternsegs[i] ) == EOF )
            return( errclose2( outfile, outpath ) );
        for ( k=0; k<nchannels; k++ )
            if ( fprintf( outfile, "      channel %-2d", k ) == EOF )
                return( errclose2( outfile, outpath ) );
        if ( fputc( '\n', outfile ) == EOF )
            return( errclose2( outfile, outpath ) );
        FP_SEG( patptr ) = patternsegs[i];
        FP_OFF( patptr ) = 0;
        for ( j=0; j<64; j++ )
        {
            patstrptr = iobuf+2;
            byte2hex( patstrptr, j );
            patstrptr += 4;
            for ( k=0; k<nchannels; k++ )
            {
                periodstring( patstrptr, patptr->noteperiod );
                patstrptr += 4;
                sampstring( patstrptr, patptr->notesampnum );
                patstrptr += 3;
                volumestring( patstrptr, patptr->notevol );
                patstrptr += 3;
                effstring( patstrptr, patptr->noteeffnum,
                    patptr->noteeffparm );
                patstrptr += 6;
                patptr++;
            }
            if ( fwrite( iobuf, sizeof( char ), linelen, outfile )
                    != linelen )
                return( errclose2( outfile, outpath ) );
        }
        if ( fputc( '\n', outfile ) == EOF )
            return( errclose2( outfile, outpath ) );
    }

    /* Close the file. */
    *errmsg = closefailmsg2;
    if ( fclose( outfile ) )
        return( -1 );

    /* Success. */
    return( 0 );
}


/*************************************************************************/
/*                                                                       */
/* errclose3() function.  This is a convenience routine used on errors   */
/* with dumpsample().  It closes outfile (whose name is outpath) and     */
/* deletes it, deletes outpath2, and returns -1.                         */
/*                                                                       */
/*************************************************************************/

static int errclose3( FILE *outfile, char *outpath, char *outpath2 )
{
    fclose( outfile );
    remove( outpath );
    remove( outpath2 );
    return( -1 );
}


/*************************************************************************/
/*                                                                       */
/* dumpsample() function.  This routine writes ("rips") a sample from    */
/* the currently-loaded music module to a file in the current directory, */
/* accompanied by a separate text file listing the sample's loop start,  */
/* loop length, finetune, and volume.  The name of the sample file is    */
/* generated from the name of the module passed in modname by appending  */
/* a number 0-9 or letter A-V to the base name and attaching a ".SAM"    */
/* extension.  The name of the text file is similarly generated, with a  */
/* ".NOT" extension attached instead.  sampnum is the number of the      */
/* sample to write out, 0-31.  Returns 0 if successful.  If unsuccess-   */
/* ful, errmsg is an error message to be displayed to the user, and -1   */
/* is returned.                                                          */
/*                                                                       */
/*************************************************************************/

int dumpsample( unsigned sampnum, char *modname, char **errmsg )
{
    char *baseptr;          /* start of base name in modname */
    char *colonplace;       /* position of colon in modname, if any */
    char *slashplace;       /* position of backslash in modname, if any */
    char samname[13];       /* file name for .sam file */
    char notname[13];       /* file name for .not file */
    char *dotplace;         /* position of period in modname, if any */
    int namelen;            /* length of base name */
    char samchar;           /* character to append to base name */
    FILE *outfile;          /* output file */
    struct SREGS segregs;   /* segment register values, we need DS */
    unsigned wordsleft;     /* number of words remaining to read */
    unsigned thistime;      /* number to read this time */
    unsigned long linaddr;  /* linear address in sound data buffer */
    unsigned insegment;     /* words in 64k segment of sound buffer */
    unsigned loopstart;     /* loop start */
    int found;              /* true if finetune found in c4spds table */
    int i;                  /* for looping */

    /* Verify the sample number. */
    *errmsg = badsammsg;
    if ( sampnum > ninstruments )
        return( -1 );

    /* Extract the base name of the mod file. */
    *errmsg = badmodnamemsg;
    if ( (baseptr = modname) == NULL )
        return( -1 );
    colonplace = strrchr( baseptr, ':' );
    if ( colonplace != NULL )
        baseptr = colonplace + 1;
    slashplace = strrchr( baseptr, '\\' );
    if ( slashplace != NULL )
        baseptr = slashplace + 1;
    if ( strlen( baseptr ) > 12 )
        return( -1 );
    strcpy( samname, baseptr );
    dotplace = strchr( samname, '.' );
    if ( dotplace != NULL )
        *dotplace = '\0';
    namelen = strlen( samname );
    if ( !namelen || namelen > 8 )
        return( -1 );

    /* Generate the names of the output .sam and .not files. */
    if ( sampnum < 10 )
        samchar = '0' + sampnum;
    else
        samchar = 'A' + sampnum - 10;
    if ( namelen == 8 )
        samname[7] = samchar;
    else
    {
        samname[namelen] = samchar;
        samname[namelen+1] = '\0';
    }
    strcpy( notname, samname );
    strcat( samname, ".SAM" );
    strcat( notname, ".NOT" );

    /* Open the sample file. */
    *errmsg = openfailmsg2;
    if ( (outfile = fopen( samname, "wb" )) == NULL )
        return( -1 );

    /* Write two nulls at the start of the file.  Every sample should have 
       these.  Don't ask me why. */
    *errmsg = writefailmsg2;
    if ( fputc( '\0', outfile ) == EOF )
        return( errclose2( outfile, samname ) );
    if ( fputc( '\0', outfile ) == EOF )
        return( errclose2( outfile, samname ) );

    /* Write the actual sample to disk.  There are two cases:  sample in 
       EMS RAM, and in conventional RAM. */
    segread( &segregs );
    if ( samplerecs[sampnum].sampemsflag )
    {
        /* Write sound data from expanded RAM buffer, starting with the 
           first 64k. */
        wordsleft = samplerecs[sampnum].samplen;
        mapems1st( samplerecs[sampnum].sampemshandle );
        insegment = 0;
        while ( wordsleft && insegment < 32768 )
        {
            thistime = (wordsleft > IOBUFSIZE/2 ? IOBUFSIZE/2 : wordsleft);
            if ( 32768 - insegment < thistime )
                thistime = 32768 - insegment;
            movedata( emsframeseg, insegment << 1, segregs.ds,
                (unsigned) iobuf, thistime << 1 );
            if ( fwrite( iobuf, 2, thistime, outfile ) != thistime )
                return( errclose2( outfile, samname ) );
            insegment += thistime;
            wordsleft -= thistime;
        }

        /* If there is more than 64k in the sample, map in the second half 
           and continue. */
        if ( wordsleft )
        {
            /* Map in the second 64k and write out the rest of the sample. 
               */
            mapems2nd( samplerecs[sampnum].sampemshandle );
            insegment = 0;
            while ( wordsleft )
            {
                thistime =
                    (wordsleft > IOBUFSIZE/2 ? IOBUFSIZE/2 : wordsleft);
                movedata( emsframeseg, insegment << 1, segregs.ds,
                    (unsigned) iobuf, thistime << 1 );
                if ( fwrite( iobuf, 2, thistime, outfile ) != thistime )
                    return( errclose2( outfile, samname ) );
                insegment += thistime;
                wordsleft -= thistime;
            }
        }
    }
    else
    {
        /* Write sound data from conventional RAM buffer. */
        wordsleft = samplerecs[sampnum].samplen;
        linaddr = (unsigned long) samplerecs[sampnum].sampseg << 4;
        while ( wordsleft )
        {
            thistime = (wordsleft > IOBUFSIZE/2 ? IOBUFSIZE/2 : wordsleft);
            movedata( linaddr >> 4, linaddr & 0xf, segregs.ds,
                (unsigned) iobuf, thistime << 1 );
            if ( fwrite( iobuf, 2, thistime, outfile ) != thistime )
                return( errclose2( outfile, samname ) );
            linaddr += thistime << 1;
            wordsleft -= thistime;
        }
    }

    /* Close the sample file. */
    *errmsg = closefailmsg2;
    if ( fclose( outfile ) )
        return( -1 );

    /* Open the text file. */
    *errmsg = openfailmsg2;
    if ( (outfile = fopen( notname, "wt" )) == NULL )
    {
        remove( samname );
        return( -1 );
    }

    /* Write out the name of the sample file to the text file. */
    if ( fprintf( outfile, "Data for sample %s:\n\n", samname ) == EOF )
        return( errclose3( outfile, notname, samname ) );

    /* Write out the loop start and loop length in both words and bytes. */
    loopstart = samplerecs[sampnum].samplstart;
    if ( loopstart )
        loopstart++;
    if ( fprintf( outfile, "Loop start:  %u words / %lu bytes\n",
            loopstart, (unsigned long) loopstart << 1 ) == EOF )
        return( errclose3( outfile, notname, samname ) );
    if ( fprintf( outfile, "Loop length:  %u words / %lu bytes\n",
            samplerecs[sampnum].sampllen,
            (unsigned long) samplerecs[sampnum].sampllen << 1 ) == EOF )
        return( errclose3( outfile, notname, samname ) );

    /* Write out the C4SPD (finetune) and sample volume. */
    if ( fprintf( outfile, "C4SPD:  %u ",
            samplerecs[sampnum].sampc4spd ) == EOF )
        return( errclose3( outfile, notname, samname ) );
    found = 0;
    for ( i=0; i<16; i++ )
        if ( c4spds[i] == samplerecs[sampnum].sampc4spd )
        {
            found = 1;
            break;
        }
    if ( found )
    {
        if ( fprintf( outfile, "(finetune %d)\n", i-8 ) == EOF )
            return( errclose3( outfile, notname, samname ) );
    }
    else
    {
        if ( fputs( "(no such finetune)\n", outfile ) == EOF )
            return( errclose3( outfile, notname, samname ) );
    }
    if ( fprintf( outfile, "Volume:  %d\n",
            samplerecs[sampnum].sampvol ) == EOF )
        return( errclose3( outfile, notname, samname ) );

    /* Close the text file. */
    *errmsg = closefailmsg2;
    if ( fclose( outfile ) )
    {
        remove( samname );
        return( -1 );
    }

    /* Success. */
    return( 0 );
}


/*************************************************************************/
/*                                                                       */
/* unloadfile() function.  Unloads the currently-loaded song file.       */
/* Assumes that a file is in fact loaded.  Returns nothing.              */
/*                                                                       */
/*************************************************************************/

void unloadfile( void )
{
    int i;              /* for looping */

    /* Release memory allocated to sound data buffers. */
    for ( i=1; i<100; i++ )
        if ( samplerecs[i].sampseg != samplerecs[0].sampseg )
        {
            if ( samplerecs[i].sampemsflag )
            {
                if ( freeemsblock( samplerecs[i].sampemshandle ) )
                    panic( allocerrmsg, 3 );
            }
            else
            {
                if ( freeblock( samplerecs[i].sampseg,
                        (samplerecs[i].samplen + 7L) / 8 ) )
                    panic( allocerrmsg, 3 );
            }
            samplerecs[i] = samplerecs[0];
        }

    /* Release memory allocated to pattern buffers. */
    for ( i=0; i<256; i++ )
        if ( patternsegs[i] )
        {
            if ( freeblock( patternsegs[i], patbufsize ) )
                panic( allocerrmsg, 3 );
            patternsegs[i] = 0;
        }
    patbufsize = 0;

    /* Set a few variables so that the display will be correct. */
    pattablesize = 0;
    nfilepatterns = 0;
    nchannels = 0;
    ninstruments = 0;
    globalvoldefl = globalvol = 0;
    ticksdefl = ticksperrow = 6;
    bpmdefl = beatspermin = 125;

    /* Internal check:  verify the state of the memory manager.  Program 
       dies if the check fails. */
    if ( checkmem() )
        panic( allocerrmsg, 3 );
}

/* Too much for one file. */
#include "modstuf.i"
