/* PhotoOrganizer - $RCSfile: PhotoImagePanel.java,v $
 * Copyright (C) 1999-2001 Dmitriy Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * $Id: PhotoImagePanel.java,v 1.7 2001/09/22 00:39:40 rogatkin Exp $
 */
package photoorganizer.renderer;

import javax.swing.*;
import java.awt.*;
import java.util.Iterator;
import java.util.ArrayList;
import java.awt.image.ImageObserver;
import java.awt.event.*;
import java.io.File;
import java.awt.Toolkit;
import javax.swing.event.MouseInputAdapter;

import rogatkin.*;
import photoorganizer.formats.*;
import photoorganizer.*;

interface Observable {
    public void update(Icon icon);
}

public class PhotoImagePanel extends JPanel implements ActionListener, Observable, IrdControllable {
    private JScrollPane scroller;
    private Observable  iviewer;
    private ThumbnailsPanel thumbspanel;
    private Controller controller;
    private BasicJpeg jpeg;

	public PhotoImagePanel(Controller controller) {
		this.controller = controller;
		setLayout(new BorderLayout());
		add(scroller=new JScrollPane(new ImageComponent()), BorderLayout.CENTER);
		setMinimumSize(Resources.MIN_PANEL_DIMENSION);
	}

	public void setThumbnailsPanel(ThumbnailsPanel thumbspanel) {
		this.thumbspanel = thumbspanel;
	}
    
	public void update(Icon icon) {
		if (iviewer == null || iviewer == this) {
			if (icon instanceof ImageIcon) 
				((ImageIcon)icon).setImage(getScaled(((ImageIcon)icon).getImage()));

			((Observable)this.scroller.getViewport().getView()).update(icon);
		} else {
			if (iviewer instanceof ImageFrame)
				((ImageFrame)iviewer).setTitle(PhotoOrganizer.PROGRAMNAME+" "+PhotoOrganizer.VERSION+" - "+jpeg.getLocationName());
			iviewer.update(icon);
		}
	}

	public void updateView(AbstractFormat format) {
		if (format instanceof BasicJpeg) {
			this.jpeg = (BasicJpeg)format;
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					Icon ico = PhotoImagePanel.this.jpeg.getIcon();
					if (ico != null)
						update(ico);
				}
				});
		}
		Container c = getParent();
		if (c != null && c instanceof JTabbedPane) {
			((JTabbedPane)c).setSelectedComponent(this);
		}
	}

	public void actionPerformed(ActionEvent a) {
		String cmd = a.getActionCommand();
		if (cmd.equals(Resources.MENU_POPUPWIN)) {
			if (iviewer != null) {
				if (iviewer instanceof ImageFrame)
					return;
				if (iviewer instanceof ImageWindow)
					((Window)iviewer).dispose();
			}
			iviewer = new ImageFrame(PhotoOrganizer.PROGRAMNAME+" "+PhotoOrganizer.VERSION+" - "+jpeg.getLocationName());
		} else if (cmd.equals(Resources.MENU_INTWIN)) {
			if (iviewer != null && iviewer != this) {
				((Window)iviewer).dispose();
				iviewer = null;
			}
		} else if (cmd.equals(Resources.MENU_FULLSCREEN)) {
			if (iviewer != null) {
				if (iviewer instanceof ImageWindow)
					return;
				if (iviewer instanceof ImageFrame)
					((Window)iviewer).dispose();
			}
			iviewer = new ImageWindow();
		} else if (cmd.equals(Resources.MENU_NEXT)) {
			thumbspanel.showNext();
		} else if (cmd.equals(Resources.MENU_PREVIOUS)) {
			thumbspanel.showPrev();
		}
	}
	
	protected Image getScaled(Image image) {
		if (!isScaling())
			return image;
		Dimension d = getSize();
		int w = image.getWidth(this);
		int h = image.getHeight(this);
		if (w <= d.width && h < d.height)
			return image;
		if (w/d.width > h/d.height) {
			return image.getScaledInstance(d.width, h*d.width/w, Image.SCALE_SMOOTH);
		} else 
		return image.getScaledInstance(w*d.height/h, d.height, Image.SCALE_FAST);
		// SCALE_DEFAULT, SCALE_FAST, SCALE_SMOOTH, SCALE_REPLICATE, SCALE_AREA_AVERAGING
	}

	protected boolean isScaling() {
		Serializer s = controller.getSerializer();
		return s.getInt(s.getProperty(AppearanceOptionsTab.SECNAME, AppearanceOptionsTab.FIT_TO_SIZE), 0) == 1;
	}
	
	protected void trigger() {
		if (iviewer != null && iviewer != this) {
			((Window)iviewer).dispose();
			iviewer = null;
		} else
			iviewer = new ImageFrame(PhotoOrganizer.PROGRAMNAME+" "+PhotoOrganizer.VERSION+" - "+jpeg.getLocationName());
	}
	
	// remote controllable
	public String getName() {
		return Resources.COMP_IMAGE_VIEWER;
	}
	
	public Iterator	getKeyMnemonics() {
		ArrayList list = new ArrayList(4);
		list.add(Resources.MENU_FULLSCREEN);
		list.add(Resources.MENU_NEXT);
		list.add(Resources.MENU_PREVIOUS);
		return list.iterator();
	}
	
	public boolean doAction(String keyCode) {
		actionPerformed(new ActionEvent(this, keyCode.hashCode(), keyCode));
		return true;
	}
	
	public void bringOnTop() {
	}

    class ImageComponent extends JLabel implements Observable {
        public ImageComponent() {
            super(new ImageIcon());
            addMouseListener(new MouseInputAdapter() {
				public void mouseClicked(MouseEvent e) {
					int m = e.getModifiers();
					if (e.getClickCount() == 2 && (m & InputEvent.BUTTON1_MASK) > 0) {
						trigger();
					} else if ((m & InputEvent.BUTTON3_MASK) > 0) { // SwingUtilities.isRightMouseButton(e)
						Point p = ((JViewport)getParent()).getViewPosition();
						new ImageRButtonMenu(PhotoImagePanel.this).show(getParent(), e.getX()-p.x, e.getY()-p.y);
					}
				}
            });
			addKeyListener(new KeyAdapter() {
				public void keyReleased(KeyEvent e) {
					switch(e.getKeyCode()) {
					case KeyEvent.VK_PAGE_UP:
					case KeyEvent.VK_SUBTRACT:
						thumbspanel.showPrev();
						break;
					case KeyEvent.VK_PAGE_DOWN:
					case KeyEvent.VK_ADD:
						thumbspanel.showNext();
						break;
					}
				}
				});
        }
        
        public void update(Image image) {
            if (image == null)
                return;
			controller.setWaitCursor(this, true);
			((ImageIcon)getIcon()).setImage(image);
			revalidate();
			repaint();
			Serializer s = controller.getSerializer();
			String o = s.arrayToString(s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.TOOLTIPMASK));
			if (o != null) 
				setToolTipText(new FileNameFormat(o).format(jpeg));
            requestFocus();
        }

        public void update(Icon icon) {
            if (icon == null)
                return;
			controller.setWaitCursor(this, true);
			setIcon(icon);
			revalidate();
			Serializer s = controller.getSerializer();
			String o = s.arrayToString(s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.TOOLTIPMASK));
			if (o != null) 
				setToolTipText(new FileNameFormat(o).format(jpeg));
            requestFocus();
        }

        public void paint(Graphics g) {
            try {
                super.paint(g);
            } catch (NullPointerException e) {
                // work around Sun's VM bug
            }
		    controller.setWaitCursor(this, false);
        }

		class ImageRButtonMenu extends JPopupMenu {
			ImageRButtonMenu(ActionListener listener) {
				JMenuItem item;
				this.add(item = new JMenuItem(Resources.MENU_POPUPWIN));
				item.addActionListener(listener);
				this.add(item = new JMenuItem(Resources.MENU_INTWIN));
				item.addActionListener(listener);
				this.add(item = new JMenuItem(Resources.MENU_FULLSCREEN));
				item.addActionListener(listener);
				this.addSeparator();
				this.add(item = new JMenuItem(Resources.MENU_NEXT));
				item.addActionListener(listener);
				this.add(item = new JMenuItem(Resources.MENU_PREVIOUS));
				item.addActionListener(listener);
			}
		}
    }

	class ImageFrame extends JFrame implements Observable {
		public ImageFrame(String caption) {
			super(caption);
			getContentPane().add(scroller);
			setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
			pack();
			setVisible(true);
			setIconImage(controller.getMainIcon());
			scroller.getViewport().getView().requestFocus();
		}
		
		public void dispose() {
			iviewer = PhotoImagePanel.this;
			PhotoImagePanel.this.add(scroller);
			PhotoImagePanel.this.revalidate();
			super.dispose();
		}

		public void update(Icon icon) {
			((Observable)scroller.getViewport().getView()).update(icon);
		}
	}

	class ImageWindow extends JWindow implements Observable {
		public ImageWindow() {
			getContentPane().add(scroller);
			pack();
			setVisible(true);
			// try Toolkit.getDesktopProperty("win.toolbar.height");
			//String [] props = (String[])Toolkit.getDefaultToolkit().getDesktopProperty("win.propNames");
			//for (int i=0;i<props.length;i++)
			//System.err.println("Desktop prop["+i+"] "+props[i]);
			setSize(Toolkit.getDefaultToolkit().getScreenSize());
			scroller.getViewport().getView().requestFocus();
		}

		public void update(Icon icon) {
			((Observable)scroller.getViewport().getView()).update(icon);
		}

		public void dispose() {
			iviewer = PhotoImagePanel.this;
			PhotoImagePanel.this.add(scroller);
			PhotoImagePanel.this.revalidate();
			super.dispose();
		}
	}
}


