/*
 *   @(#) MPEG-I Video Decoder 1.0 Demicron (demicron@demicron.com)
 *
 *   AD_Decoder.java   2002-08-20
 *
 *   Copyright (C) 2002  Demicron
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */




import java.io.IOException;



/**
 * The <code>Decoder</code> class encapsulates the details of
 * decoding an MPEG audio frame.
 *
 */

public class AD_Decoder implements AD_DecoderErrors
{
	static private final Params DEFAULT_PARAMS = new Params();

	/**
	 * The AD_Obuffer instance that will receive the decoded
	 * PCM samples.
	 */
	private AD_Obuffer			output;

	/**
	 * Synthesis filter for the left channel.
	 */
	private AD_SynthesisFilter			filter1;

	/**
	 * Sythesis filter for the right channel.
	 */
	private AD_SynthesisFilter			filter2;

	/**
	 * The decoder used to decode layer III frames.
	 */
	// private LayerIIIDecoder			l3decoder;
	private AD_LayerIIDecoder			l2decoder;
	private AD_LayerIDecoder			l1decoder;

	private int						outputFrequency;
	private int						outputChannels;

	private AD_Equalizer				equalizer = new AD_Equalizer();

	private Params					params;

	private boolean					initialized;


	/**
	 * Creates a new <code>Decoder</code> instance with default
	 * parameters.
	 */

	public AD_Decoder() {
		this(null);
	}

	/**
	 * Creates a new <code>Decoder</code> instance with default
	 * parameters.
	 *
	 * @param params	The <code>Params</code> instance that describes
	 *					the customizable aspects of the decoder.
	 */
	public AD_Decoder(Params params0) {
		if (params0 == null)
			params0 = DEFAULT_PARAMS;

		params = params0;

		AD_Equalizer eq = params.getInitialAD_EqualizerSettings();
		if (eq!=null) {
			equalizer.setFrom(eq);
		}
	}

	static public Params getDefaultParams() {
		return (Params)DEFAULT_PARAMS.clone();
	}

	public void setEqualizer(AD_Equalizer eq) {
		if (eq==null)
			eq = AD_Equalizer.PASS_THRU_EQ;

		equalizer.setFrom(eq);

		float[] factors = equalizer.getBandFactors();
		if (filter1!=null)
			filter1.setEQ(factors);

		if (filter2!=null)
			filter2.setEQ(factors);
	}

	/**
	 * Decodes one frame from an MPEG audio bitstream.
	 *
	 * @param header		The header describing the frame to decode.
	 * @param bitstream		The bistream that provides the bits for the body of the frame.
	 *
	 * @return An AD_Obuffer containing the decoded samples.
	 */
	public AD_Obuffer decodeFrame(AD_Header header, BufferedBitStream stream) throws InterruptedException, AD_DecoderException {
		if (!initialized) {
			initialize(header);
		}

		int layer = header.layer();

		output.clear_buffer();

		AD_FrameDecoder decoder = retrieveDecoder(header, stream, layer);


		// BYOHack:
		// System.out.println("layer: " + layer);
		// System.out.println("num subbands:" + header.number_of_subbands());
		decoder.decodeFrame();

		output.write_buffer(1);

		return output;
	}

	/**
	 * Changes the output buffer. This will take effect the next time
	 * decodeFrame() is called.
	 */
	public void setOutputBuffer(AD_Obuffer out)
	{
		output = out;
	}

	/**
	 * Retrieves the sample frequency of the PCM samples output
	 * by this decoder. This typically corresponds to the sample
	 * rate encoded in the MPEG audio stream.
	 *
	 * @param the sample rate (in Hz) of the samples written to the
	 *		output buffer when decoding.
	 */
	public int getOutputFrequency()
	{
		return outputFrequency;
	}

	/**
	 * Retrieves the number of channels of PCM samples output by
	 * this decoder. This usually corresponds to the number of
	 * channels in the MPEG audio stream, although it may differ.
	 *
	 * @return The number of output channels in the decoded samples: 1
	 *		for mono, or 2 for stereo.
	 *
	 */
	public int getOutputChannels()
	{
		return outputChannels;
	}

	/**
	 * Retrieves the maximum number of samples that will be written to
	 * the output buffer when one frame is decoded. This can be used to
	 * help calculate the size of other buffers whose size is based upon
	 * the number of samples written to the output buffer. NB: this is
	 * an upper bound and fewer samples may actually be written, depending
	 * upon the sample rate and number of channels.
	 *
	 * @return The maximum number of samples that are written to the
	 *		output buffer when decoding a single frame of MPEG audio.
	 */
	public int getOutputBlockSize()
	{
		return output.OBUFFERSIZE;
	}


	protected AD_DecoderException newDecoderException(int errorcode)
	{
		return new AD_DecoderException(errorcode, null);
	}

	protected AD_DecoderException newDecoderException(int errorcode, Throwable throwable)
	{
		return new AD_DecoderException(errorcode, throwable);
	}

	protected AD_FrameDecoder retrieveDecoder(AD_Header header, BufferedBitStream stream, int layer)
		throws AD_DecoderException
	{
		AD_FrameDecoder decoder = null;

		// REVIEW: allow channel output selection type
		// (LEFT, RIGHT, BOTH, DOWNMIX)
		switch (layer) {
		case 3:
			// Layer III support temorarily removed (saves space in jar)
			throw newDecoderException(UNSUPPORTED_LAYER, null);
			/*
			if (l3decoder == null) {
				l3decoder = new LayerIIIDecoder(stream,
												header, filter1, filter2,
												output, AD_OutputChannels.BOTH_CHANNELS);
			}
			decoder = l3decoder;
			break;
			*/
		case 2:
			if (l2decoder == null) {
				l2decoder = new AD_LayerIIDecoder();
				l2decoder.create(stream,
								 header, filter1, filter2,
								 output, AD_OutputChannels.BOTH_CHANNELS);
			}
			decoder = l2decoder;
			break;
		case 1:
			if (l1decoder == null) {
				l1decoder = new AD_LayerIDecoder();
				l1decoder.create(stream,
								 header, filter1, filter2,
								 output, AD_OutputChannels.BOTH_CHANNELS);
			}
			decoder = l1decoder;
			break;
		}

		if (decoder == null) {
			throw newDecoderException(UNSUPPORTED_LAYER, null);
		}

		return decoder;
	}

	private void initialize(AD_Header header)
		throws AD_DecoderException
	{

		// REVIEW: allow customizable scale factor
		float scalefactor = 32700.0f;

		int mode = header.mode();
		int layer = header.layer();
		int channels = mode==AD_Header.SINGLE_CHANNEL ? 1 : 2;

		// set up output buffer if not set up by client.
		if (output==null)
			output = new AD_Obuffer(header.frequency(), channels);

		float[] factors = equalizer.getBandFactors();
		filter1 = new AD_SynthesisFilter(0, scalefactor, factors, header.sample_frequency(),
									  params.getDownconvert());

		// REVIEW: allow mono output for stereo
		if (channels==2)
			filter2 = new AD_SynthesisFilter(1, scalefactor, factors, header.sample_frequency(),
										  params.getDownconvert());

		outputChannels = channels;
		outputFrequency = header.frequency();

		initialized = true;
	}

	public Params getParams() {
		return params;
	}

	/**
	 * The <code>Params</code> class presents the customizable
	 * aspects of the decoder.
	 * <p>
	 * Instances of this class are not thread safe.
	 */
	public static class Params implements Cloneable
	{
		private AD_OutputChannels	outputChannels = AD_OutputChannels.BOTH;
		private AD_Equalizer		equalizer = new AD_Equalizer();
		private boolean			downConvert = false;

		public Params() { }

		public Object clone() {
			try {
				return super.clone();
			}
			catch (CloneNotSupportedException ex) {
				throw new InternalError(this + ": " + ex);
			}
		}

		public void setAD_OutputChannels(AD_OutputChannels out) {
			if (out == null)
				throw new NullPointerException("out");
			outputChannels = out;
		}

		public AD_OutputChannels getAD_OutputChannels() {
			return outputChannels;
		}

		// Output converts to 8khz ulaw
		public void setDownconvert() {
			downConvert = true;
		}

		public boolean getDownconvert() {
			return downConvert;
		}

		/**
		 * Retrieves the equalizer settings that the decoder's equalizer
		 * will be initialized from.
		 * <p>
		 * The <code>AD_Equalizer</code> instance returned
		 * cannot be changed in real time to affect the
		 * decoder output as it is used only to initialize the decoders
		 * EQ settings. To affect the decoder's output in realtime,
		 * use the AD_Equalizer returned from the getAD_Equalizer() method on
		 * the decoder.
		 *
		 * @return	The <code>AD_Equalizer</code> used to initialize the
		 *			EQ settings of the decoder.
		 */
		public AD_Equalizer getInitialAD_EqualizerSettings() {
			return equalizer;
		}

	};
}
