/* PhotoOrganizer $RCSfile: JTreeTable.java,v $
 * Copyright (C) 1999-2001 Dmitriy Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *  $Id: JTreeTable.java,v 1.11 2001/08/07 08:49:17 rogatkin Exp $
 */
/*
 * Copyright 1997, 1998 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 */
package photoorganizer.directory;

import java.util.EventObject;
import java.util.Vector;
import java.util.Iterator;
import java.util.Arrays;
import java.io.File;
import java.io.IOException;
import java.awt.Dimension;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Container;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.table.*;
import javax.swing.event.*;

import rogatkin.*;
import photoorganizer.renderer.*;
import photoorganizer.*;
import photoorganizer.formats.*;
import photoorganizer.media.*;
/**
 * This code is based on an example of Philip Milne, and Scott Violet
 */
// TODO: implement refresh
public class JTreeTable extends JTable implements ActionListener, Persistable, IrdControllable {
    final static String SECNAME = "JTreeTable";
    final static String COLWIDTH = "ColumnWidthes";
    final static String DRIVE = "Drive";
    
    protected TreeTableCellRenderer tree;
    private PhotoImagePanel imagepanel;
    private ThumbnailsPanel thumbnailspanel;
    private PhotoCollectionPanel collectionpanel;
    private AlbumPane albumpanel;
    private Controller controller;
    
	public JTreeTable(TreeTableModel treeTableModel, Controller controller) {
		this.controller = controller;
		this.imagepanel = (PhotoImagePanel)controller.component(Controller.COMP_IMAGEPANEL);
		thumbnailspanel = (ThumbnailsPanel)controller.component(Controller.COMP_THUMBPANEL);
		collectionpanel = (PhotoCollectionPanel)controller.component(Controller.COMP_COLLECTION);
		albumpanel      = (AlbumPane)controller.component(Controller.COMP_ALBUMPANEL);
		// Create the tree. It will be used as a renderer and editor. 
		tree = new TreeTableCellRenderer(treeTableModel); 
		
		// Install a tableModel representing the visible rows in the tree. 
		super.setModel(new TreeTableModelAdapter(treeTableModel, tree));
		
		// Force the JTable and JTree to share their row selection models. 
		ListToTreeSelectionModelWrapper selectionWrapper = new 
			ListToTreeSelectionModelWrapper();
		tree.setSelectionModel(selectionWrapper);
		setSelectionModel(selectionWrapper.getListSelectionModel()); 
		
		getSelectionModel().addListSelectionListener(new ListSelectionListener() {
            public void valueChanged(ListSelectionEvent e) {
                if (isRowSelected(e.getLastIndex()) || isRowSelected(e.getFirstIndex())) {
                    FileNode[] nodes = getSelectedNodes();
					
					if (nodes != null) {
						boolean wasDir = false;
						boolean wasImage = false;
						boolean wasMP3 = false;
						boolean wasFile = false;
						AbstractFormat []formats = new AbstractFormat[nodes.length];
						for (int i=0; i<nodes.length; i++) {							
							formats[i] = nodes[i].getFormat();
							wasImage |= formats[i] instanceof BasicJpeg;
							wasMP3 |= formats[i] instanceof MP3;
							wasDir |= nodes[i].getFile().isDirectory();
							wasFile |= nodes[i].getFile().isFile();
						}
						JTreeTable.this.setCursor(Controller.WAIT_CURSOR);
						if (wasDir /*&& showFolderContent*/) {
							Vector targetList = new Vector(10);
							buildFormatList(nodes, targetList, true);
							JTreeTable.this.controller.getUiUpdater().notify(thumbnailspanel.updateMedias((AbstractFormat[])targetList.toArray(new AbstractFormat[targetList.size()])), UiUpdater.FILE_SELECTED);
						} else
							JTreeTable.this.controller.getUiUpdater().notify(thumbnailspanel.updateMedias(formats), UiUpdater.FILE_SELECTED);
						JTreeTable.this.controller.updateCaption((formats[0]!=null?formats[0].toString():nodes[0].toString())+(formats.length>1?"...":""));
						JTreeTable.this.controller.getUiUpdater().notify(wasDir, UiUpdater.DIRECTORY_SELECTED);
						JTreeTable.this.controller.getUiUpdater().notify(wasImage, UiUpdater.GRAPHICS_FILE_SELECTED);
						JTreeTable.this.controller.getUiUpdater().notify(wasMP3, UiUpdater.MEDIA_FILE_SELECTED);
						JTreeTable.this.controller.getUiUpdater().notify(wasFile, UiUpdater.FILE_SELECTED);
						JTreeTable.this.setCursor(Controller.DEFAULT_CURSOR);
					}
				} else {
					JTreeTable.this.controller.getUiUpdater().notify(false, UiUpdater.FILE_SELECTED);
					JTreeTable.this.controller.getUiUpdater().notify(false, UiUpdater.DIRECTORY_SELECTED);
					JTreeTable.this.controller.getUiUpdater().notify(false, UiUpdater.GRAPHICS_FILE_SELECTED);
					JTreeTable.this.controller.getUiUpdater().notify(false, UiUpdater.MEDIA_FILE_SELECTED);
				}
            }
        });
	
        // Install the tree editor renderer and editor. 
		setDefaultRenderer(TreeTableModel.class, tree); 
		setDefaultEditor(TreeTableModel.class, new TreeTableCellEditor());  
		setShowGrid(false); // No grid.
		setIntercellSpacing(new Dimension(0, 0)); // No intercell spacing
		// And update the height of the trees row to match that of the table.
		if (tree.getRowHeight() < 1) {
			setRowHeight(18); // Metal looks better like this.
		}
	
        addMouseListener(new MouseInputAdapter() {
            public void mouseClicked(MouseEvent e) {
                if ((e.getModifiers() & InputEvent.BUTTON3_MASK) > 0) {
                    Point p = ((JViewport)getParent()).getViewPosition();
                    FastMenu fm = new FastMenu(JTreeTable.this, JTreeTable.this.controller);
                    JMenuItem item;
                    fm.add(new JPopupMenu.Separator());
					fm.add(item = new JMenuItem(Resources.MENU_COMMENT));
					item.addActionListener(JTreeTable.this);
					item.setEnabled(JTreeTable.this.controller.getUiUpdater().isEnabled(UiUpdater.MEDIA_FILE_SELECTED));
					fm.add(new JPopupMenu.Separator());
                    fm.add(item = new JMenuItem(Resources.MENU_DRIVE_SEL));
                    item.addActionListener(JTreeTable.this);
                    fm.show(getParent(), e.getX()-p.x, e.getY()-p.y);
                } else if (e.getClickCount() == 2) {
                    actionPerformed(new ActionEvent(this, 0, Resources.MENU_SHOW));
                }
            }
        });
        setMinimumSize(Resources.MIN_PANEL_DIMENSION);
    }
    
	FileNode[] getSelectedNodes() {
		TreePath []tps = tree.getSelectionPaths();
		if (tps == null || tps.length == 0)
			return null;
		FileNode []result = new FileNode[tps.length];
		int ri = 0;
		for (int n=0; n<tps.length; n++) {
			Object ps = tps[n].getLastPathComponent();
			if (ps instanceof FileNode)
				result[ri++] = (FileNode)ps;
		}
		return result;
	}
					  
	File[] getSelectedFiles() {
		TreePath []tps = tree.getSelectionPaths();
		if (tps == null || tps.length == 0)
			return null;
		File []result = new File[tps.length];
		int ri = 0;
		for (int n=0; n<tps.length; n++) {
			Object ps = tps[n].getLastPathComponent();
			if (ps instanceof FileNode)
				result[ri++] = ((FileNode)ps).getFile();
		}
		// if (ri < tps.length) squeeze the array
		return result;
	}
    
	public void fireDriveChanged() {
		FileSystemModel treemodel = (FileSystemModel)tree.getModel();
		File[] drivers = treemodel.getRoots();
		// TODO: should have been rewritten, currently it removes first
		// and other null drives for UNIX
		Vector real_drivers = new Vector(drivers.length);
		for (int i = 0; i < drivers.length; i++)
			if (drivers[i] != null)
				real_drivers.addElement(drivers[i]);
		drivers = new File[real_drivers.size()];
		real_drivers.copyInto(drivers);
		Object selectedValue = JOptionPane.showInputDialog(this, 
														   Resources.LABEL_CHOOSE_DRIVE, Resources.TITLE_CHANGE_DRIVE,
														   JOptionPane.INFORMATION_MESSAGE, null,
														   drivers, drivers[0]);
		if (selectedValue == null)
			return;
		treemodel.setRoot(new FileNode((File)selectedValue));
		treemodel.fireTreeStructureChanged(this, new Object[] {selectedValue}, null, null);
	}
    
	public void actionPerformed(ActionEvent a) {
		String cmd = a.getActionCommand();
		if (cmd.equals(Resources.MENU_DRIVE_SEL) || 
			(a.getSource() instanceof JButton && ((JButton)a.getSource()).getToolTipText().equals(Resources.MENU_DRIVE_SEL))) {
			fireDriveChanged();
			return;
		}
		final File[] files = getSelectedFiles();
		if (files == null)
			return;
		TreePath tpchanged = tree.getSelectionPath().getParentPath();
		
		if (cmd.equals(Resources.MENU_PROPERTIES)) {
			AbstractFormat format = getFirstSelectedMedia();
			if (format != null)
				PropertiesPanel.showProperties(format, controller);
			return;
		} else if (cmd.equals(Resources.MENU_ADDTOCOLLECT) || // from tool bar
			(a.getSource() instanceof JButton && ((JButton)a.getSource()).getToolTipText().equals(Resources.MENU_ADDTOCOLLECT))) {
			//collectionpanel.add((File[])new PlaybackRequest(files).buildList(controller));
			collectionpanel.add(new PlaybackRequest(files, controller.getSerializer()));
			return;
		} else if (cmd.equals(Resources.MENU_ADDTOALBUM)) {
			AlbumPane albumpane = (AlbumPane)controller.component(Controller.COMP_ALBUMPANEL);
			AlbumSelectionDialog asd = albumpane.getSelectionDialog();
			asd.setTitle(Resources.TITLE_SELECT_ALBUM+":"+files[0]+(files.length == 1?"":"..."));
			asd.setVisible(true);
			TreePath[] tps = asd.getSelectedAlbums();
			if (tps != null) {
				Vector totalList = new Vector(); // although I'd prefer array list
				buildFormatList(files, totalList, true);				
				albumpanel.addToAlbum((AbstractFormat[])totalList.toArray(new AbstractFormat[totalList.size()]), tps);
			}
		} else if (cmd.equals(Resources.MENU_RENAME)) {
			FileNode[] nodes = getSelectedNodes();
			for (int i=0; i<nodes.length; i++) {
				Object value = nodes[i].getFile().getName();
				AbstractFormat format = nodes[i].getFormat();
				Serializer s = controller.getSerializer();
				if (format != null && format.isValid()) {
					if (Serializer.getInt(s.getProperty(RenameOptionsTab.SECNAME, RenameOptionsTab.ASKEDIT), 0) == 0)
						format.renameTo(new File(nodes[i].getFile().getParent(), FileNameFormat.makeValidPathName(RenameOptionsTab.getNewName(format, controller))));
					else {
						value = nodes[i].getFile().getParent()+File.separatorChar+
								FileNameFormat.makeValidPathName(RenameOptionsTab.getNewName(format, controller));
						value = JOptionPane.showInputDialog(this,
															Resources.LABEL_NEW_NAME, Resources.TITLE_RENAME,
															JOptionPane.QUESTION_MESSAGE, null, null, value);
						if (value != null)
							format.renameTo(new File(value.toString()));
					}
				} else {
					value = JOptionPane.showInputDialog(this,
														Resources.LABEL_NEW_NAME, Resources.TITLE_RENAME,
														JOptionPane.QUESTION_MESSAGE, null, null, value);
					if (value != null)
						nodes[i].getFile().renameTo(new File(value.toString()));
				}
			}
		} else if (cmd.equals(Resources.MENU_DELETE)) {
			for (int i=0; i<files.length; i++)
				if (JOptionPane.YES_OPTION == JOptionPane.showConfirmDialog(this,
					files[i].getName()+Resources.LABEL_CONFIRM_DEL, Resources.TITLE_DELETE,
					JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE))
					if (!files[i].delete())
						((StatusBar)controller.component(Controller.COMP_STATUSBAR)).flashInfo("Cannot delete "+files[i], true);
						
		} else if (cmd.equals(Resources.MENU_PRINT)) {
			controller.print(files);
			return;
		} else if (cmd.equals(Resources.MENU_SHOW) ||
			(a.getSource() instanceof JButton && ((JButton)a.getSource()).getToolTipText().equals(Resources.MENU_SHOW))) {
			AbstractFormat format = getFirstSelectedMedia();
			if(format != null && format.isValid() && format instanceof BasicJpeg)  {
				imagepanel.updateView((BasicJpeg)format);
				controller.updateCaption(files[0].toString());
			} else
				new PlaybackRequest(files).playList(controller);
					
			return;
		} else if (cmd.equals(Resources.MENU_COMMENT)) {
			AbstractFormat format = getFirstSelectedMedia(); // TODO: maybe loop over all
			if (format instanceof MP3)
				Id3TagEditor.editTag(controller, format);
		}
		((AbstractTreeTableModel)tree.getModel()).fireTreeStructureChanged(this, tpchanged.getPath(), null, null);
	}
	
	protected AbstractFormat getFirstSelectedMedia() {
		FileNode[] nodes = getSelectedNodes();
		if (nodes != null) {
			for (int i = 0; i < nodes.length; i++) {
				AbstractFormat result = nodes[i].getFormat();
				if (result != null && result.isValid())
					return result;
			}
		}
		return null;	
	}
    
    /* Workaround for BasicTableUI anomaly. Make sure the UI never tries to 
    * paint the editor. The UI currently uses different techniques to 
    * paint the renderers and editors and overriding setBounds() below 
    * is not the right thing to do for an editor. Returning -1 for the 
    * editing row in this case, ensures the editor is never painted. 
    */
    public int getEditingRow() {
        return (getColumnClass(editingColumn) == TreeTableModel.class) ? -1 : editingRow;  
    }
    /**
    * Overridden to message super and forward the method to the tree.
    * Since the tree is not actually in the component hieachy it will
    * never receive this unless we forward it in this manner.
    */
	public void updateUI() {
		super.updateUI();
		if(tree != null) {
			tree.updateUI();
		}
	// Use the tree's default foreground and background colors in the
	// table. 
        LookAndFeel.installColorsAndFont(this, "Tree.background",
	    "Tree.foreground", "Tree.font");
    }
    
    /**
    * Overridden to pass the new rowHeight to the tree.
    */
	public void setRowHeight(int rowHeight) { 
		super.setRowHeight(rowHeight); 
		if (tree != null && tree.getRowHeight() != rowHeight) {
			tree.setRowHeight(getRowHeight()); 
		}
	}
	
	public void save() {
		Serializer s = controller.getSerializer();
		Integer[] w = new Integer[getColumnCount()];
		for (int i = 0; i < w.length; i++) {
			try {
				w[i] = new Integer(getColumn(getColumnName(i)).getWidth());
			} catch(IllegalArgumentException iae) {
				w[i] = Resources.I_NO;;
			}
		}
		s.setProperty(SECNAME, COLWIDTH, w);
		FileSystemModel treemodel = (FileSystemModel)tree.getModel();
		FileNode root = (FileNode)treemodel.getRoot();
		s.setProperty(SECNAME, DRIVE, root);
	}
	
	public void load() {
		Serializer s = controller.getSerializer();
		Object[] w = (Object[])s.getProperty(SECNAME, COLWIDTH);
		if (w == null)
			return;
		for (int i = 0; i < w.length; i++) {
			getColumn(getColumnName(i)).setWidth(((Integer)w[i]).intValue());
			getColumn(getColumnName(i)).setPreferredWidth(((Integer)w[i]).intValue());
		}  //setAutoResizeMode(AUTO_RESIZE_SUBSEQUENT_COLUMNS/*AUTO_RESIZE_OFF*/);
		Object drive = s.getProperty(SECNAME, DRIVE); // can be string or filenode
		if (drive != null) {
			FileSystemModel fsm = (FileSystemModel)tree.getModel();
			File r = new File(drive.toString());
			if (!r.exists())
				r = fsm.findFirstDrive();
			fsm.setRoot(new FileNode(r));
			fsm.fireTreeStructureChanged(this, new Object[] {r}, null, null);
			revalidate();
			repaint();
		}
	}
	
	/**
	 * Returns the tree that is being shared between the model.
	 */
	public JTree getTree() {
		return tree;
	}

	//
	// remote controllable
	//
	public String getName() {
		return Resources.COMP_BROWSER;
	}

	public String toString() {
		return getName();
	}

	public Iterator	getKeyMnemonics() {
		return Arrays.asList(new Object[0]).iterator();
	}
	
	public boolean doAction(String keyCode) {
		return false;
	}
	
	public void bringOnTop() {
		controller.selectTab(Resources.TAB_BROWSE);
	}

	// 
	// The renderer used to display the tree nodes, a JTree.  
	//
	public class TreeTableCellRenderer extends JTree implements TableCellRenderer {
		
		protected int visibleRow;
		
		public TreeTableCellRenderer(TreeModel model) { 
			super(model); 
		}
		
		/**
		 * updateUI is overridden to set the colors of the Tree's renderer
		 * to match that of the table.
		 */
		public void updateUI() {
			super.updateUI();
			// Make the tree's cell renderer use the table's cell selection
			// colors. 
			TreeCellRenderer tcr = getCellRenderer();
			if (tcr instanceof DefaultTreeCellRenderer) {
				DefaultTreeCellRenderer dtcr = ((DefaultTreeCellRenderer)tcr); 
				// For 1.1 uncomment this, 1.2 has a bug that will cause an
				// exception to be thrown if the border selection color is
				// null.
				// dtcr.setBorderSelectionColor(null);
				dtcr.setTextSelectionColor(UIManager.getColor
					("Table.selectionForeground"));
				dtcr.setBackgroundSelectionColor(UIManager.getColor
					("Table.selectionBackground"));
			}
		}
		
		/**
		 * Sets the row height of the tree, and forwards the row height to
		 * the table.
		 */
		public void setRowHeight(int rowHeight) { 
			if (rowHeight > 0) {
				super.setRowHeight(rowHeight); 
				if (JTreeTable.this != null &&
					JTreeTable.this.getRowHeight() != rowHeight) {
					JTreeTable.this.setRowHeight(getRowHeight()); 
				}
			}
		}
		
		public void setBounds(int x, int y, int w, int h) {
			super.setBounds(x, 0, w, JTreeTable.this.getHeight());
		}
		
		public void paint(Graphics g) {
			g.translate(0, -visibleRow * getRowHeight());
			super.paint(g);
		}
		
		public Component getTableCellRendererComponent(JTable table,
													   Object value,
													   boolean isSelected,
													   boolean hasFocus,
													   int row, int column) {
			if(isSelected)
				setBackground(table.getSelectionBackground());
			else
				setBackground(table.getBackground());
			
			visibleRow = row;
			return this;
		}
	}
	
	// 
	// The editor used to interact with tree nodes, a JTree.  
	//
	public class TreeTableCellEditor extends AbstractCellEditor implements TableCellEditor {
		public Component getTableCellEditorComponent(JTable table, Object value,
													 boolean isSelected, int r, int c) {
			return tree;
		}
	}
	/**
	 * Overridden to return false, and if the event is a mouse event
	 * it is forwarded to the tree.<p>
	 * The behavior for this is debatable, and should really be offered
	 * as a property. By returning false, all keyboard actions are
	 * implemented in terms of the table. By returning true, the
	 * tree would get a chance to do something with the keyboard
	 * events. For the most part this is ok. But for certain keys,
	 * such as left/right, the tree will expand/collapse where as
	 * the table focus should really move to a different column. Page
	 * up/down should also be implemented in terms of the table.
	 * By returning false this also has the added benefit that clicking
	 * outside of the bounds of the tree node, but still in the tree
	 * column will select the row, whereas if this returned true
	 * that wouldn't be the case.
	 * <p>By returning false we are also enforcing the policy that
	 * the tree will never be editable (at least by a key sequence).
	 */
	public boolean isCellEditable(EventObject e) {
		if (e instanceof MouseEvent) {
			for (int counter = getColumnCount() - 1; counter >= 0;
				 counter--) {
				if (getColumnClass(counter) == TreeTableModel.class) {
					MouseEvent me = (MouseEvent)e;
					MouseEvent newME = new MouseEvent(tree, me.getID(),
													  me.getWhen(), me.getModifiers(),
													  me.getX() - getCellRect(0, counter, true).x,
													  me.getY(), me.getClickCount(),
													  me.isPopupTrigger());
					tree.dispatchEvent(newME);
					break;
				}
			}
		}
		return false;
	}
		
	protected void buildFormatList(FileNode[] nodes, Vector targetList, boolean recursively) {
		for (int i=0; i<nodes.length; i++) {
			FileNode[] children = (FileNode[])nodes[i].getChildren();
			if (recursively && children != null) {
				buildFormatList(children, targetList, false);
			} else {
				targetList.addElement(nodes[i].getFormat());
			}
		}
	}

	protected void buildFormatList(File[] files, Vector targetList, boolean recursively) {
		for (int i=0; i<files.length; i++) {
			if (recursively && files[i].isDirectory()) {
					buildFormatList(files[i].listFiles(/*filter?*/), targetList, recursively);
			} else {
				AbstractFormat format = MediaFormatFactory. createMediaFormat(files[i]);
				if (format != null && format.isValid())
					targetList.addElement(format);
			}
		}
	}		
		
	/**
	 * ListToTreeSelectionModelWrapper extends DefaultTreeSelectionModel
	 * to listen for changes in the ListSelectionModel it maintains. Once
	 * a change in the ListSelectionModel happens, the paths are updated
	 * in the DefaultTreeSelectionModel.
	 */
	class ListToTreeSelectionModelWrapper extends DefaultTreeSelectionModel { 
		/** Set to true when we are updating the ListSelectionModel. */
		protected boolean         updatingListSelectionModel;
		
		public ListToTreeSelectionModelWrapper() {
			super();
			getListSelectionModel().addListSelectionListener
				(createListSelectionListener());
		}
		
		/**
		 * Returns the list selection model. ListToTreeSelectionModelWrapper
		 * listens for changes to this model and updates the selected paths
		 * accordingly.
		 */
		ListSelectionModel getListSelectionModel() {
			return listSelectionModel; 
		}
		
		/**
		 * This is overridden to set <code>updatingListSelectionModel</code>
		 * and message super. This is the only place DefaultTreeSelectionModel
		 * alters the ListSelectionModel.
		 */
		public void resetRowSelection() {
			if(!updatingListSelectionModel) {
				updatingListSelectionModel = true;
				try {
					super.resetRowSelection();
				}
				finally {
					updatingListSelectionModel = false;
				}
			}
			// Notice how we don't message super if
			// updatingListSelectionModel is true. If
			// updatingListSelectionModel is true, it implies the
			// ListSelectionModel has already been updated and the
			// paths are the only thing that needs to be updated.
		}
		
		/**
		 * Creates and returns an instance of ListSelectionHandler.
		 */
		protected ListSelectionListener createListSelectionListener() {
			return new ListSelectionHandler();
		}
		
		/**
		 * If <code>updatingListSelectionModel</code> is false, this will
		 * reset the selected paths from the selected rows in the list
		 * selection model.
		 */
		protected void updateSelectedPathsFromSelectedRows() {
			if(!updatingListSelectionModel) {
				updatingListSelectionModel = true;
				try {
					// This is way expensive, ListSelectionModel needs an
					// enumerator for iterating.
					int        min = listSelectionModel.getMinSelectionIndex();
					int        max = listSelectionModel.getMaxSelectionIndex();
					
					clearSelection();
					if(min != -1 && max != -1) {
						for(int counter = min; counter <= max; counter++) {
							if(listSelectionModel.isSelectedIndex(counter)) {
								TreePath selPath = tree.getPathForRow
									(counter);
								
								if(selPath != null) {
									addSelectionPath(selPath);
								}
							}
						}
					}
				}
				finally {
					updatingListSelectionModel = false;
				}
			}
		}
		
	/**
	 * Class responsible for calling updateSelectedPathsFromSelectedRows
	 * when the selection of the list changse.
	 */
		class ListSelectionHandler implements ListSelectionListener {
			public void valueChanged(ListSelectionEvent e) {
				updateSelectedPathsFromSelectedRows();
			}
		}
	}
}
