// Misc.cpp : miscellaneous routines
//
// Copyright (c) 1999-2000 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

/*

MODULE NAME:    NUMSCALE - Scale value for readability

Usage:          CString NumScale(double val)

Returns:        A string containing the scaled number (see below)
  
Parameters:     val = a value to be scaled

Description:    Converts a number to a string which is more easily read.
                At most 5 significant digits are generated (and hence
                there is a loss of precision) to aid readability.  If
                scaling is necessary a metric modifier is appended to
                the string.  Eg:

                0.0 becomes "0 "
                0.1 becomes "0 "
                1.0 becomes "1 "
                100.0 becomes "100 "
                99999.0 becomes "99,999 "
                100000.0 becomes "100 K"
                99999000.0 becomes "99,999 K"
                100000000.0 becomes "100 M"
                99999000000.0 becomes "99,999 M"
                100000000000.0 becomes "100 G"
                99999000000000.0 becomes "99,999 G"
                100000000000000.0 becomes "100 T"
                99999000000000000.0 becomes "99,999 T"
                100000000000000000.0 returns "1.000e16 "

                Note that scaling values between 1 and -1  (milli, micro, etc)
                are not supported (produce 0).  Negative numbers are scaled
                identically to positive ones but preceded by a minus sign.
                Numbers are rounded to the nearest whole number.

------------------------------------------------------
MODULE NAME:    ADDCOMMAS - Add commas to string of digits

Usage:          void AddCommas(CString &str)

Parameters:     str = string to add commas to

Description:    Adds commas to a string of digits.  The addition of commas
                stops when a character which is not a digit (or comma or
                space) is found and the rest of the string is just appended
                to the result.

                The number may have a leading plus or minus sign and contain
                commas and spaces.  The sign is preserved but any existing commas
                and spaces are removed, unless they occur after the comma
                addition has stopped.

                eg. "" becomes ""
                eg. "1" becomes "1"
                eg. "A,B C" becomes "A,B C"
                eg. "1234567" becomes "1,234,567"
                eg. " - 1 , 234" becomes "-1,234"
                eg. "+1234 ABC" becomes "+1,234 ABC"
                eg. "1234 - 1" becomes "1,234 - 1"
------------------------------------------------------
MODULE NAME:    ADDSPACES - Add spaces to string of hex digits

Usage:          void AddSpaces(CString &str)

Parameters:     str = string to add spaces to

Description:    Like AddCommas() above but adds spaces to a hex number rather
                than commas to a decimal number.

*/
#endif

#include "stdafx.h"
#include "HexEdit.h"
#include "EmailDlg.h"
#include <ctype.h>
#include <assert.h>
#include <locale.h>

#include <MAPI.h>       // for MAPI constants
#include <direct.h>             // For _getdrive()

#include "misc.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

CString NumScale(double val)
{
    CString retval;
    char unitmod = '\0';
    char sep_char = ',';                // Used to separate groups of digits
    struct lconv *plconv = localeconv(); // Pointer to locale info

    // Note we're always grouping digits into thousands (groups of 3) but if
    // native locale also used groups of 3 then use its thousands separator.
    if (strlen(plconv->thousands_sep) == 1 && *plconv->grouping == '\3')
        sep_char = *plconv->thousands_sep;

    // Allow for negative values (may give "-0")
    if (val < 0.0)
    {
        val = -val;
        retval = "-";
    }
    else
        retval = "";

    // If too big just print in scientific notation
    if (val >= 99999.5e12)
    {
        retval.Format("%.2e ", val);
        return retval;
    }

    if (val >= 99999.5)
    {
        unitmod = 'K';
        val = val / 1000.0;
    }
    if (val >= 99999.5)
    {
        unitmod = 'M';
        val = val / 1000.0;
    }
    if (val >= 99999.5)
    {
        unitmod = 'G';
        val = val / 1000.0;
    }
    if (val >= 99999.5)
    {
        unitmod = 'T';
        val = val / 1000.0;
    }

    val += 0.5;         /* Round rather than truncate */

    if (val >= 1000.0)
        retval.Format("%ld%c%03ld %c", (long)val/1000L, sep_char, (long)val%1000L, unitmod);
    else
        retval.Format("%d %c", (int)val, unitmod);

    return retval;
}

void AddCommas(CString &str)
{
    const char *pp, *end;               // Ptrs into orig. string
    int ndigits = 0;                    // Number of digits in string
    char sep_char = ',';                // Used to separate groups of digits
    int group = 3;                          // How many is in a group
    struct lconv *plconv = localeconv(); // Pointer to locale info

    if (strlen(plconv->thousands_sep) == 1) // Small concession to users who
    {
        sep_char = *plconv->thousands_sep; // Use native number representation
        group = *plconv->grouping;
    }

    // Allocate enough space (allowing for space padding)
    char *out = new char[(str.GetLength()*(group+1))/group + 2]; // Result created here
    char *dd = out;                     // Ptr to current output char

    // Skip leading whitespace
    pp = str.GetBuffer(0);
    while (isspace(*pp))
        pp++;

    // Keep initial sign if any
    if (*pp == '+' || *pp == '-')
        *dd++ = *pp++;

    // Find end of number and work out how many digits it has
    end = pp + strspn(pp, ", \t0123456789");
    for (const char *qq = pp; qq < end; ++qq)
        if (isdigit(*qq))
            ++ndigits;

    for ( ; ndigits > 0; ++pp)
        if (isdigit(*pp))
        {
            *dd++ = *pp;
            if (--ndigits > 0 && ndigits%group == 0)
                *dd++ = sep_char;
        }
    ASSERT(pp <= end);
    strcpy(dd, pp);

    str = out;
    delete[] out;
}

void AddSpaces(CString &str)
{
    const char *pp, *end;               // Ptrs into orig. string
    int ndigits = 0;                    // Number of digits in string
    const char sep_char = ' ';  // Used to separate groups of digits
    const int group = 4;                // How many is in a group

    // Allocate enough space (allowing for space padding)
    char *out = new char[(str.GetLength()*(group+1))/group + 2]; // Result created here
    char *dd = out;                     // Ptr to current output char

    // Skip leading whitespace
    pp = str.GetBuffer(0);
    while (isspace(*pp))
        pp++;

    // Find end of number and work out how many digits it has
    end = pp + strspn(pp, " \t0123456789ABCDEFabcdef");
    for (const char *qq = pp; qq < end; ++qq)
        if (isxdigit(*qq))
            ++ndigits;

    for ( ; ndigits > 0; ++pp)
        if (isxdigit(*pp))
        {
            *dd++ = *pp;
            if (--ndigits > 0 && ndigits%group == 0)
                *dd++ = sep_char;
        }
    ASSERT(pp <= end);
    strcpy(dd, pp);

    str = out;
    delete[] out;
}

void BrowseWeb(UINT id)
{
	CString str;
	VERIFY(str.LoadString(id));

    ::ShellExecute(AfxGetMainWnd()->m_hWnd, _T("open"), str, NULL, NULL, SW_SHOWNORMAL);
}

BOOL SendEmail(int def_type /*=0*/, const char *def_text /*=NULL*/, const char *def_name)
{
    BOOL retval = FALSE;
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    if (::GetProfileInt("MAIL", "MAPI", 0) != 1)
    {
        ::HMessageBox("MAPI not supported on this machine");
        return FALSE;                                 // MAPI mail not supported
    }

    HINSTANCE hmapi = ::LoadLibrary("MAPI32.DLL");
    if (hmapi == (HINSTANCE)0)
    {
        ::HMessageBox("MAPI32.DLL not found");
        return FALSE;
    }

    LPMAPISENDMAIL pf = (LPMAPISENDMAIL)GetProcAddress(hmapi, "MAPISendMail");
    if (pf == (LPMAPISENDMAIL)0)
    {
        ::HMessageBox("MAPI32.DLL failure");
        ::FreeLibrary(hmapi);
        return FALSE;
    }

    CEmailDlg dlg;
    dlg.type_ = def_type;
    dlg.text_ = def_text;
    dlg.name_ = def_name;
    if (def_type == 2)
    {
        dlg.to_ = "sales@ExpertComSoft.com";
        if (dlg.name_.IsEmpty())
            dlg.subject_ = "Single-machine licence";
        else
            dlg.subject_ = "Single-user licence";
    }

    while (dlg.DoModal() == IDOK)
    {
        // Send (or try to send) the mail
        CString subject, text;          // Where subject & body of mail is built

        // Set up the sender and receiver of the email
        MapiRecipDesc from, to;
        memset(&from, '\0', sizeof(from));
        from.ulRecipClass = MAPI_ORIG;
        from.lpszName = getenv("USERNAME");

        CString address = "SMTP:" + dlg.to_;
        memset(&to, '\0', sizeof(to));
        to.ulRecipClass = MAPI_TO;
        to.lpszName = "andrew";
        to.lpszAddress = address.GetBuffer(0);

        // Build the subject and message text from the info the user entered
        switch (dlg.type_)
        {
        case 0:
            subject = "HEXEDIT BUG: ";
            text = "TYPE: BUG REPORT\n";
            break;
        case 1:
            subject = "HEXEDIT REQ: ";
            text = "TYPE: ENHANCEMENT REQUEST\n";
            break;
        default:
            subject = "HEXEDIT OTH: ";
            text = "TYPE: OTHER\n";
        }
        subject += dlg.subject_;

        // Add the system information to the email
        text += "SYSTEM: ";
        if (dlg.systype_.CompareNoCase("This system") == 0)
        {
            OSVERSIONINFO osvi;
            osvi.dwOSVersionInfoSize = sizeof(osvi);
            GetVersionEx(&osvi);

            if (osvi.dwPlatformId == VER_PLATFORM_WIN32_NT &&
                osvi.dwMajorVersion == 3 && osvi.dwMinorVersion >= 50)
                text += "NT 3.5X";
            else if (osvi.dwPlatformId == VER_PLATFORM_WIN32_NT && osvi.dwMajorVersion == 4)
                text += "NT 4.0";
            else if (osvi.dwPlatformId == VER_PLATFORM_WIN32_NT && osvi.dwMajorVersion == 5)
                text += "NT 5.0";
            else if (osvi.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS &&
                osvi.dwMajorVersion == 4 && osvi.dwMinorVersion == 0)
                text += "Windows 95";
            else if (osvi.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS &&
                osvi.dwMajorVersion == 4 && osvi.dwMinorVersion == 10)
                text += "Windows 98";
            CString version_info;
            version_info.Format("  %ld.%ld (build %ld)", (long)osvi.dwMajorVersion,
                                (long)osvi.dwMinorVersion, (long)osvi.dwBuildNumber);
            text += version_info;
        }
        else
            text += dlg.systype_;

        // Add info about the version of HexEdit and user details
        text += "\nVERSION: ";
        text += dlg.version_;
        text += "\nNAME: ";
        text += dlg.name_;
        text += "\nEMAIL: ";
        text += dlg.address_;
        text += "\n";

        text += dlg.text_;

        // Set up the email header info
        MapiMessage mm;
        memset(&mm, '\0', sizeof(mm));
        mm.lpszSubject = subject.GetBuffer(0);
        mm.lpszNoteText = text.GetBuffer(0);
        mm.lpOriginator = &from;
        mm.nRecipCount = 1;
        mm.lpRecips = &to;

        if (pf(0, 0, &mm, MAPI_LOGON_UI, 0) == 0)
        {
            retval = TRUE;
            break;
        }
        
        if (::HMessageBox("Error encountered creating email session\r"
                          "Try again?", MB_YESNO) != IDYES)
        {
            break;
        }
    }

    ::FreeLibrary(hmapi);

    // BG search finished message may be lost if modeless dlg running
    aa->CheckBGSearchFinished();

    return retval;
}

// FileErrorMessage - generate a meaningful error string from a CFileException
// fe = pointer to error information (including file name)
// mode = this parameter is used to make the message more meaningful depending
//        on the type of operations being performed on the file.  If the only
//        file operations being performed are reading file(s) the use
//        CFile::modeRead.  If only creating/writing files use CFile::modeWrite.
//        Otherwise use the default (both flags combined) to get messages that
//        are less specific but will not mislead the user if the flag does not
//        match the operations.
CString FileErrorMessage(const CFileException *fe, UINT mode /*=CFile::modeRead|CFile::modeWrite*/)
{
        CString retval;                                         // Returned error message
        CFileStatus fs;                                         // Current file status (if access error)
        UINT drive_type;                                        // Type of drive where the file is
    char rootdir[4] = "?:\\";			// Use with GetDriveType

        if (mode == CFile::modeRead)
                retval.Format("An error occurred reading from the file \"%s\".\n", fe->m_strFileName);
        else if (mode = CFile::modeWrite)
                retval.Format("An error occurred writing to the file \"%s\".\n", fe->m_strFileName);
        else
                retval.Format("An error occurred using the file \"%s\".\n", fe->m_strFileName);

        switch (fe->m_cause)
        {
        case CFileException::none:
                ASSERT(0);                                                      // There should be an error for this function to be called
                retval += "Apparently there was no error!";
                break;
        case CFileException::generic:
                retval += "The error is not specified.";
                break;
        case CFileException::fileNotFound:
                ASSERT(mode != CFile::modeWrite);       // Should be reading from an existing file
                retval += "The file does not exist.";
                break;
        case CFileException::badPath:
                retval += "The file name contains invalid characters or the drive\n"
                              "or one or more component directories do not exist.";
                break;
        case CFileException::tooManyOpenFiles:
                retval += "There are too many open files in this system.\n"
                                  "Try closing some programs or rebooting the system.";
                break;
        case CFileException::accessDenied:
                // accessDenied (or errno == EACCES) is a general purpose error
                // value and can mean many things.  We try to find out more about
                // the file to work out what exactly went wrong.
                if (!CFile::GetStatus(fe->m_strFileName, fs))
                {
                        if (fe->m_strFileName.Compare("\\") == 0 ||
                            fe->m_strFileName.GetLength() == 3 && fe->m_strFileName[1] == ':' && fe->m_strFileName[2] == '\\')
                        {
                                retval += "This is the root directory.\n"
                                                  "You cannot use it as a file.";
                        }
                        else if (mode == CFile::modeWrite)
                        {
                                retval += "Check that you have permission to write\n"
                                                  "to the directory and that the disk is\n"
                                                  "not write-protected or read-only media.";
                        }
                        else
                                retval += "Access denied. Check that you have permission\n"
                                                  "to read the directory and use the file.";
                }
                else if (fs.m_attribute & CFile::directory)
                        retval += "This is a directory - you cannot use it as a file.";
                else if (fs.m_attribute & (CFile::volume|CFile::hidden|CFile::system))
                        retval += "The file is a special system file.";
                else if ((fs.m_attribute & CFile::readOnly) && mode != CFile::modeRead)
                        retval += "You cannot write to a read-only file.";
                else
                        retval += "You cannot access this file.";
                break;
        case CFileException::invalidFile:
                ASSERT(0);                                                              // Uninitialised or corrupt file handle
                retval += "A software error occurred: invalid file handle.\n"
                                  "Please report this defect to the software vendor.";
                break;
        case CFileException::removeCurrentDir:
                retval += "An attempt was made to remove the current directory.";
                break;
        case CFileException::directoryFull:
                ASSERT(mode != CFile::modeRead);        // Must be creating a file
                retval += "The file could not be created because the directory\n"
                                  "for the file is full.  Delete some files from the\n"
                                  "root directory or use a sub-directory.";
                break;
        case CFileException::badSeek:
                ASSERT(0);                                                      // Normally caused by a bug
                retval += "A software error occurred: seek to bad file position.";
                break;
        case CFileException::hardIO:
                if (fe->m_strFileName.GetLength() > 2 && fe->m_strFileName[1] == ':')
                        rootdir[0] = fe->m_strFileName[0];
                else
                        rootdir[0] = _getdrive() + 'A' - 1;

                drive_type = GetDriveType(rootdir);
                switch (drive_type)
                {
                case DRIVE_REMOVABLE:
                case DRIVE_CDROM:
                        retval += "There was a problem accessing the drive.\n"
                                          "Please ensure that the medium is present.";
                        break;
                case DRIVE_REMOTE:
                        retval += "There was a problem accessing the file.\n"
                                          "There may be a problem with your network.";
                        break;
                default:
                        retval += "There was a hardware error.  There may be\n"
                                          "a problem with your computer or disk drive.";
                        break;
                }
                break;
        case CFileException::sharingViolation:
                retval += "SHARE.EXE is not loaded or the file is in use.\n"
                                  "Check that SHARE.EXE is installed, then try\n"
                                  "closing other programs or rebooting the system";
                break;
        case CFileException::lockViolation:
                retval += "The file (or part thereof) is in use.\n"
                                  "Try closing other programs or rebooting the system";
                break;
        case CFileException::diskFull:
                ASSERT(mode != CFile::modeRead);        // Must be writing to a file
                retval += "The file could not be written as the disk is full.\n"
                                  "Please delete some files to make room on the disk.";
                break;
        case CFileException::endOfFile:
                ASSERT(mode != CFile::modeWrite);       // Should be reading from a file
                retval += "An attempt was made to access an area past the end of the file.";
                break;
        default:
                ASSERT(0);
                retval += "An undocumented file error occurred.";
                break;
        }
        return retval;
}

