// This makes your compiler to catch more errors in the program
#define STRICT

// This makes compilation faster
#define WIN32_LEAN_AND_MEAN

#include <stdexcept>

#include <windows.h>
#include <objbase.h>
#include <objidl.h>
#include <shlobj.h>

#include "fldtree.h"

#define ID_UPDATE 102

HINSTANCE hInstance;

// The accelerator table used by the program. Consist only from the F5 key
// which synchronizes the tree with the filesystem.
HACCEL hAccel;

HWND main_wnd;
HWND tree_ctrl;

int busy_cursor = 0;

// Class representing a Win32(R) error code
class windows_exception: public std::exception
{
    DWORD code;
public:
    windows_exception(DWORD _code=GetLastError()) : code(_code) { }
    DWORD get_code() const { return code; }
};

// Class representing a COM error code
class com_exception: public std::exception
{
    HRESULT code;
public:
    com_exception(HRESULT _code) : code(_code) { }
    HRESULT get_code() const { return code; }
};

// The following two functions display error-message boxes

void WindowsErrorDialog(DWORD code=GetLastError())
{
    TCHAR *buf;
    if( FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER|FORMAT_MESSAGE_FROM_SYSTEM,
        NULL, code, 0, (TCHAR*)&buf, 0, NULL) )
    {
        MessageBox(main_wnd, buf, TEXT("FoldersTree: Error"), MB_OK|MB_ICONERROR);
        LocalFree(buf);
    }
}

void COMErrorDialog(HRESULT code)
{
    if(code==NOERROR) return;
    if(code==E_OUTOFMEMORY)
    	WindowsErrorDialog(ERROR_OUTOFMEMORY);
    else if(HRESULT_FACILITY(code)==FACILITY_WIN32)
    	WindowsErrorDialog(HRESULT_CODE(code));
    else
    	MessageBox(main_wnd, TEXT("Unknown error"), TEXT("FoldersTree: Error"), MB_OK|MB_ICONERROR);
}

// This functions throws an exception if its argument indicates a COM error
void CheckHRESULT(HRESULT result)
{
    if(result) throw com_exception(result);
}

LRESULT CALLBACK WindowProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch(msg) {
        case WM_DESTROY:
            PostQuitMessage(0);
            return 0;
        case WM_SIZE: {
            RECT client_rect;
            if( !GetClientRect(hwnd, &client_rect) )
                throw windows_exception();
            if( !MoveWindow(tree_ctrl, 0,0, client_rect.right, client_rect.bottom, TRUE) )
                throw windows_exception();
            return 0;
        }
        case WM_SETFOCUS:
            (void)SetFocus(tree_ctrl);
            return 0;
        case WM_COMMAND:
            switch( LOWORD(wParam) ) {
            	// Reload the tree from the filesystem
                case ID_UPDATE: {
                    HCURSOR hBusyCursor = LoadCursor(NULL,IDC_WAIT);
                    if(!hBusyCursor) throw windows_exception();
                    HCURSOR hCursor = SetCursor(hBusyCursor);
                    if(!hCursor) throw windows_exception();
                    ++busy_cursor;
                    // If it is the limited (non-commercial) version of fldtree.dll,
                    // FoldersTree_Update() doesn't work but shows the error dialog.
                    // We don't terminate the program in this case.
                    HRESULT hres = FoldersTree_Update(tree_ctrl,TVI_ROOT);
                    if(hres!=E_NOTIMPL) CheckHRESULT(hres);
                    --busy_cursor;
                    (void)SetCursor(hCursor);
                    break;
                }
            }
            return 0;
        // By definition (see the help), for every FoldersTree control WM_NOTIFY
        // must be processed by calling FoldersTree_ProcessNotificationMessage().
        // NOTE: It is processed in the window procedure of the PARENT control!
        case WM_NOTIFY:
            return FoldersTree_ProcessNotificationMessage((NMHDR*)lParam);
        default:
            return DefWindowProc(hwnd, msg, wParam, lParam);
    }
}

// Initialize the program before main window cycle
void init(int show_cmd)
{
    InitCommonControls();
    CheckHRESULT( CoInitialize(NULL) );

    // The class of the main window
    WNDCLASS wnd_class;
    wnd_class.style = 0;
    wnd_class.lpfnWndProc = WindowProc;
    wnd_class.cbClsExtra = 0;
    wnd_class.cbWndExtra = 0;
    wnd_class.hInstance = hInstance;
    wnd_class.hIcon = NULL;
    wnd_class.hCursor = NULL;
    wnd_class.hbrBackground = NULL;
    wnd_class.lpszMenuName = NULL;
    wnd_class.lpszClassName = TEXT("MainWndClass");
    if( !RegisterClass(&wnd_class) ) throw windows_exception();

    main_wnd = CreateWindowEx(WS_EX_APPWINDOW, TEXT("MainWndClass"), TEXT("Test of Extreme Code's FoldersTree"), WS_OVERLAPPEDWINDOW|WS_CLIPCHILDREN, 30,30, 300,300, NULL, NULL, hInstance, 0);
    if(!main_wnd) throw windows_exception();
    // FoldersTree control has WC_TREEVIEW window class (the one of common tree view control).
    // It differs only in processing windows messages. (See the window procedure.)
    tree_ctrl = CreateWindow(WC_TREEVIEW, TEXT(""), WS_CHILD|WS_VISIBLE|TVS_HASLINES|TVS_HASBUTTONS, 0,0, 150,150, main_wnd, NULL, hInstance, NULL);
    if(!tree_ctrl) throw windows_exception();

    CheckHRESULT( FoldersTree_InitClass() );
    CheckHRESULT( FoldersTree_Init(tree_ctrl) );
    // Uncomment the following to show all shell objects
    // (not only folders, but files also):
    //CheckHRESULT( FoldersTree_SetFilterFlags(tree_ctrl,SHCONTF_FOLDERS|SHCONTF_NONFOLDERS) );

    // We must allocate item ID lists in shell memory
    // (See the documentation written by Microsoft(R).)
    IMalloc *shell_malloc;
    if( SHGetMalloc(&shell_malloc) != NOERROR ) throw com_exception(E_UNEXPECTED);

    ITEMIDLIST *desktop_pidl = (ITEMIDLIST*)shell_malloc->Alloc(2);
    if(!desktop_pidl) throw windows_exception(ERROR_OUTOFMEMORY);
    // Two byte zero is the item ID list of the desktop folder
    // (the top of folders' hierarchy), we assign it to *desktop_pidl.
    *(WORD*)desktop_pidl = 0;

    // Inserts desktop folder as the root of the tree
    HTREEITEM hRootItem;
    // Note that this program fragment frees memory even in the case of an error
    HRESULT hres = FoldersTree_InsertFileSimple(tree_ctrl, TVI_ROOT, TVI_FIRST, desktop_pidl, &hRootItem);
    shell_malloc->Free(desktop_pidl);
    CheckHRESULT(hres);

    TreeView_Expand(tree_ctrl,hRootItem,TVE_EXPAND);

    ACCEL accel[] = {
            { FVIRTKEY, VK_F5, ID_UPDATE }
        };
    hAccel = CreateAcceleratorTable(accel,1);
    if(!hAccel) throw windows_exception();

    (void)ShowWindow(main_wnd,show_cmd);
    if( !UpdateWindow(main_wnd) )
// In Windows(R) 95, 98 and ME we cannot know the exact error
#ifdef _WIN32_WINNT
        throw windows_exception();
#else
        throw com_exception(E_UNEXPECTED);
#endif
}

// main window cycle
WPARAM run()
{
    MSG msg;
    for(;;) {
        BOOL ret = GetMessage(&msg, (HWND) NULL, 0, 0);
        if(!ret) break;
        if(ret==-1) throw windows_exception(); // something wrong
        if (!TranslateAccelerator(main_wnd,hAccel,&msg)) {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
    }

    return msg.wParam;
}

int WINAPI WinMain(HINSTANCE hInst, HINSTANCE /*hPrevInst*/, LPSTR /*cmd_line*/, int show_cmd)
{
    hInstance = hInst;
    // ry to run the program
    try {
    	init(show_cmd);
    	return run();
    }
    // In case of an error show the error dialog
    catch(const windows_exception &e) {
    	WindowsErrorDialog(e.get_code());
    }
    catch(const com_exception &e) {
    	COMErrorDialog(e.get_code());
    }

    // I deliberately make a small error here to simplify the program:
    // In the case an exception was thrown in the init() function
    // (such as in the case of out-of-memory)
    // the following tries to destroy not constructed objects.
    // This may cause hanging.

    if( !DestroyAcceleratorTable(hAccel) ) WindowsErrorDialog();

    COMErrorDialog( FoldersTree_Destroy(tree_ctrl) );
    COMErrorDialog( FoldersTree_DestroyClass() );

    CoUninitialize();
}
