/* ------------------------------------------------------------------------ */
/* System       : Library Routines                                          */
/* Program      : chkbmp.c                                                  */
/* Description  : .BMP file header validation routine                       */
/*                (C) CopyLeft Bill Buckels 1997.                           */
/*                All Rights Reversed.                                      */
/*                                                                          */
/* You have a royalty-free right to use, modify, reproduce and              */
/* distribute this source code in any way you find useful,                  */
/* provided that you agree that Bill Buckels has no warranty obligations    */
/* or liability resulting from said distribution in any way whatsoever.     */
/* ------------------------------------------------------------------------ */
#define _CHKBMP_C_

#include "stdkeys.h"
#include "pixhdr.h"

/* -------------------------------------------------------------------- */
/* function checkbmpheader                                              */
/*                                                                      */
/* checks a bmp file header for a specified bmp type                    */
/*   supported header types are                                         */
/*     2   color - MONO_VGA_VIDEO                                       */
/*     16  color - VGA_VIDEO                                            */
/*     256 color - MCGA_VIDEO                                           */
/*     16M color - TRUECOLOR_VIDEO                                      */
/*   non-compressed only - BI_RGB (this is the usual type anyway)       */
/*   scan line length (in pixels) must not exceed maximum specified     */
/*                                                                      */
/* args                                                                 */
/*   bmpptr - BITMAPINFO structure pointer to info read from file       */
/*   bmfptr - BITMAPFILEINFO pointer...                                 */
/*   iMax - maximum requested width in pixels                           */
/*   vidmode - this is really the bmp type specification (see above)    */
/*                                                                      */
/* returns                                                              */
/*   SUCCESS if success                                                 */
/*   INVALID_HEADER if this is not a bmp file                           */
/*   TOO_BIG if width in pixels is too wide (avoid buffer oveflow)      */
/*   INVALID_TYPE if image does not match type specified                */
/* -------------------------------------------------------------------- */
int checkbmpheader(BITMAPINFO *bmpptr, BITMAPFILEHEADER *bmfptr,
                   unsigned iMax, unsigned char vidmode)
{
   long lMax;
   unsigned uiPlanes , uiBitCount;
   if (bmfptr->bfType[0] != 'B' ||  /* test the signature...    */
       bmfptr->bfType[1] != 'M')    /* first two bytes are "BM" */
     return (INVALID_HEADER);

   lMax = 1L * iMax;                          /* test the image width */
   if(bmpptr->bmiHeader.biWidth > lMax)
     return (TOO_BIG);

   if (bmpptr->bmiHeader.biCompression != BI_RGB) /* non-compressed only */
     return INVALID_TYPE;
    
   switch (vidmode) {                         /* test valid image type */
     case TRUECOLOR_VIDEO:
       uiPlanes   = 1;
       uiBitCount = 24;
     case MONO_VGA_VIDEO:
       uiPlanes   = 1;
       uiBitCount = 1;
       break;
     case VGA_VIDEO:
       uiPlanes   = 1;
       uiBitCount = 4;
       break;
     case MCGA_VIDEO:
       uiPlanes   = 1;
       uiBitCount = 8;
       break;
     default:
       return INVALID_TYPE;
   }

   if (bmpptr->bmiHeader.biPlanes != uiPlanes)
     return (INVALID_TYPE);
   if (bmpptr->bmiHeader.biBitCount != uiBitCount)
     return (INVALID_TYPE);
   return (SUCCESS);
}
