/*******************  start of original comments  ********************/
/*
 * Written by Douglas Thomson (1989/1990)
 *
 * This source code is released into the public domain.
 */

/*
 * Name:    dte - Doug's Text Editor program - block commands module
 * Purpose: This file contains all the commands than manipulate blocks.
 * File:    block.c
 * Author:  Douglas Thomson
 * System:  this file is intended to be system-independent
 * Date:    October 1, 1989
 */
/*********************  end of original comments   ********************/

/*
 * In the DTE editor, Doug only supported functions for STREAM blocks.
 *
 * The block routines have been EXTENSIVELY rewritten.  This editor uses LINE,
 * STREAM, and BOX blocks.  That is, one may mark entire lines, streams of
 * characters, or column blocks.  Block operations are done in place.  There
 * are no paste and cut buffers.  In limited memory situations, larger block
 * operations can be carried out.  Block operations can be done within or
 * across files.
 *
 * In TDE, version 1.1, I separated the BOX and LINE actions.
 *
 * In TDE, version 1.3, I put STREAM blocks back in.  Added block upper case,
 *  block lower case, and block strip high bit.
 *
 * In TDE, version 1.4, I added a block number function.  Here at our lab,
 *  I often need to number samples, lines, etc..., comes in fairly useful.
 *
 * In TDE, version 2.0, I added a box block sort function.
 *
 * In TDE, version 2.0e, I added BlockRot13, BlockFixUUE, and BlockEmailReply.
 *
 * In TDE, version 2.2, the big text buffer was changed to a double linked list.
 *  all characters in the line of each node must be accurately counted.
 *
 * New editor name:  TDE, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991, version 1.0
 * Date:             July 29, 1991, version 1.1
 * Date:             October 5, 1991, version 1.2
 * Date:             January 20, 1992, version 1.3
 * Date:             February 17, 1992, version 1.4
 * Date:             April 1, 1992, version 1.5
 * Date:             June 5, 1992, version 2.0
 * Date:             October 31, 1992, version 2.1
 * Date:             April 1, 1993, version 2.2
 * Date:             June 5, 1993, version 3.0
 * Date:             August 29, 1993, version 3.1
 * Date:             November 13, 1993, version 3.2
 * Date:             June 5, 1994, version 4.0
 * Date:             December 5, 1998, version 5.0 (jmh)
 *
 * This modification of Douglas Thomson's code is released into the
 *  public domain, Frank Davis.  You may distribute it freely.
 */

#include "tdestr.h"
#include "common.h"
#include "tdefunc.h"
#include "define.h"


/*
 * Name:     mark_block
 * Class:    primary editor function
 * Purpose:  To record the position of the start of the block in the file.
 * Date:     June 5, 1991
 * Modified: August 9, 1997, Jason Hood - removed redundant logic, added markers
 * Passed:   window:  pointer to current window
 * Notes:    Assume the user will mark begin and end of a block in either
 *            line, stream, or box mode.  If the user mixes types, then block
 *            type defaults to current block type.
 *
 * jmh 980728: Allow explicit setting of the block beginning/end with the
 *              MarkBegin and MarkEnd functions. If no block is set, default
 *              to box.
 *             Removed the markers - now inherent in goto_marker.
 */
int  mark_block( TDE_WIN *window )
{
int type;
int num;
long lnum;
register file_infos *file;      /* temporary file variable */
register TDE_WIN *win;          /* put window pointer in a register */
int rc;

   win  = window;
   file = win->file_info;
   if (win->rline > file->length || win->ll->len == EOF)
      return( ERROR );

   switch (g_status.command) {
      case MarkBox:    type = BOX;    break;
      case MarkLine:   type = LINE;   break;
      case MarkStream: type = STREAM; break;
      case MarkBegin:
      case MarkEnd:    type = (g_status.marked) ? file->block_type : BOX;
                       break;
      default:
         return( ERROR );
   }

   if (g_status.marked == FALSE) {
      g_status.marked = TRUE;
      g_status.marked_file = file;
   }

   rc = OK;
   /*
    * define blocks for only one file.  it is ok to modify blocks in any window
    * pointing to original marked file.
    */
   if (file == g_status.marked_file) {

      /*
       * mark beginning and ending column regardless of block mode.
       */
      if (file->block_type <= NOTMARKED) {
         file->block_bc = file->block_ec = win->rcol;
         file->block_br = file->block_er = win->rline;

      } else if (g_status.command == MarkBegin) {
         file->block_bc = win->rcol;
         file->block_br = win->rline;

      } else if (g_status.command == MarkEnd || file->block_br <= win->rline) {
         file->block_ec = win->rcol;
         file->block_er = win->rline;

      } else {
         file->block_br = win->rline;
         if (file->block_bc < win->rcol && type != STREAM)
            file->block_ec = win->rcol;
         else
            file->block_bc = win->rcol;
      }
      /*
       * if user marks ending line less than beginning line then switch
       */
      if (file->block_er < file->block_br) {
         lnum = file->block_er;
         file->block_er = file->block_br;
         file->block_br = lnum;
      }
      /*
       * if user marks ending column less than beginning column then switch
       */
      if ((file->block_ec < file->block_bc) &&
          (type != STREAM || file->block_br == file->block_er)) {
         num = file->block_ec;
         file->block_ec = file->block_bc;
         file->block_bc = num;
      }
      file->block_type = type;
      file->dirty = GLOBAL;
   } else {
      /*
       * block already defined
       */
      error( WARNING, win->bottom_line, block1 );
      rc = ERROR;
   }
   return( rc );
}


/*
 * Name:     unmark_block
 * Class:    primary editor function
 * Purpose:  To set all block information to NULL or 0
 * Date:     June 5, 1991
 * Modified: August 9, 1997, Jason Hood - remove block beginning/end markers
 * Passed:   window:  pointer to current window
 * Notes:    Reset all block variables if marked, otherwise return.
 *            If a marked block is unmarked then redraw the screen(s).
 * jmh:      October 2, 1997 - I'm sometimes too hasty in unmarking blocks, so
 *            I'd like to be able to re-mark it. Don't reset the markers, and
 *            remember the globals (in static variables, 'cos I don't want to
 *            add even more global status variables). Don't re-mark in a macro
 *            though, because it might require a block to be set (this is how
 *            I did DeleteWordBack before I wrote the function).
 *
 * jmh:      November 26 - made old_file global so it won't re-mark a closed
 *            file. (I'll add it to g_status when I feel like recompiling
 *            everything.) (January 24, 1998 - done.)
 *
 * jmh 980728: re-mark the block in the current file, using a negative value
 *              when unmarking to remember the type. This removes the need
 *              for old_file and allows "multiple" blocks.
 *             Remove markers.
 */
int  unmark_block( TDE_WIN *window )
{
register file_infos *marked_file = NULL;

   if (g_status.marked == TRUE) {
      marked_file          = g_status.marked_file;
      g_status.marked_file = NULL;

   } else if (!mode.record && !g_status.macro_executing) {
      marked_file          = window->file_info;
      if (marked_file->block_type < NOTMARKED)
         g_status.marked_file = marked_file;
      else
         marked_file = NULL;
   }
   if (marked_file != NULL) {
      g_status.marked         = !g_status.marked;
      marked_file->dirty      = GLOBAL;
      marked_file->block_type = -marked_file->block_type;
   }
   return( OK );
}


/*
 * Name:    restore_marked_block
 * Class:   helper function
 * Purpose: To restore block beginning and ending row after an editing function
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 *          net_change: number of bytes added or subtracted
 * Notes:   If a change has been made before the marked block then the
 *           beginning and ending row need to be adjusted by the number of
 *           lines added or subtracted from file.
 * jmh:     November 26, 1997 - adjust the beginning/end markers (how did I
 *           miss this the first time?).
 * jmh 980728: Remove markers.
 * jmh 980821: update unmarked ("hidden") blocks, too.
 */
void restore_marked_block( TDE_WIN *window, int net_change )
{
long length;
register file_infos *marked_file;
int  update;

   if (net_change != 0) {
      marked_file = window->file_info;
      if (marked_file->block_type != NOTMARKED) {
         update = (g_status.marked && g_status.marked_file == marked_file);

         /*
          * if cursor is before marked block then adjust block by net change.
          */
         if (marked_file->block_br > window->rline) {
            marked_file->block_br += net_change;
            marked_file->block_er += net_change;
            if (update)
               marked_file->dirty = GLOBAL;
         /*
          * if cursor is somewhere in marked block don't restore, do redisplay
          */
         } else if (marked_file->block_er >= window->rline && update)
            marked_file->dirty = GLOBAL;

         /*
          * check for lines of marked block beyond end of file
          */
         length = marked_file->length;
         if (marked_file->block_br >= length) {
            if (update)
               unmark_block( window );
         } else if (marked_file->block_er >= length) {
            marked_file->block_er = length-1;
            if (update)
               marked_file->dirty = GLOBAL;
         }
      }
   }
}


/*
 * Name:    prepare_block
 * Class:   helper function
 * Purpose: To prepare a window/file for a block read, move or copy.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 *          file: pointer to file information.
 *          text_line: pointer to line in file to prepare.
 *          lend: line length.
 *          bc: beginning column of BOX.
 * Notes:   The main complication is that the cursor may be beyond the end
 *           of the current line, in which case extra padding spaces have
 *           to be added before the block operation can take place.
 *           this only occurs in BOX and STREAM operations.
 *          since we are padding a line, do not trim trailing space.
 */
int  prepare_block( TDE_WIN *window, line_list_ptr ll, int bc )
{
register int pad = 0;   /* amount of padding to be added */
register int len;

   assert( bc >= 0 );
   assert( bc < MAX_LINE_LENGTH );
   assert( ll->len != EOF );
   assert( g_status.copied == FALSE );

   copy_line( ll );
   detab_linebuff( );

   len = g_status.line_buff_len;
   pad = bc - len;
   if (pad > 0) {
      /*
       * insert the padding spaces
       */

      assert( pad >= 0 );
      assert( pad < MAX_LINE_LENGTH );

      memset( g_status.line_buff+len, ' ', pad );
      g_status.line_buff_len += pad;
   }
   /*
    * if mode.inflate_tabs, let's don't entab the line until we get
    *   thru processing this line, e.g. copying, numbering....
    */
   return( un_copy_line( ll, window, FALSE ) );
}


/*
 * Name:    pad_dest_line
 * Class:   helper function
 * Purpose: To prepare a window/file for a block move or copy.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 *          dest_file: pointer to file information.
 *          dest_line: pointer to line in file to prepare.
 *          ll:        pointer to linked list node to insert new node
 * Notes:   We are doing a BOX action (except DELETE).   We have come
 *          to the end of the file and have no more lines.  All this
 *          routine does is add a blank line to file.
 */
int  pad_dest_line( TDE_WIN *window, file_infos *dest_file, line_list_ptr ll )
{
int rc;
text_ptr l;
line_list_ptr new_node;

   rc = OK;
   l = NULL;
   new_node = (line_list_ptr)my_malloc( sizeof(line_list_struc), &rc );
   if (rc == OK) {
      new_node->len  = 0;
      new_node->type = 0;
      new_node->line = l;
      insert_node( window->file_info, ll, new_node );
      ++dest_file->length;
   } else {
      /*
       * file too big
       */
      error( WARNING, window->bottom_line, block4 );
      if (new_node != NULL)
         my_free( new_node );
      rc = ERROR;
   }
   return( rc );
}


/*
 * Name:    block_operation
 * Class:   primary editor function
 * Purpose: Master BOX, STREAM, or LINE routine.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Operations on BOXs, STREAMs, or LINEs require several common
 *           operations.  All require finding the beginning and ending marks.
 *          This routine will handle block operations across files.  Since one
 *           must determine the relationship of source and destination blocks
 *           within a file, it is relatively easy to expand this relationship
 *           across files.
 *          This is probably the most complicated routine in the editor.  It
 *           is not easy to understand.
 *
 * jmh 980728: update syntax highlight flags.
 * jmh 980801: Add BORDER action.
 * jmh 980810: renamed from move_copy_delete_overlay_block;
 *             added left/right/center justify line and box blocks.
 */
int  block_operation( TDE_WIN *window )
{
int  action;
TDE_WIN *source_window;         /* source window for block moves */
line_list_ptr source;           /* source for block moves */
line_list_ptr dest;             /* destination for block moves */
int  lens;                      /* length of source line */
int  lend;                      /* length of destination line */
int  add;                       /* characters being added from another line */
int  block_len;                 /* length of the block */
line_list_ptr block_start;      /* start of block in file */
line_list_ptr block_end;  /* end of block in file - not same for LINE or BOX */
int  prompt_line;
int  same;                      /* are dest and source files the same file? */
int  source_only;               /* does operation affect only the block? */
int  source_changed;            /* does the operation affect only the dest? */
file_infos *source_file;
file_infos *dest_file;
int  rcol, bc, ec;              /* temporary column variables */
long rline;                     /* temporary real line variable */
long br, er;                    /* temporary line variables */
long block_num;                 /* starting number for block number */
long block_inc;                 /* increment to use for block number */
int  block_just;                /* left or right justify numbers? */
int  block_type;
int  fill_char;
char style[8];                  /* border characters */
int  rc;
LANGUAGE *syntax;
char *errmsg = NULL;
long break_point;
long number;

   /*
    * initialize block variables
    */
   entab_linebuff( );
   rc = un_copy_line( window->ll, window, TRUE );
   if (g_status.marked == FALSE || rc == ERROR)
      return( ERROR );

   source_only = TRUE;
   source_changed = TRUE;
   switch (g_status.command) {
      case MoveBlock :
         action = MOVE;
         source_only = FALSE;
         break;
      case DeleteBlock :
         action = DELETE;
         break;
      case CopyBlock :
         action = COPY;
         source_only = FALSE;
         source_changed = FALSE;
         break;
      case KopyBlock :
         action = KOPY;
         source_only = FALSE;
         source_changed = FALSE;
         break;
      case FillBlock :
         action = FILL;
         break;
      case OverlayBlock :
         action = OVERLAY;
         source_only = FALSE;
         source_changed = FALSE;
         break;
      case NumberBlock :
         action = NUMBER;
         break;
      case SwapBlock :
         action = SWAP;
         source_only = FALSE;
         break;
      case BorderBlock :
         action = BORDER;
         break;
      case BlockLeftJustify :
      case BlockRightJustify :
      case BlockCenterJustify :
         action = JUSTIFY;
         break;
      default :
         return( ERROR );
   }
   source_file = g_status.marked_file;
   source_window = g_status.window_list;
   for (; ptoul( source_window->file_info ) != ptoul( source_file );)
      source_window = source_window->next;
   prompt_line = window->bottom_line;
   dest_file = window->file_info;
   check_block( );
   if (g_status.marked == FALSE)
      return( ERROR );
   block_start = source_file->block_start;
   block_end = source_file->block_end;
   if (block_start == NULL  ||  block_end == NULL)
      return( ERROR );

   block_type = source_file->block_type;
   if (block_type != LINE  &&  block_type != STREAM  &&  block_type != BOX)
      return( ERROR );

   dest = window->ll;
   rline = window->rline;
   if (!source_only)
      if (dest->next == NULL || (dest->prev == NULL && block_type != LINE))
         return( ERROR );
   rc = OK;

   /*
    * jmh 980810: let's verify the block type at the beginning.
    */
   if (block_type != BOX) {
      switch (action) {
         case OVERLAY: errmsg = block5;  break;
         case NUMBER:  errmsg = block3a; break;
         case FILL:    errmsg = block2;  break;
         case BORDER:  errmsg = block26; break;
      }
   }
   if (block_type == STREAM) {
      switch (action) {
         case SWAP:    errmsg = block3b; break;
         case JUSTIFY: errmsg = block29; break;
      }
   }
   if (errmsg != NULL) {
      error( WARNING, prompt_line, errmsg );
      return( ERROR );
   }

   /*
    * set up Beginning Column, Ending Column, Beginning Row, Ending Row
    */
   bc = source_file->block_bc;
   ec = source_file->block_ec;
   br = source_file->block_br;
   er = source_file->block_er;

   /*
    * if we are BOX FILLing or BOX NUMBERing, beginning column is bc,
    *   not the column of cursor
    * jmh 980801: as with BORDERing.
    * jmh 980810: and JUSTIFYing.
    */
   rcol = (source_only) ? bc : window->rcol;

   /*
    * must find out if source and destination file are the same.
    */
   same = FALSE;
   if (source_only) {
      rc = (action == FILL)   ? get_block_fill_char( window, &fill_char ) :
           (action == NUMBER) ? get_block_numbers( window, &block_num,
                                                   &block_inc, &block_just ) :
           (action == BORDER) ? get_block_border_style( window, style ) :
           OK;
      if (rc == ERROR)
         return( ERROR );
      dest = block_start;
      same = TRUE;
   }

   if (source_file == dest_file && !source_only) {
      same = TRUE;
      if (block_type == BOX) {
         if (action == MOVE) {
            if (rline == br  &&  (rcol >= bc && rcol <= ec+1))
               return( ERROR );
         } else if (action == SWAP) {
            /*
             * jmh 980811: can't swap a block within itself
             */
            if (rline >= br - (er - br) && rline <= er &&
                rcol  >= bc - (ec - bc) && rcol  <= ec && rcol != bc)
               return( ERROR );
         }
      } else if (block_type == LINE) {
         if (rline >= br && rline <= er) {
             /*
              * if COPYing or KOPYing within the block itself, reposition the
              * destination to the next line after the block (if it exists)
              */
            if (action == COPY || action == KOPY)
               dest = block_end;
             /*
              * a block moved to within the block itself has no effect
              * jmh 980810: a block can't swap with itself, either
              */
            else if (action == MOVE || action == SWAP)
               return( ERROR );
         }
         /*
          * jmh 980817: test a few more overlapping SWAPs
          */
         break_point = dest_file->break_point;
         if (break_point) {
            if ((rline < br && break_point >= br) ||
                (rline > er && break_point <= er) ||
                (break_point >= br && break_point <= er))
               return( ERROR );
         }
      } else if (rline >= br && rline <= er) {

         /*
          * to find out if cursor is in a STREAM block we have to do
          * a few more tests.  if cursor is on the beginning row or
          * ending row, then check the beginning and ending column.
          */
         if ((rline > br && rline < er) ||
             (br == er && rcol >= bc && rcol <= ec) ||
             (br != er && ((rline == br && rcol >= bc) ||
                           (rline == er && rcol <= ec)))) {

            /*
             * if the cursor is in middle of STREAM, make destination
             * the last character following the STREAM block.
             */
            if (action == COPY || action == KOPY) {
               dest = block_end;
               rcol = ec + 1;
               rline = er;
            } else if (action == MOVE)
               return( ERROR );
         }
      }
   }

   /*
    * 1. can't create lines greater than MAX_LINE_LENGTH
    * 2. if we are FILLing a BOX - fill block buff once right here
    * 3. only allow overlaying BOXs
    */
   block_len = (ec+1) - bc;
   if (block_type == BOX) {
      if (!source_only) {
         if (rcol + block_len > MAX_LINE_LENGTH) {
            /*
             * Error: line too long
             */
            error( INFO, prompt_line, ltol );
            return( ERROR );
         }
      }
   } else if (block_type == LINE) {
      block_len = 0;
   } else if (block_type == STREAM) {
      lend = block_end->len;
      if (action == DELETE || action == MOVE) {

         /*
          * Is what's left on start of STREAM block line plus what's left at
          * end of STREAM block line too long?
          */
         if (lend > ec)
            lend -= ec;
         else
            lend = 0;
         if (bc + lend > MAX_LINE_LENGTH) {
            error( INFO, prompt_line, ltol );
            return( ERROR );
         }
      }

      if (action != DELETE) {

         /*
          * We are doing a MOVE, COPY, or KOPY.  Find out if what's on the
          * current line plus the start of the STREAM line are too long.
          * Then find out if end of the STREAM line plus what's left of
          * the current line are too long.
          */
         lens = block_start->len;

         /*
          * if we had to move the destination of the STREAM COPY or KOPY
          * to the end of the STREAM block, then dest and window->ll->line
          * will not be the same.  In this case, set length to length of
          * first line in STREAM block.  Then we can add the residue of
          * the first line in block plus residue of the last line of block.
          */
         if (dest->line == window->ll->line)
            add = dest->len;
         else
            add = lens;

         /*
          * Is current line plus start of STREAM block line too long?
          */
         if (lens > bc)
            lens -= bc;
         else
            lens = 0;
         if (rcol + lens > MAX_LINE_LENGTH) {
            error( INFO, prompt_line, ltol );
            return( ERROR );
         }

         /*
          * Is residue of current line plus residue of STREAM block line
          * too long?
          */
         if (add > bc)
            add -= bc;
         else
            add = 0;
         if (lend > ec)
            lend -= ec;
         else
            lend = 0;
         if (add + lend > MAX_LINE_LENGTH) {
            error( INFO, prompt_line, ltol );
            return( ERROR );
         }
      }
      if (ptoul( block_start ) == ptoul( block_end )) {
         block_type = BOX;
         block_len = (ec+1) - bc;
      }
   }

   if (mode.do_backups == TRUE) {
      if (!source_only) {
         window->file_info->modified = TRUE;
         rc = backup_file( window );
      }
      if (source_changed) {
         source_window->file_info->modified = TRUE;
         if (rc != ERROR)
            rc = backup_file( source_window );
      }
      if (rc == ERROR)
         return( ERROR );
   }
   source = block_start;

   assert( block_start != NULL );
   assert( block_start->len != EOF );
   assert( block_end != NULL );
   assert( block_end->len != EOF );

   if (block_type == BOX)
      do_box_block( window, source_window, action,
                    source_file, dest_file, source, dest, br, er,
                    block_inc, rline, block_num, block_just, fill_char, style,
                    same, block_len, bc, ec, rcol, &rc );

   else if (block_type == LINE) {
      if (action != JUSTIFY)
         do_line_block( window, source_window, action,
                        source_file, dest_file, same, block_start, block_end,
                        source, dest, br, er, rline, &rc );
      else
         justify_line_block( source_window, block_start, br, er, &rc );

   } else
      do_stream_block( window, source_window, action,
                       source_file, dest_file, block_start, block_end,
                       source, dest, rline, br, er, bc, ec, rcol, &rc );

   if (!source_only) {
      dest_file->modified = TRUE;
      dest_file->dirty = GLOBAL;
   }
   if (source_only || source_changed) {
      source_file->modified = TRUE;
      source_file->dirty = GLOBAL;
   }

   /*
    * Update the syntax highlight flags of affected lines.
    * Start with the source lines.
    */
   syntax = source_file->syntax;
   if (source_changed) {
      if (block_type == BOX)
         syntax_check_block( br, er, block_start, syntax );
      else if (block_type == STREAM)
         syntax_check_lines( block_start, syntax );
      /*
       * else block_type == LINE which is done in do_line_block
       */
   }

   /*
    * Now check the destination lines.
    */
   if (!source_only && block_type != LINE)
      syntax_check_block( br, er, dest, dest_file->syntax );

   /*
    * unless we are doing a KOPY, FILL, NUMBER, or OVERLAY we need to unmark
    * the block.  if we just did a KOPY, the beginning and ending may have
    * changed.  so, we must readjust beginning and ending rows.
    * jmh - don't unmark SWAP, as well.
    * jmh 980801: turned the unmark conditional around.
    * jmh 980814: move the mark along with the block.
    *             Go back to unmarking SWAP.
    */
   number = (er+1) - br;
   if ((action == KOPY || action == COPY) && block_type != BOX &&
       same && br >= rline  &&  rc != ERROR) {
      source_file->block_br += number - (block_type == STREAM);
      source_file->block_er += number - (block_type == STREAM);
   }
   if (action == COPY   || action == MOVE ||
       action == DELETE || action == SWAP) {
      unmark_block( window );
      if (action == SWAP && block_type == LINE)
         dest_file->break_point = 0;

      if (action == MOVE) {
         if (same && block_type == BOX && rline == br && rcol > bc) {
            dest_file->block_bc = rcol - block_len;
            dest_file->block_ec = rcol - 1;
         } else {
            dest_file->block_bc = rcol;
            if (block_type != STREAM)
               dest_file->block_ec = rcol + ec - bc;
         }
         --number;
         if (same && rline > br && block_type != BOX) {
            dest_file->block_br = rline - number;
            dest_file->block_er = rline;
         } else {
            dest_file->block_br = rline;
            dest_file->block_er = rline + number;
            if (block_type == LINE) {
               ++dest_file->block_br;
               ++dest_file->block_er;
            }
         }
      }
   }
   show_avail_mem( );
   g_status.copied = FALSE;
   return( rc );
}


/*
 * Name:    do_line_block
 * Purpose: delete, move, copy, or kopy a LINE block
 * Date:    April 1, 1993
 * Passed:  window:  pointer to current window
 * Passed:  window:         pointer to destination window (current window)
 *          source_window:  pointer to source window
 *          action:         block action  --  KOPY, MOVE, etc...
 *          source_file:    pointer to source file structure
 *          dest_file:      pointer to destination file
 *          same:           are source and destination files same? T or F
 *          block_start:    pointer to first node in block
 *          block_end:      pointer to last node in block
 *          source:         pointer to source node
 *          dest:           pointer to destination node
 *          br:             beginning line number in marked block
 *          er:             ending line number in marked block
 *          rline:          current line number in destination file
 *          rc:             return code
 *
 * jmh:     September 11, 1997 - added SWAP
 * jmh 980817: if a break point is defined, SWAP will swap the block with the
 *              block delineated by the cursor and break point. If there is no
 *              break point, it swaps the same number of lines, or till eof.
 * jmh 981125: corrected a MOVE syntax check (set dest_end).
 */
void do_line_block( TDE_WIN *window,  TDE_WIN *source_window,  int action,
                    file_infos *source_file,  file_infos *dest_file,  int same,
                    line_list_ptr block_start,  line_list_ptr block_end,
                    line_list_ptr source,  line_list_ptr dest,
                    long br,  long er,  long rline,  int *rc )
{
line_list_ptr temp_ll;          /* temporary list pointer */
line_list_ptr temp2;            /* temporary list pointer */
line_list_ptr dest_end;
text_ptr l;
int  lens;                      /* length of source line */
long li;                        /* temporary line variables */
long diff;
long diff_dest = 1;
long break_point;

   dest_end = dest;
   if (action == COPY || action == KOPY) {

      assert( br >= 1 );
      assert( br <= source_file->length );
      assert( er >= br );
      assert( er <= source_file->length );

      for (li=br; li <= er  &&  *rc == OK; li++) {
         lens = source->len;

         assert( lens * sizeof(char) < MAX_LINE_LENGTH );

         l = (text_ptr)my_malloc( lens * sizeof(char), rc );
         temp_ll = (line_list_ptr)my_malloc( sizeof(line_list_struc), rc );
         if (*rc == OK) {
            if (lens > 0)
               my_memcpy( l, source->line, lens );
            temp_ll->line = l;
            temp_ll->len  = lens;
            temp_ll->type = source->type | DIRTY;

            insert_node( dest_file, dest_end, temp_ll );

            dest_end = temp_ll;
            source = source->next;
         } else {
            /*
             * file too big
             */
            error( WARNING, window->bottom_line, main4 );
            if (l != NULL)
               my_free( l );
            if (temp_ll != NULL)
               my_free( temp_ll );
            *rc = ERROR;
            er = li - 1;
         }
      }
   } else if (action == MOVE) {
      temp_ll = block_start;
      for (li=br; li <= er; li++) {
         temp_ll->type |= DIRTY;
         temp_ll = temp_ll->next;
      }
      block_start->prev->next = block_end->next;
      block_end->next->prev   = block_start->prev;
      /*
       * jmh 980806: The block has effectively been deleted; syntax check
       *             the lines after it while the block line pointers
       *             are still valid.
       */
      syntax_check_lines( block_end->next, source_file->syntax );
      dest->next->prev  = block_end;
      block_start->prev = dest;
      block_end->next   = dest->next;
      dest->next        = block_start;
      dest_end          = block_end;

   } else if (action == DELETE) {
      block_end->next->prev   = block_start->prev;
      block_start->prev->next = block_end->next;
      /*
       * jmh 980806: as with MOVE
       */
      syntax_check_lines( block_end->next, source_file->syntax );
      block_end->next = NULL;
      while (block_start != NULL) {
         temp_ll = block_start;
         block_start = block_start->next;
         if (temp_ll->line != NULL)
            my_free( temp_ll->line );
         my_free( temp_ll );
      }
   } else if (action == SWAP) {
      dest->type |= DIRTY;
      break_point = dest_file->break_point;
      if (break_point) {
         if (rline > break_point) {
            for (; rline > break_point; --rline) {
               dest = dest->prev;
               dest->type |= DIRTY;
               ++diff_dest;
            }
         } else if (rline < break_point) {
            for (; rline < break_point; ++rline) {
               dest_end = dest_end->next;
               dest_end->type |= DIRTY;
               ++diff_dest;
            }
         }
      } else {
         for (li = br; li < er  &&  dest_end->next->next != NULL; li++) {
            if (same && ++rline == br)
               break;
            dest_end = dest_end->next;
            dest_end->type |= DIRTY;
            ++diff_dest;
         }
      }
      temp_ll = block_start;
      for (li = br; li <= er; ++li) {
         temp_ll->type |= DIRTY;
         temp_ll = temp_ll->next;
      }
      temp_ll           = block_start->prev;
      temp2             = dest->prev;
      temp_ll->next     = dest;
      temp2->next       = block_start;
      block_start->prev = temp2;
      dest->prev        = temp_ll;

      temp_ll           = block_end->next;
      temp2             = dest_end->next;
      temp_ll->prev     = dest_end;
      temp2->prev       = block_end;
      block_end->next   = temp2;
      dest_end->next    = temp_ll;
   }

   diff =  er + 1L - br;
   if (action == COPY || action == KOPY || action == MOVE) {
      dest_file->length += diff;
      adjust_windows_cursor( window, diff );
   }
   if (action == DELETE || action == MOVE) {
      source_file->length -= diff;
      adjust_windows_cursor( source_window, -diff );
   }
   if (action == SWAP) {
      source_file->length += diff_dest - diff;
      dest_file->length   += diff - diff_dest;
      adjust_windows_cursor( source_window, diff_dest - diff );
      adjust_windows_cursor( window,        diff - diff_dest );
   }
   /*
    * do the syntax highlighting check
    */
   if (action == DELETE && source_window->rline >= br) {
      source_window->rline -= diff;
      if (source_window->rline < br)
         source_window->rline = br;
   }
   if (action != DELETE) {
      syntax_check_lines( (action == SWAP) ? dest : dest->next,
                          dest_file->syntax );
      syntax_check_lines( dest_end->next, dest_file->syntax );
      if (action == SWAP) {
         syntax_check_lines( block_start, source_file->syntax );
         syntax_check_lines( block_end->next, source_file->syntax );
      }
   }
   /*
    * restore all cursors in all windows
    */
   restore_cursors( dest_file );
   if (dest_file != source_file)
      restore_cursors( source_file );
   show_avail_mem( );
}


/*
 * Name:    do_stream_block
 * Purpose: delete, move, copy, or kopy a STREAM block
 * Date:    June 5, 1991
 * Passed:  window:         pointer to destination window (current window)
 *          source_window:  pointer to source window
 *          action:         block action  --  KOPY, MOVE, etc...
 *          source_file:    pointer to source file structure
 *          dest_file:      pointer to destination file
 *          block_start:    pointer to first node in block
 *          block_end:      pointer to last node in block
 *          source:         pointer to source node
 *          dest:           pointer to destination node
 *          rline:          current line number in destination file
 *          br:             beginning line number in marked block
 *          er:             ending line number in marked block
 *          bc:             beginning column of block
 *          ec:             ending column of block
 *          rcol:           current column of cursor
 *          rc:             return code
 */
void do_stream_block( TDE_WIN *window,  TDE_WIN *source_window,  int action,
                      file_infos *source_file,  file_infos *dest_file,
                      line_list_ptr block_start,  line_list_ptr block_end,
                      line_list_ptr source,  line_list_ptr dest, long rline,
                      long br,  long er, int bc, int ec, int rcol, int *rc )
{
line_list_ptr temp_ll;          /* temporary list pointer */
text_ptr l;
int  lens;                      /* length of source line */
int  lend;                      /* length of destination line */
long li;                        /* temporary line variables */
long diff;
TDE_WIN s_w, d_w;               /* a couple of temporary TDE_WINs */

   dup_window_info( &s_w, source_window );
   dup_window_info( &d_w, window );
   s_w.rline   = br;
   s_w.ll      = block_start;
   s_w.visible = FALSE;
   d_w.rline   = rline;
   d_w.ll      = dest;
   d_w.visible = FALSE;

   /*
    * pad the start of the STREAM block if needed.
    */
   lens = block_start->len;
   detab_a_line( block_start->line, &lens );
   if (lens < bc || mode.inflate_tabs)
      *rc = prepare_block( &s_w, block_start, bc );

   /*
    * pad the end of the STREAM block if needed.
    */
   lens = block_end->len;  
   detab_a_line( block_end->line, &lens );
   if (*rc == OK  &&  (lens < ec+1  ||  mode.inflate_tabs))
      *rc = prepare_block( &s_w, block_end, ec+1 );

   /*
    * pad the destination line if necessary
    */
   copy_line( dest );
   detab_linebuff( );
   *rc = un_copy_line( dest, &d_w, FALSE );
   lend = dest->len;
   if (*rc == OK && (action==MOVE || action==COPY || action==KOPY)) {
      if (lend < rcol || mode.inflate_tabs)
         *rc = prepare_block( &d_w, dest, rcol );
   }

   if ((action == COPY || action == KOPY) && *rc == OK) {

      /*
       * concatenate the end of the STREAM block with the end of the
       *   destination line.
       */
      lens = dest->len - rcol;

      assert( lens >= 0 );
      assert( lens <= MAX_LINE_LENGTH );
      assert( ec + 1 >= 0 );
      assert( ec + 1 <= MAX_LINE_LENGTH );
      assert( rcol >= 0 );

      my_memcpy( g_status.line_buff, block_end->line, ec+1 );
      my_memcpy( g_status.line_buff+ec+1, dest->line+rcol, lens );
      lens += ec + 1;
      g_status.line_buff_len = lens;

      temp_ll = (line_list_ptr)my_malloc( sizeof(line_list_struc), rc );
      if (*rc == OK) {
         temp_ll->line = NULL;
         temp_ll->len  = 0;
         temp_ll->type = 0;
         g_status.copied = TRUE;
         *rc = un_copy_line( temp_ll, &d_w, TRUE );

         if (*rc == OK) {
            dest->next->prev = temp_ll;
            temp_ll->next = dest->next;
            dest->next = temp_ll;
            temp_ll->prev = dest;
         } else
            if (temp_ll != NULL)
               my_free( temp_ll );
      } else {
         if (temp_ll != NULL)
            my_free( temp_ll );
      }

      /*
       * file too big
       */
      if (*rc != OK)
         error( WARNING, window->bottom_line, main4 );

      if (*rc == OK) {
         g_status.copied = FALSE;
         copy_line( dest );
         lens = block_start->len - bc;

         assert( lens >= 0 );
         assert( lens <= MAX_LINE_LENGTH );
         assert( bc >= 0 );
         assert( bc <= MAX_LINE_LENGTH );
         assert( rcol >= 0 );

         my_memcpy( g_status.line_buff+rcol, block_start->line+bc, lens );
         lens = rcol + lens;
         g_status.line_buff_len = lens;
         *rc = un_copy_line( dest, &d_w, TRUE );
      }

      source = block_start->next;
      for (li=br+1; li < er  &&  *rc == OK; li++) {
         lens = source->len;
         temp_ll = (line_list_ptr)my_malloc( sizeof(line_list_struc), rc );

         assert( lens >= 0 );
         assert( lens <= MAX_LINE_LENGTH );

         l = (text_ptr)my_malloc( lens * sizeof(char), rc );
         if (*rc == OK) {
            if (lens > 0)
               my_memcpy( l, source->line, lens );
            temp_ll->line  = l;
            temp_ll->len   = lens;
            temp_ll->type |= DIRTY;
            insert_node( window->file_info, dest, temp_ll );
            dest   = temp_ll;
            source = source->next;
         } else {
            /*
             * file too big
             */
            error( WARNING, window->bottom_line, main4 );
            if (l != NULL)
               my_free( l );
            if (temp_ll != NULL)
               my_free( temp_ll );
            *rc = WARNING;
         }
      }
   } else if (action == MOVE) {

      /*
       * is the dest on the same line as the block_start?
       */
      if (ptoul( dest ) == ptoul( block_start )) {

         /*
          * move the text between rcol and bc in block_start->line
          *   to block_end->line + ec.
          */
         lens = bc - rcol;
         lend = block_end->len - (ec + 1);
         g_status.copied = FALSE;
         copy_line( block_end );


         assert( lens >= 0 );
         assert( lens <= MAX_LINE_LENGTH );
         assert( lend >= 0 );
         assert( lend <= MAX_LINE_LENGTH );
         assert( ec + lens + 1 <= MAX_LINE_LENGTH );
         assert( rcol >= 0 );


         my_memmove( g_status.line_buff + ec + lens + 1,
                     g_status.line_buff + ec + 1,  lend );
         my_memcpy( g_status.line_buff+ec+1, block_start->line+rcol, lens );
         g_status.line_buff_len = block_end->len + lens;
         *rc = un_copy_line( block_end, &d_w, TRUE );

         /*
          * now, remove the text between rcol and bc on block_start->line
          */
         if (*rc == OK) {
            lend = block_start->len - bc;
            copy_line( block_start );

            assert( lend >= 0 );
            assert( lend < MAX_LINE_LENGTH );

            my_memmove( g_status.line_buff + rcol,
                        g_status.line_buff + bc, lend );

            assert( block_start->len - (bc - rcol) >= 0 );
            assert( block_start->len - (bc - rcol) <= MAX_LINE_LENGTH );

            g_status.line_buff_len = block_start->len - (bc - rcol);
            *rc = un_copy_line( block_start, &d_w, TRUE );
         }

      /*
       * is the dest on the same line as the block_end?
       */
      } else if (ptoul( dest ) == ptoul( block_end )) {

         /*
          * move the text between rcol and ec on block_end->line to
          *   block_start->line + bc.
          */
         lens = rcol - ec;
         lend = block_start->len - bc;
         g_status.copied = FALSE;
         copy_line( block_start );

         assert( lens >= 0 );
         assert( lens <= MAX_LINE_LENGTH );
         assert( lend >= 0 );
         assert( lend <= MAX_LINE_LENGTH );
         assert( bc + lens <= MAX_LINE_LENGTH );
         assert( ec + 1 >= 0 );

         my_memmove( g_status.line_buff + bc + lens,
                     g_status.line_buff + bc,  lend );
         my_memcpy( g_status.line_buff+bc, block_end->line+ec+1, lens );

         assert( block_start->len + lens >= 0 );
         assert( block_start->len + lens <= MAX_LINE_LENGTH );

         g_status.line_buff_len = block_start->len + lens;
         *rc = un_copy_line( block_start, &d_w, TRUE );

         /*
          * now, remove the text on block_end->line between rcol and ec
          */
         if (*rc == OK) {
            lend = block_end->len - (rcol + 1);
            copy_line( block_end );

            assert( lend >= 0 );
            assert( lend <= MAX_LINE_LENGTH );
            assert( ec + 1 >= 0 );
            assert( rcol + 1 >= 0 );
            assert( ec + 1 <= MAX_LINE_LENGTH );
            assert( rcol + 1 <= MAX_LINE_LENGTH );
            assert( block_end->len - (rcol - ec) >= 0 );
            assert( block_end->len - (rcol - ec) <= MAX_LINE_LENGTH );


            my_memmove( g_status.line_buff + ec + 1,
                        g_status.line_buff + rcol + 1, lend );
            g_status.line_buff_len = block_end->len - (rcol - ec);
            *rc = un_copy_line( block_end, &d_w, TRUE );
         }
      } else {

         lens = dest->len - rcol;

         assert( ec + 1 >= 0 );
         assert( ec + 1 <= MAX_LINE_LENGTH );
         assert( lens >= 0 );
         assert( lens <= MAX_LINE_LENGTH );
         assert( rcol >= 0 );
         assert( rcol <= MAX_LINE_LENGTH );

         my_memcpy( g_status.line_buff, block_end->line, ec+1 );
         my_memcpy( g_status.line_buff+ec+1, dest->line+rcol, lens );
         lens += ec + 1;
         g_status.line_buff_len = lens;

         temp_ll = (line_list_ptr)my_malloc( sizeof(line_list_struc), rc );
         if (*rc == OK) {
            temp_ll->line = NULL;
            temp_ll->len  = 0;
            temp_ll->type = 0;
            g_status.copied = TRUE;
            *rc = un_copy_line( temp_ll, &d_w, TRUE );

            if (*rc != ERROR) {
               dest->next->prev = temp_ll;
               temp_ll->next = dest->next;
               dest->next = temp_ll;
               temp_ll->prev = dest;
            } else
               if (temp_ll != NULL)
                  my_free( temp_ll );
         } else {
            if (temp_ll != NULL)
               my_free( temp_ll );
         }

         /*
          * file too big
          */
         if (*rc != OK)
            error( WARNING, window->bottom_line, main4 );

         if (*rc == OK) {
            copy_line( dest );
            lens = block_start->len - bc;

            assert( bc >= 0 );
            assert( bc <= MAX_LINE_LENGTH );
            assert( lens >= 0 );
            assert( lens <= MAX_LINE_LENGTH );
            assert( rcol >= 0 );
            assert( rcol <= MAX_LINE_LENGTH );

            my_memcpy( g_status.line_buff+rcol, block_start->line+bc, lens );
            g_status.line_buff_len = lens + rcol;
            *rc = un_copy_line( dest, &d_w, TRUE );
            dest->type |= DIRTY;
         }

         if (*rc == OK  &&  ptoul( block_start->next ) != ptoul( block_end )) {
            block_start->next->prev = dest;
            temp_ll->prev = block_end->prev;
            block_end->prev->next = temp_ll;
            dest->next = block_start->next;
         }

         if (*rc == OK) {
            copy_line( block_start );
            detab_linebuff( );
            lend = bc;
            lens = block_end->len - (ec + 1);

            assert( bc >= 0 );
            assert( bc <= MAX_LINE_LENGTH );
            assert( lens >= 0 );
            assert( lens <= MAX_LINE_LENGTH );
            assert( lend >= 0 );
            assert( lend <= MAX_LINE_LENGTH );
            assert( ec + 1 >= 0 );
            assert( ec + 1 <= MAX_LINE_LENGTH );
            assert( lens + lend >= 0 );
            assert( lens + lend <= MAX_LINE_LENGTH );

            my_memcpy( g_status.line_buff+bc, block_end->line+ec+1, lens );
            g_status.line_buff_len = lend + lens;
            *rc = un_copy_line( block_start, &s_w, TRUE );
            block_start->type |= DIRTY;
            block_start->next = block_end->next;
            block_end->next->prev = block_start;
            if (block_end->line != NULL)
               my_free( block_end->line );
            my_free( block_end );
         }
      }
   } else if (action == DELETE) {
      copy_line( block_start );
      lens = block_end->len - (ec + 1);

      assert( bc >= 0 );
      assert( bc <= MAX_LINE_LENGTH );
      assert( lens >= 0 );
      assert( lens <= MAX_LINE_LENGTH );
      assert( ec + 1 >= 0 );
      assert( ec + 1 <= MAX_LINE_LENGTH );
      assert( bc + lens >= 0 );
      assert( bc + lens <= MAX_LINE_LENGTH );

      my_memcpy( g_status.line_buff+bc, block_end->line + ec+1, lens );
      g_status.line_buff_len = bc + lens;
      *rc = un_copy_line( block_start, &s_w, TRUE );
      block_start->type |= DIRTY;
      source = block_start->next;
      block_start->next = block_end->next;
      block_end->next->prev = block_start;
      block_end->next = NULL;
      while (source != NULL) {
         temp_ll = source;
         source = source->next;
         if (temp_ll->line != NULL)
            my_free( temp_ll->line );
         my_free( temp_ll );
      }
   }

   if (*rc == OK) {
      diff = er - br;
      if (action == COPY || action == KOPY || action == MOVE) {
         dest_file->length += diff;
         adjust_windows_cursor( window, diff );
      }
      if (action == DELETE || action == MOVE) {
         source_file->length -= diff;
         adjust_windows_cursor( source_window, -diff );
      }
      if (action == DELETE && source_window->rline >= br) {
         source_window->rline -= diff;
         if (source_window->rline < br)
            source_window->rline = br;
      }
   }

   /*
    * restore all cursors in all windows
    */
   restore_cursors( dest_file );
   if (dest_file != source_file)
      restore_cursors( source_file );
   show_avail_mem( );
}


/*
 * Name:    do_box_block
 * Purpose: delete, move, copy, or kopy a BOX block
 * Date:    June 5, 1991
 * Passed:  window:         pointer to destination window (current window)
 *          source_window:  pointer to source window
 *          action:         block action  --  OVERLAY, FILL, etc...
 *          source_file:    pointer to source file structure
 *          dest_file:      pointer to destination file
 *          source:         pointer to source node
 *          dest:           pointer to destination node
 *          br:             beginning line number in marked block
 *          er:             ending line number in marked block
 *          block_inc:      increment used to number a block
 *          rline:          current line number in destination file
 *          block_num:      starting number when numbering a block
 *          block_just:     LEFT or RIGHT justified numbers in block
 *          fill_char:      character to fill a block
 *          style:          border characters
 *          same:           are source and destination files same? T or F
 *          block_len:      width of box block
 *          bc:             beginning column of block
 *          ec:             ending column of block
 *          rcol:           current column of cursor
 *          rc:             return code
 *
 * jmh 980801: added BORDER action / style parameter.
 * jmh 980811: added JUSTIFY action.
 */
void do_box_block( TDE_WIN *window,  TDE_WIN *source_window,  int action,
                   file_infos *source_file,  file_infos *dest_file,
                   line_list_ptr source,  line_list_ptr dest, long br,
                   long er, long block_inc,
                   long rline, long block_num, int block_just,
                   int fill_char, char *style,
                   int same, int block_len, int bc, int ec, int rcol, int *rc )
{
line_list_ptr p;                /* temporary list pointer */
int  lens;                      /* length of source line */
int  lend;                      /* length of destination line */
int  add;                       /* characters being added from another line */
char *block_buff;
char *swap_buff = NULL;
int  xbc, xec;                  /* temporary column variables */
long li;                        /* temporary line variables */
long dest_add;                  /* number of bytes added to destination file */
TDE_WIN s_w, d_w;       /* a couple of temporary TDE_WINs for BOX stuff */
TDE_WIN *w;
int  padded_file = FALSE;
int (*justify)( TDE_WIN * ) = 0;
int lm = 0, rm = 0;

   if (action == SWAP)
      swap_buff = (char *)calloc( BUFF_SIZE + 2, sizeof(char) );
   block_buff   = (char *)calloc( BUFF_SIZE + 2, sizeof(char) );
   if (block_buff == NULL) {
      if (swap_buff != NULL)
         free( swap_buff );
      error( WARNING, window->bottom_line, block4 );
      *rc = ERROR;
      return;
   }

   /*
    * see if we need to add pad lines at eof.
    */
   if (action == MOVE || action == COPY || action == KOPY ||
       action == SWAP || action == OVERLAY) {
      dest_add = rline - br;
      if (dest_add + er > dest_file->length) {
         dest_add -= (dest_file->length - er);
         p = dest_file->line_list_end->prev;
         for (; dest_add > 0  &&  *rc == OK; dest_add--)
            *rc = pad_dest_line( window, dest_file, p );
         padded_file = TRUE;
      }
   }
   if (*rc == OK) {

      dup_window_info( &s_w, source_window );
      dup_window_info( &d_w, window );
      s_w.rline   = br;
      s_w.ll      = source;
      s_w.visible = FALSE;
      d_w.rline   = rline;
      d_w.ll      = dest;
      d_w.visible = FALSE;

      if (action == JUSTIFY) {
         justify = (g_status.command == BlockLeftJustify)  ? flush_left  :
                   (g_status.command == BlockRightJustify) ? flush_right :
                                                             flush_center;
         lm = mode.left_margin;
         rm = mode.right_margin;
         mode.left_margin  = 0;
         mode.right_margin = block_len - 1;
      }

      /*
       * If copying, kopying, moving, overlaying or swapping, and the block
       * overlaps the destination, need to move backward through the block.
       */
      if (same  &&  rline > br && rline <= er  &&
          (action == COPY || action == KOPY || action == MOVE ||
           action == SWAP || action == OVERLAY)) {

         /*
          * move source and dest pointers to the end of the block
          */
         for (li = er - br; li > 0; li--) {
            load_undo_buffer( dest_file, dest->line, dest->len );
            source = source->next;
            dest   = dest->next;
            ++s_w.rline;
            ++d_w.rline;
         }

         for (li = er; *rc == OK  &&  li >= br  &&  !g_status.control_break;
              li--, s_w.rline--, d_w.rline--) {
            lens = find_end( source->line, source->len );
            lend = find_end( dest->line, dest->len );
            if (lens != 0 || lend != 0) {
               load_box_buff( block_buff, source, bc, ec, ' ' );
               if (action == SWAP)
                  load_box_buff( swap_buff, dest, rcol, rcol+block_len, ' ' );
               *rc = copy_buff_2file( &d_w, block_buff, dest, rcol,
                                      block_len, action );
               dest->type |= DIRTY;
               if (action == SWAP && *rc == OK) {
                  *rc = copy_buff_2file( &s_w, swap_buff, source, bc,
                                         block_len, action );
                  source->type |= DIRTY;
               }
            }
            if (action == MOVE) {
               if (lens >= (bc + 1)) {
                  source->type |= DIRTY;
                  add = block_len;
                  if (lens <= (ec + 1))
                     add = lens - bc;
                  if (add > 0)
                     *rc = delete_box_block( &s_w, source, bc, add );
               }
            }
            source = source->prev;
            dest   = dest->prev;
         }
      } else {
         if (action == FILL)
            block_fill( block_buff, fill_char, block_len );

         for (li = br; *rc == OK  &&  li <= er  &&  !g_status.control_break;
              li++, s_w.rline++, d_w.rline++) {
            switch (action) {
               case BORDER  :
               case FILL    :
               case NUMBER  :
               case DELETE  :
               case MOVE    :
               case SWAP    :
                  load_undo_buffer( source_file, source->line, source->len );
                  if (action != SWAP)
                     break;
                  /* else fall through */
               case COPY    :
               case KOPY    :
               case OVERLAY :
                  load_undo_buffer( dest_file, dest->line, dest->len );
                  break;
            }
            lens = find_end( source->line, source->len );
            lend = find_end( dest->line, dest->len );

            /*
             * with FILL and NUMBER operations, we're just adding chars
             *   to the file at the source location.  we don't have to
             *   worry about bookkeeping.
             * jmh 980801: as with BORDER.
             * jmh 980811: as with JUSTIFY.
             */
            if (action == FILL   || action == NUMBER ||
                action == BORDER || action == JUSTIFY) {
               if (action == NUMBER) {
                  number_block_buff( block_buff, block_len, block_num,
                                     block_just );
                  block_num += block_inc;
               } else if (action == BORDER) {
                  if (li == br) {
                     block_buff[0] = style[0];
                     block_buff[block_len-1] = style[1];
                     memset( block_buff+1, style[6], block_len-2 );
                  } else if (li == er) {
                     block_buff[0] = style[2];
                     block_buff[block_len-1] = style[3];
                     memset( block_buff+1, style[7], block_len-2 );
                  } else {
                     load_box_buff( block_buff, source, bc, ec, ' ' );
                     block_buff[0] = style[4];
                     block_buff[block_len-1] = style[5];
                  }
               } else if (action == JUSTIFY) {
                  load_box_buff( g_status.line_buff, source, bc, ec, ' ' );
                  g_status.line_buff_len = block_len;
                  g_status.copied = TRUE;
                  justify( &s_w );
                  memset( block_buff, ' ', block_len );
                  memcpy( block_buff, g_status.line_buff,
                          g_status.line_buff_len );
                  g_status.copied = FALSE;
               }

               *rc = copy_buff_2file( &s_w, block_buff, source, rcol,
                                      block_len, action );
               source->type |= DIRTY;

            /*
             * if we are doing a BOX action and both the source and
             * destination are 0 then we have nothing to do.
             */
            } else if (lens != 0 || lend != 0) {

               /*
                * do actions that may require adding to file
               if (action == MOVE     ||  action == COPY || action == KOPY ||
                   action == OVERLAY  ||  action == SWAP)
                */
               if (action != DELETE) {
                  xbc = bc;
                  xec = ec;
                  if (action != OVERLAY  &&  action != SWAP  &&  same) {
                     if (rcol < bc && rline > br && rline <= er)
                        if (li >= rline) {
                           xbc = bc + block_len;
                           xec = ec + block_len;
                        }
                  }
                  load_box_buff( block_buff, source, xbc, xec, ' ' );
                  if (action == SWAP)
                     load_box_buff(swap_buff, dest, rcol, rcol+block_len, ' ');
                  *rc = copy_buff_2file( &d_w, block_buff, dest, rcol,
                                         block_len, action );
                  dest->type |= DIRTY;
                  if (action == SWAP && *rc == OK) {
                     *rc = copy_buff_2file( &s_w, swap_buff, source, xbc,
                                            block_len, action );
                     source->type |= DIRTY;
                  }
               }

               /*
                * do actions that may require deleting from file
                */
               if (action == MOVE || action == DELETE) {
                  lens = find_end( source->line, source->len );
                  if (lens >= (bc + 1)) {
                     source->type |= DIRTY;
                     add = block_len;
                     xbc = bc;
                     if (lens <= (ec + 1))
                        add = lens - bc;
                     if (same && action == MOVE) {
                        if (rcol < bc && rline >= br && rline <= er)
                           if (li >= rline) {
                              xbc = bc + block_len;
                              if (lens <= (ec + block_len + 1))
                                 add = lens - xbc;
                           }
                     }
                     if (add > 0)
                        *rc = delete_box_block( &s_w, source, xbc, add );
                  }
               }
            }
            source = source->next;
            dest   = dest->next;
         }
      }
      if (action == JUSTIFY) {
         mode.left_margin  = lm;
         mode.right_margin = rm;
      }
   }
   if (padded_file) {
      w = g_status.window_list;
      while (w != NULL) {
         if (w->file_info == dest_file  &&  w->visible)
            show_size( w );
         w = w->next;
      }
   }
   free( block_buff );
   free( swap_buff );
   show_avail_mem( );
}


/*
 * Name:    load_box_buff
 * Class:   helper function
 * Purpose: copy the contents of a BOX to a block buffer.
 * Date:    June 5, 1991
 * Passed:  block_buff: local buffer for block moves
 *          ll:         node to source line in file to load
 *          bc:     beginning column of BOX. used only in BOX operations.
 *          ec:     ending column of BOX. used only in BOX operations.
 *          filler: character to fill boxes that end past eol
 * Notes:   For BOX blocks, there are several things to take care of:
 *            1) The BOX begins and ends within a line - just copy the blocked
 *            characters to the block buff.  2) the BOX begins within a line
 *            but ends past the eol - copy all the characters within the line
 *            to the block buff then fill with padding.  3) the BOX begins and
 *            ends past eol - fill entire block buff with padding (filler).
 *          the fill character varies with the block operation.  for sorting
 *            a box block, the fill character is '\0'.  for adding text to
 *            the file, the fill character is a space.
 */
void load_box_buff( char *block_buff, line_list_ptr ll, int bc, int ec,
                    char filler )
{
int len;
int avlen;
register int i;
register char *bb;
text_ptr s;

   assert( bc >= 0 );
   assert( ec >= bc );
   assert( ec < MAX_LINE_LENGTH );

   bb = block_buff;
   len = ll->len;
   s = detab_a_line( ll->line, &len );
   /*
    * block start may be past eol
    */
   if (len < ec + 1) {
      /*
       * does block start past eol? - fill with pad
       */
      assert( ec + 1 - bc >= 0 );

      memset( block_buff, filler, (ec + 1) - bc );
      if (len >= bc) {
         /*
          * block ends past eol - fill with pad
          */
         avlen = len - bc;
         s += bc;
         for (i=avlen; i>0; i--)
            *bb++ = *s++;
      }
   } else {
      /*
       * block is within line - copy block to buffer
       */
      avlen = (ec + 1) - bc;
      s += bc;
      for (i=avlen; i>0; i--)
         *bb++ = *s++;
   }
}


/*
 * Name:    copy_buff_2file
 * Class:   helper function
 * Purpose: copy the contents of block buffer to destination file
 * Date:    June 5, 1991
 * Passed:  window:     pointer to current window
 *          block_buff: local buffer for moves
 *          dest:       pointer to destination line in destination file
 *          rcol:       if in BOX mode, destination column in destination file
 *          block_len:  if in BOX mode, width of block to copy
 *          action:     type of block action
 * Notes:   In BOX mode, the destination line has already been prepared.
 *          Just copy the BOX buffer to the destination line.
 */
int  copy_buff_2file( TDE_WIN *window, char *block_buff, line_list_ptr dest,
                      int rcol, int block_len, int action )
{
char *s;
char *d;
int len;
int pad;
int add;

   copy_line( dest );
   if (mode.inflate_tabs)
      detab_linebuff( );

   len = g_status.line_buff_len;

   assert( len >= 0 );
   assert( len < MAX_LINE_LENGTH );
   assert( rcol >= 0 );
   assert( rcol < MAX_LINE_LENGTH );
   assert( block_len >= 0 );
   assert( block_len < BUFF_SIZE );

   if (rcol > len) {
      pad = rcol - len;

      assert( pad >= 0 );
      assert( pad < MAX_LINE_LENGTH );

      memset( g_status.line_buff + len, ' ', pad );
      len += pad;
   }

   s = g_status.line_buff + rcol;

   /*
    * s is pointing to location to perform BOX operation.  If we do a
    * FILL or OVERLAY, we do not necessarily add any extra space.  If the
    * line does not extend all the thru the BOX then we add.
    * we always add space when we COPY, KOPY, or MOVE
    */
   if (action == COPY || action == KOPY || action == MOVE) {
      d = s + block_len;
      add = len - rcol;

      assert( add >= 0 );
      assert( add < MAX_LINE_LENGTH );

      memmove( d, s, add );
      len += block_len;
   } else {
      add = len - rcol;
      if (add < block_len) {
         pad = block_len - add;

         assert( pad >= 0 );
         assert( pad < MAX_LINE_LENGTH );

         memset( g_status.line_buff + len, ' ', pad );
         len += pad;
      }
   }

   assert( rcol + block_len <= len );
   assert( len >= 0 );
   assert( len < MAX_LINE_LENGTH );

   memmove( s, block_buff, block_len );
   g_status.line_buff_len = len;
   if (mode.inflate_tabs)
      entab_linebuff( );
   return( un_copy_line( dest, window, TRUE ) );
}


/*
 * Name:    block_fill
 * Class:   helper function
 * Purpose: fill the block buffer with character
 * Date:    June 5, 1991
 * Passed:  block_buff: local buffer for moves
 *          fill_char:  fill character
 *          block_len:  number of columns in block
 * Notes:   Fill block_buffer for block_len characters using fill_char.  This
 *          function is used only for BOX blocks.
 */
void block_fill( char *block_buff, int fill_char, int block_len )
{
   assert( block_len >= 0 );
   assert( block_len < BUFF_SIZE );
   assert( block_buff != NULL );

   memset( block_buff, fill_char, block_len );
}


/*
 * Name:    number_block_buff
 * Class:   helper function
 * Purpose: put a number into the block buffer
 * Date:    June 5, 1991
 * Passed:  block_buff: local buffer for moves
 *          block_len:  number of columns in block
 *          block_num:  long number to fill block
 *          just:       LEFT or RIGHT justified?
 * Notes:   Fill block_buffer for block_len characters with number.
 *          This function is used only for BOX blocks.
 */
void number_block_buff( char *block_buff, int block_len, long block_num,
                        int just )
{
int len;                /* length of number buffer */
int i;
char temp[MAX_COLS+2];

   assert( block_len >= 0 );
   assert( block_len < BUFF_SIZE );

   block_fill( block_buff, ' ', block_len );
   len = strlen( my_ltoa( block_num, temp, 10 ) );
   if (just == RIGHT) {
      block_len--;
      len--;
      for (;block_len >= 0 && len >= 0; block_len--, len--)
         block_buff[block_len] = temp[len];
   } else {
      for (i=0; block_len > 0 && i < len; block_len--, i++)
         block_buff[i] = temp[i];
   }
}


/*
 * Name:    restore_cursors
 * Class:   helper function
 * Purpose: a file has been modified - must restore all cursor pointers
 * Date:    June 5, 1991
 * Passed:  file:  pointer to file with changes
 * Notes:   Go through the window list and adjust the cursor pointers
 *          as needed.
 */
void restore_cursors( file_infos *file )
{
register TDE_WIN *window;
line_list_ptr ll;
long n;

   assert( file != NULL );

   window = g_status.window_list;
   while (window != NULL) {
      if (window->file_info == file) {
         window->bin_offset = 0;
         if (window->rline < 0L)
            window->rline = 0L;
         if (window->rline > file->length)
            window->rline = file->length;
         ll = file->line_list;
         for (n = 0L; n < window->rline; n++) {
            window->bin_offset += ll->len;
            ll = ll->next;
         }
         window->ll = ll;
         n = window->top_line + window->ruler;
         if (window->rline < (window->cline - n))
            window->cline = (int)window->rline + (int)n;
         if (window->cline < n)
            window->cline = (int)n;
         if (window->visible)
            show_size( window );
      }
      window = window->next;
   }
}


/*
 * Name:    delete_box_block
 * Class:   helper function
 * Purpose: delete the marked text
 * Date:    June 5, 1991
 * Passed:  s_w:    source window
 *          source: pointer to line with block to delete
 *          bc:     beginning column of block - BOX mode only
 *          add:    number of characters in block to delete
 * Notes:   Used only for BOX blocks.  Delete the block.
 */
int  delete_box_block( TDE_WIN *s_w, line_list_ptr source, int bc, int add )
{
char *s;
int number;

   assert( s_w != NULL );
   assert( source != NULL );
   assert( bc >= 0 );
   assert( bc < MAX_LINE_LENGTH );
   assert( add >= 0 );
   assert( add < MAX_LINE_LENGTH );

   copy_line( source );
   detab_linebuff( );
   number = g_status.line_buff_len - bc;
   s = g_status.line_buff + bc + add;

   assert( number >= 0 );
   assert( number < MAX_LINE_LENGTH );
   assert( bc + add >= 0 );
   assert( bc + add < MAX_LINE_LENGTH );
   assert( add <= g_status.line_buff_len );

   memmove( s - add, s, number );
   g_status.line_buff_len -= add;
   entab_linebuff( );
   return( un_copy_line( source, s_w, TRUE ) );
}


/*
 * Name:    check_block
 * Class:   helper function
 * Purpose: To check that the block is still valid.
 * Date:    June 5, 1991
 * Notes:   After some editing, the marked block may not be valid. For example,
 *          deleting all the lines in a block in another window.  We don't
 *          need to keep up with the block text pointers while doing normal
 *          editing; however, we need to refresh them before doing block stuff.
 *
 * jmh 980525: only unmark if a block is marked.
 */
void check_block( void )
{
register file_infos *file;
TDE_WIN filler;

   file = g_status.marked_file;
   if (file == NULL || file->block_br > file->length) {
      if (g_status.marked == TRUE)
         unmark_block( &filler );
   } else {
      if (file->block_er > file->length)
         file->block_er = file->length;
      find_begblock( file );
      find_endblock( file );
   }
}


/*
 * Name:    find_begblock
 * Class:   helper editor function
 * Purpose: find the beginning line in file with marked block
 * Date:    June 5, 1991
 * Passed:  file: file containing marked block
 * Notes:   file->block_start contains starting line of marked block.
 */
void find_begblock( file_infos *file )
{
line_list_ptr ll;
long li;           /* line counter (long i) */

   assert( file != NULL );
   assert( file->length > 0 );

   ll = file->line_list->next;
   for (li=1; li<file->block_br && ll->next != NULL; li++)
      ll = ll->next;

   file->block_start = ll;
}


/*
 * Name:    find_endblock
 * Class:   helper function
 * Purpose: find the ending line in file with marked block
 * Date:    June 5, 1991
 * Passed:  file: file containing marked block
 * Notes:   If in LINE mode, file->block_end is set to end of line of last
 *          line in block.  If in BOX mode, file->block_end is set to
 *          beginning of last line in marked block.  If the search for the
 *          ending line of the marked block goes past the eof, set the
 *          ending line of the block to the last line in the file.
 */
void find_endblock( file_infos *file )
{
line_list_ptr ll; /* start from beginning of file and go to end */
long i;           /* line counter */
register file_infos *fp;

   assert( file != NULL );
   assert( file->block_start != NULL );

   fp = file;
   ll = fp->block_start;
   if (ll != NULL) {
      for (i=fp->block_br;  i < fp->block_er && ll->next != NULL; i++)
         ll = ll->next;
      if (ll != NULL)
         fp->block_end = ll;
      else {

         /*
          * last line in marked block is NULL.  if LINE block, set end to
          * last character in the file.  if STREAM or BOX block, set end to
          * start of last line in file.  ending row, or er, is then set to
          * file length.
          */
         fp->block_end = fp->line_list_end->prev;
         fp->block_er = fp->length;
      }
   }
}


/*
 * Name:    block_write
 * Class:   primary editor function
 * Purpose: To write the currently marked block to a disk file.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   If the file already exists, the user gets to choose whether
 *           to overwrite or append.
 */
int  block_write( TDE_WIN *window )
{
int prompt_line;
int rc;
file_infos *file;
int block_type;
char temp[MAX_COLS+2];
DISPLAY_BUFF;
fattr_t fattr;

   /*
    * make sure block is marked OK
    */
   entab_linebuff( );
   rc = un_copy_line( window->ll, window, TRUE );

   check_block( );
   if (rc == OK  &&  g_status.marked == TRUE) {
      prompt_line = window->bottom_line;
      file        = g_status.marked_file;

      assert( file != NULL );

      block_type  = file->block_type;

      /*
       * find out which file to write to
       */

      save_screen_line( 0, prompt_line, display_buff );
      *g_status.rw_name = '\0';
      if (get_name( block6, prompt_line, g_status.rw_name ) == OK) {
         /*
          * if the file exists, find out whether to overwrite or append
          */
         rc = get_fattr( g_status.rw_name, &fattr );
         if (rc == OK) {
            /*
             * file exists. overwrite or append?
             */
            switch (get_response( block7, prompt_line, R_PROMPT,
                           2, L_OVERWRITE, A_OVERWRITE, L_APPEND, A_APPEND)) {
               case A_OVERWRITE :
                  change_mode( g_status.rw_name, prompt_line );
                  /*
                   * writing block to
                   */
                  combine_strings( temp, block8, g_status.rw_name, "'" );
                  s_output( temp, prompt_line, 0, g_display.message_color );

#if defined( __UNIX__ )
                  refresh( );
#endif

                  rc = hw_save( g_status.rw_name, file, file->block_br,
                                file->block_er, block_type );
                  if (rc == ERROR)
                     /*
                      * could not write block
                      */
                     error( WARNING, prompt_line, block9 );
                  break;
               case A_APPEND :
                  /*
                   * appending block to
                   */
                  combine_strings( temp, block10, g_status.rw_name, "'" );
                  s_output( temp, prompt_line, 0, g_display.message_color );

#if defined( __UNIX__ )
                  refresh( );
#endif

                  rc = hw_append( g_status.rw_name, file, file->block_br,
                                  file->block_er, block_type );
                  if (rc == ERROR)
                     /*
                      * could not append block
                      */
                     error( WARNING, prompt_line, block11 );
                  break;
               default :
                  rc = ERROR;
                  break;
            }
         } else {
            /*
             * writing block to
             */
            combine_strings( temp, block12, g_status.rw_name, "'" );
            s_output( temp, prompt_line, 0, g_display.message_color );

#if defined( __UNIX__ )
            refresh( );
#endif

            if (hw_save( g_status.rw_name, file, file->block_br,
                         file->block_er, block_type ) == ERROR) {
               /*
                * could not write block
                */
               error( WARNING, prompt_line, block9 );
               rc = ERROR;
            }
         }
      }
      restore_screen_line( 0, prompt_line, display_buff );
   } else
      rc = ERROR;
   return( rc );
}


/*
 * Name:     block_print
 * Class:    primary editor function
 * Purpose:  Print an entire file or the currently marked block.
 * Date:     June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 * Passed:   window:  pointer to current window
 * Notes:    With the added Critical Error Handler routine, let's fflush
 *           the print buffer first.
 *
 * jmh 980528: modified the block line testing code.
 */
int  block_print( TDE_WIN *window )
{
int  col;
int  prompt_line;
line_list_ptr block_start;   /* start of block in file */
file_infos *file;
int  block_type;
char *p;
int  len;
int  bc;
int  ec;
int  last_c;
long lbegin;
long lend;
long l;
int  color;
int  rc;
char answer[MAX_COLS+2];
DISPLAY_BUFF;

   color = g_display.message_color;
   entab_linebuff( );
   if (un_copy_line( window->ll, window, TRUE ) == ERROR)
      return( ERROR );
   rc = OK;

   prompt_line = window->bottom_line;
   save_screen_line( 0, prompt_line, display_buff );
   /*
    * print entire file or just marked block?
    */
   col = get_response( block13, prompt_line, R_ALL, 2, L_FILE,  L_FILE,
                                                       L_BLOCK, L_BLOCK );
   if (col == ERROR)
      rc = ERROR;

   if (rc == OK) {
      /*
       * if everything is everything, flush the printer before we start
       *   printing.  then, check the critical error flag after the flush.
       */
      fflush( my_stdprn );
      if (ceh.flag == ERROR)
         rc = ERROR;
   }

   if (rc != ERROR) {
      file = window->file_info;
      block_type  = NOTMARKED;
      lend =  l   = file->length;
      block_start = file->line_list->next;

      if (col == L_BLOCK) {
         check_block( );
         if (g_status.marked == TRUE) {
            file        = g_status.marked_file;
            block_start = file->block_start;
            block_type  = file->block_type;
            lend =   l  = file->block_er + 1l - file->block_br;
         } else
            rc = ERROR;
      }

      if (rc != ERROR) {
         eol_clear( 0, prompt_line, color );
         /*
          * printing line   of    press control-break to cancel.
          */
         s_output( block14, prompt_line, 0, color );
         my_ltoa( l, answer, 10 );
         s_output( answer, prompt_line, 25, color );
         xygoto( BLOCK14_LINE_SLOT, prompt_line );

         /*
          * bc and ec are used only in BOX and STREAM blocks.
          * last_c is last column in a BOX block
          */
         bc = file->block_bc;
         ec = file->block_ec;
         last_c = ec + 1 - bc;

         p = g_status.line_buff;
         lbegin = 1;
         for (col=OK; l>0 && col == OK && !g_status.control_break; l--) {
            my_ltoa( lbegin, answer, 10 );
            s_output( answer, prompt_line, BLOCK14_LINE_SLOT, color );
            g_status.copied = FALSE;
            if (block_type == BOX || (block_type == STREAM && lend == 1)) {
               load_box_buff( p, block_start, bc, ec, ' ' );
               len = last_c;
            } else {
               copy_line( block_start );
               detab_linebuff( );
               len = g_status.line_buff_len;
               if (block_type == STREAM) {
                  if (lbegin == 1) {
                     if (bc > len)
                        len = 0;
                     else {
                        len -= bc;

                        assert( len >= 0 );
                        assert( len < MAX_LINE_LENGTH );

                        my_memcpy( p, p + bc, len );
                     }
                  } else if (l == 1L) {
                     if (len > ec + 1)
                        len = ec + 1;
                  }
               }
            }

            assert( len >= 0 );
            assert( len < MAX_LINE_LENGTH );

            *(p+len) = '\r';
            ++len;
            *(p+len) = '\n';
            ++len;
            if (fwrite( p, sizeof( char ), len, my_stdprn ) < (unsigned)len ||
                ceh.flag == ERROR)
               col = ERROR;
            block_start = block_start->next;
            ++lbegin;
         }
         g_status.copied = FALSE;
         if (ceh.flag != ERROR)
            fflush( my_stdprn );
         else
            rc = ERROR;
      }
   }
   g_status.copied = FALSE;
   restore_screen_line( 0, prompt_line, display_buff );
   return( rc );
}


/*
 * Name:    get_block_fill_char
 * Class:   helper function
 * Purpose: get the character to fill marked block.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 *          c: address of character to fill block
 * jmh 980723: recognize macros.
 */
int  get_block_fill_char( TDE_WIN *window, int *c )
{
register long key;
int prompt_line;
int rc;
DISPLAY_BUFF;

   rc = OK;
   prompt_line = window->bottom_line;
   save_screen_line( 0, prompt_line, display_buff );
   /*
    * enter character to fill block (esc to exit)
    */
   set_prompt( block15, prompt_line );
   key = getkey_macro( );
   if (key < 256)
      *c = (int)key;
   else
      rc = ERROR;
   restore_screen_line( 0, prompt_line, display_buff );
   return( rc );
}


/*
 * Name:    get_block_numbers
 * Class:   helper function
 * Purpose: get the starting number and increment
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 *          block_num: address of number to start numbering
 *          block_inc: address of number to add to block_num
 *          just:      left or right justify numbers in block?
 */
int  get_block_numbers( TDE_WIN *window, long *block_num, long *block_inc,
                        int *just )
{
int prompt_line;
register int rc;
char answer[MAX_COLS+2];

   prompt_line = window->bottom_line;

   /*
    * don't assume anything on starting number - start w/ null string.
    */
   answer[0] = '\0';
   /*
    * enter starting number
    */
   rc = get_name( block16, prompt_line, answer );
   if (answer[0] == '\0')
      rc = ERROR;
   if (rc != ERROR) {
      *block_num = atol( answer );

      /*
       * assume increment is 1
       */
      answer[0] = '1';
      answer[1] = '\0';
      /*
       * enter increment
       */
      rc = get_name( block17, prompt_line, answer );
      if (answer[0] == '\0')
         rc = ERROR;
      if (rc != ERROR) {
         *block_inc = atol( answer );

         /*
          * left or right justify?
          */
         rc = get_response( block18, prompt_line, R_ALL, 2, L_RIGHT, RIGHT,
                                                            L_LEFT,  LEFT );
         if (rc != ERROR) {
            *just = rc;
            rc = OK;
         }
      }
   }
   /*
    * if everything is everything then return code = OK.
    */
   return( rc );
}


/*
 * Name:    block_trim_trailing
 * Class:   primary editor function
 * Purpose: Trim trailing space in a LINE block.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Use copy_line and un_copy_line to do the work.
 */
int  block_trim_trailing( TDE_WIN *window )
{
int prompt_line;
int rc;
line_list_ptr p;             /* pointer to block line */
file_infos *file;
TDE_WIN *sw, s_w;
long er;
int  trailing;               /* save trailing setting */

   /*
    * make sure block is marked OK and that this is a LINE block
    */
   prompt_line = window->bottom_line;
   entab_linebuff( );
   rc = un_copy_line( window->ll, window, TRUE );
   check_block( );
   if (rc != ERROR && g_status.marked == TRUE) {

      trailing = mode.trailing;
      mode.trailing = TRUE;
      file = g_status.marked_file;
      if (file->block_type != LINE) {
         /*
          * can only trim trailing space in line blocks
          */
         error( WARNING, prompt_line, block21 );
         return( ERROR );
      }

      /*
       * initialize everything
       */
      sw = g_status.window_list;
      for (; ptoul( sw->file_info ) != ptoul( file );)
         sw = sw->next;
      if (mode.do_backups == TRUE) {
         file->modified = TRUE;
         rc = backup_file( sw );
      }
      dup_window_info( &s_w, sw );

      /*
       * set window to invisible so the un_copy_line function will
       * not display the lines while trimming.
       */
      s_w.visible = FALSE;

      p  = file->block_start;
      er = file->block_er;
      s_w.rline = file->block_br;
      for (; rc == OK && s_w.rline <= er  &&  !g_status.control_break;
           s_w.rline++) {
         /*
          * use the line buffer to trim space.
          */
         copy_line( p );
         rc = un_copy_line( p, &s_w, TRUE );
         p = p->next;
      }

      /*
       * IMPORTANT:  we need to reset the copied flag because the cursor may
       * not necessarily be on the last line of the block.
       */
      g_status.copied = FALSE;
      file->dirty = GLOBAL;
      mode.trailing = trailing;
      show_avail_mem( );
   }
   return( rc );
}


/*
 * Name:    block_email_reply
 * Class:   primary editor function
 * Purpose: insert the standard replay character '>' at beginning of line
 * Date:    June 5, 1992
 * Passed:  window:  pointer to current window
 * Notes:   it is customary to prepend "> " to the initial text and
 *             ">" to replies to replies to etc...
 */
int  block_email_reply( TDE_WIN *window )
{
int prompt_line;
int add;
int len;
int rc;
char *source;    /* source for block move to make room for c */
char *dest;      /* destination for block move */
line_list_ptr p;                     /* pointer to block line */
file_infos *file;
TDE_WIN *sw, s_w;
long er;

   /*
    * make sure block is marked OK and that this is a LINE block
    */
   prompt_line = window->bottom_line;
   entab_linebuff( );
   rc = un_copy_line( window->ll, window, TRUE );
   check_block( );
   if (rc != ERROR  &&  g_status.marked == TRUE) {
      file = g_status.marked_file;
      if (file->block_type != LINE) {
         /*
          * can only reply line blocks
          */
         error( WARNING, prompt_line, block25 );
         return( ERROR );
      }

      /*
       * find a window that points to the file with a marked block.
       */
      sw = g_status.window_list;
      for (; ptoul( sw->file_info ) != ptoul( file );)
         sw = sw->next;
      if (mode.do_backups == TRUE) {
         file->modified = TRUE;
         rc = backup_file( sw );
      }

      /*
       * use a local window structure to do the dirty work.  initialize
       *   the local window structure to the beginning of the marked
       *   block.
       */
      dup_window_info( &s_w, sw );

      /*
       * set s_w to invisible so the un_copy_line function will
       * not display the lines while doing block stuff.
       */
      s_w.visible = FALSE;
      s_w.rline = file->block_br;
      p  = file->block_start;
      er = file->block_er;

      /*
       * for each line in the marked block, prepend the reply character(s)
       */
      for (; rc == OK  &&  s_w.rline <= er  &&  !g_status.control_break;
                                                             s_w.rline++) {

         /*
          * put the line in the g_status.line_buff.  use add to count the
          *   number of characters to insert at the beginning of a line.
          *   the original reply uses "> ", while replies to replies use ">".
          */
         copy_line( p );
         if (*(p->line) == '>')
            add = 1;
         else
            add = 2;

         /*
          * see if the line has room to add the ">" character.  if there is
          *   room, move everything down to make room for the
          *   reply character(s).
          */
         len = g_status.line_buff_len;
         if (len + add < MAX_LINE_LENGTH) {
            source = g_status.line_buff;
            dest = source + add;

            assert( len >= 0 );
            assert( len < MAX_LINE_LENGTH );
            assert( len + add < MAX_LINE_LENGTH );

            memmove( dest, source, len );
            *source = '>';
            if (add > 1)
              *(source+1) = ' ';
            g_status.line_buff_len = len + add;
            rc = un_copy_line( p, &s_w, TRUE );
         }
         p = p->next;
         g_status.copied = FALSE;
      }

      /*
       * IMPORTANT:  we need to reset the copied flag because the cursor may
       * not necessarily be on the last line of the block.
       */
      g_status.copied = FALSE;
      syntax_check_block( file->block_br, er, file->block_start, file->syntax );
      file->dirty = GLOBAL;
      show_avail_mem( );
   }
   return( OK );
}


/*
 * Name:     block_convert_case
 * Class:    primary editor function
 * Purpose:  convert characters to lower case, upper case, strip hi bits,
 *           or e-mail functions
 * Date:     June 5, 1991
 * Modified: August 10, 1997, Jason Hood - correctly handles real tabs.
 *                             corrected STREAM beginning/ending lines.
 *           November 14, Jason Hood - corrected one-line STREAM.
 * Passed:   window:  pointer to current window
 */
int  block_convert_case( TDE_WIN *window )
{
int  len;
int  block_type;
line_list_ptr begin;
register file_infos *file;
TDE_WIN *sw;
long number;
long er;
unsigned int count;
int  bc, ec;
int  block_len;
int  rc;
void (*char_func)( text_ptr, size_t );

   /*
    * make sure block is marked OK
    */
   entab_linebuff( );
   if (un_copy_line( window->ll, window, TRUE ) == ERROR)
      return( ERROR );
   rc = OK;
   check_block( );
   if (g_status.marked == TRUE) {

      /*
       * set char_func() to the required block function in tdeasm.c
       */
      switch (g_status.command) {
         case BlockUpperCase  :
            char_func = upper_case;
            break;
         case BlockLowerCase  :
            char_func = lower_case;
            break;
         case BlockRot13      :
            char_func = rot13;
            break;
         case BlockFixUUE     :
            char_func = fix_uue;
            break;
         case BlockStripHiBit :
            char_func = strip_hi;
            break;
         default :
            return( ERROR );
      }

      file  = g_status.marked_file;
      file->modified = TRUE;
      if (mode.do_backups == TRUE) {
         sw = g_status.window_list;
         for (; ptoul( sw->file_info ) != ptoul( file );)
            sw = sw->next;
         rc = backup_file( sw );
      }

      if (rc == OK) {
         block_type = file->block_type;
         ec = file->block_ec;

         begin  = file->block_start;

         er = file->block_er;
         block_len = ec + 1 - file->block_bc;
         for (number=file->block_br; number <= er; number++) {
            begin->type |= DIRTY;
            copy_line( begin );
            detab_linebuff( );
            count = len = g_status.line_buff_len;
            bc = 0;
            if (block_type == BOX ||
                (block_type == STREAM && number == file->block_br &&
                                         number == er)) {
               bc = file->block_bc;
               count =  len >= ec ? block_len : len - bc;
            } else if (block_type == STREAM) {
               if (number == file->block_br) {
                  bc = file->block_bc;
                  count = len - bc;
               } else if (number == er) {
                  if (ec < len)
                     count = ec+1;
               }
            }
            if (len > bc) {

               assert( count < MAX_LINE_LENGTH );
               assert( bc >= 0 );
               assert( bc < MAX_LINE_LENGTH );

               (*char_func)( (text_ptr)g_status.line_buff+bc, count );
            }
            entab_linebuff( );
            un_copy_line( begin, window, TRUE );
            begin = begin->next;
         }

         /*
          * IMPORTANT:  we need to reset the copied flag because the cursor may
          * not necessarily be on the last line of the block.
          */
         g_status.copied = FALSE;
         file->dirty = GLOBAL;
      }
   } else
      rc = ERROR;
   return( rc );
}


/*
 * Name:    upper_case
 * Purpose: To convert all lower case characters to upper characters
 * Date:    June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 * Passed:  s:    the starting point
 *          count: number of characters to convert
 * Returns: none
 * Notes:   upper_lower[] is defined in bottom of prompts.c.  modify as
 *           needed for your favorite language.
 */
void upper_case( text_ptr s, size_t count )
{
   if (s != NULL) {
      for (; count > 0; s++, count-- ) {
         if (bj_islower( *s ))          /* if lowercase letter */
            *s = upper_lower[*s];       /*    then upcase it   */
      }
   }
}


/*
 * Name:    lower_case
 * Purpose: To convert all upper case characters to lower characters
 * Date:    June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 * Passed:  s:    the starting point
 *          count: number of characters to convert
 * Returns: none
 * Notes:   upper_lower[] is defined in bottom of prompts.c.  modify as
 *           needed for your favorite language.
 */
void lower_case( text_ptr s, size_t count )
{
   if (s != NULL) {
      for (; count > 0; s++, count-- ) {
         if (bj_isupper( *s ))          /* if uppercase letter */
            *s = upper_lower[*s];       /*    then lowcase it  */
      }
   }
}


/*
 * Name:    rot13
 * Purpose: To rotate all alphabet characters by 13
 * Date:    June 5, 1991
 * Passed:  s:    the starting point
 *          count: number of characters to convert
 * Returns: none
 * Notes:   simple rot13
 *          i really don't know how to handle rot13 for alphabets
 *           other than English.
 */
void rot13( text_ptr s, size_t count )
{
register size_t c;

   if (s != NULL) {
      for (; count > 0; s++, count-- ) {
         c = *s;
         if (c >= 'a'  &&  c <=  'm')
            c += 13;
         else if (c >= 'n'  &&  c <= 'z')
            c -= 13;
         else if (c >= 'A'  &&  c <= 'M')
            c += 13;
         else if (c >= 'N'  &&  c <= 'Z')
            c -= 13;
         *s = (unsigned char)c;
      }
   }
}


/*
 * Name:    fix_uue
 * Purpose: To fix EBCDIC ==> ASCII translation problem
 * Date:    June 5, 1991
 * Passed:  s:    the starting point
 *          count: number of characters to convert
 * Returns: none
 * Notes:   to fix the EBCDIC to ASCII translation problem, three characters
 *           need to be changed,  0x5d -> 0x7c, 0xd5 -> 0x5b, 0xe5 -> 0x5d
 */
void fix_uue( text_ptr s, size_t  count )
{
   if (s != NULL) {
      for (; count > 0; s++, count-- ) {
         switch (*s) {
            case 0x5d :
               *s = 0x7c;
               break;
            case 0xd5 :
               *s = 0x5b;
               break;
            case 0xe5 :
               *s = 0x5d;
               break;
            default :
               break;
         }
      }
   }
}


/*
 * Name:    strip_hi
 * Purpose: To strip bit 7 from characters
 * Date:    June 5, 1991
 * Passed:  s:    the starting point, which should be normalized to a segment
 *          count: number of characters to strip (size_t)
 *                 count should not be greater than MAX_LINE_LENGTH
 * Returns: none
 * Notes:   this function is useful on WordStar files.  to make a WordStar
 *           file readable, the hi bits need to be anded off.
 *          incidentally, soft CRs and LFs may be converted to hard CRs
 *           and LFs.  this function makes no attempt to split lines
 *           when soft returns are converted to hard returns.
 */
void strip_hi( text_ptr s, size_t count )
{
   if (s != NULL) {
      for (; count > 0; s++, count-- ) {
         if (*s >= 0x80)
            *s &= 0x7f;
      }
   }
}


/*
 * Name:    get_block_border_style
 * Class:   helper function
 * Purpose: ask for the characters to draw the border
 * Author:  Jason Hood
 * Date:    July 31, 1998
 * Passed:  window:  pointer to current window
 *          style:   pointer to eight-character buffer
 * Notes:   The style depends on length:
 *            no characters - use current graphic set
 *            one           - entire border is one character
 *            two           - first char is left/right edge, second top/bottom
 *            three         - corner, left/right, top/bottom
 *            four          - left, right, top, bottom (corners are left/right)
 *            five          - top-left/right, bottom-left/right and the edges
 *            six           - corners, left/right and top/bottom edges
 *            eight         - corners, left, right, top and bottom edges
 *          Anything else is an error.
 *          If graphic characters are off, turn them on.
 */
int  get_block_border_style( TDE_WIN *window, char *style )
{
static char prev_style[9] = { 0 };      /* Remember previous style */
char answer[MAX_COLS+2];
int  len;
char *gc;
int  old_gc;
int  prompt_line;
DISPLAY_BUFF;
int  rc;

   rc = OK;
   prompt_line = window->bottom_line;
   save_screen_line( 0, prompt_line, display_buff );
   strcpy( answer, prev_style );
   old_gc = g_status.graphic_chars;
   if (old_gc < 0)
      toggle_graphic_chars( NULL );

   /*
    * Box style:
    */
   rc = get_name( block27, prompt_line, answer );
   if (rc == OK) {
      len = strlen( answer );
      if (len == 7 || len > 8) {
         error( WARNING, prompt_line, block28 ); /* invalid style */
         rc = ERROR;
      } else {
         strcpy( prev_style, answer );
         switch( len ) {
            case 0:
               len = abs( g_status.graphic_chars );
               gc  = graphic_char[len-1];
               if (len == 5) {
                  memset( style, gc[5], 6 );
                  style[6] = gc[8];
                  style[7] = gc[2];
               } else {
                  style[0] = gc[7];
                  style[1] = gc[9];
                  style[2] = gc[1];
                  style[3] = gc[3];
                  style[4] =
                  style[5] = gc[0];
                  style[6] =
                  style[7] = gc[10];
               }
               break;
            case 1:
               memset( style, *answer, 8 );
               break;
            case 2:
               memset( style, *answer, 6 );
               style[6] = style[7] = answer[1];
               break;
            case 3:
               memset( style, *answer, 4 );
               style[4] = style[5] = answer[1];
               style[6] = style[7] = answer[2];
               break;
            case 4:
               style[0] = style[2] = style[4] = answer[0];
               style[1] = style[3] = style[5] = answer[1];
               style[6] = answer[2];
               style[7] = answer[3];
               break;
            case 5:
               memcpy( style, answer, 4 );
               memset( style+4, answer[4], 4 );
               break;
            case 6:
               memcpy( style, answer, 4 );
               style[4] = style[5] = answer[4];
               style[6] = style[7] = answer[5];
               break;
            case 8:
               memcpy( style, answer, 8 );
               break;
         }
      }
   }
   if (old_gc < 0 && g_status.graphic_chars > 0)
      toggle_graphic_chars( NULL );
   restore_screen_line( 0, prompt_line, display_buff );

   return( rc );
}


/*
 * Name:    justify_line_block
 * Purpose: To left/right/center justify each line in the block
 * Author:  Jason Hood
 * Date:    August 11, 1998
 * Passed:  window:       pointer to source window
 *          block_start:  pointer to first node in block
 *          br:           beginning line number in marked block
 *          er:           ending line number in marked block
 *          rc:           return code
 */
void justify_line_block( TDE_WIN *window, line_list_ptr block_start,
                         long br, long er, int *rc )
{
TDE_WIN w;
int (*justify)( TDE_WIN * );

   dup_window_info( &w, window );
   w.visible = FALSE;
   w.ll = block_start;
   justify = (g_status.command == BlockLeftJustify)  ? flush_left  :
             (g_status.command == BlockRightJustify) ? flush_right :
                                                       flush_center;

   for (w.rline = br; w.rline <= er && *rc == OK && !g_status.control_break;
        ++w.rline) {
      *rc = justify( &w );
      if (*rc == OK)
         *rc = un_copy_line( w.ll, &w, TRUE );
      w.ll = w.ll->next;
   }
}


/*
 * Name:    block_indent
 * Purpose: to indent and undent line and box blocks
 * Author:  Jason Hood
 * Date:    August 11, 1998
 * Passed:  window:  pointer to current window
 * Notes:   Block[IU]ndentN asks for a number and adds/removes that many spaces
 *           from the beginning of the block; Block[IU]ndent performs a tab /
 *           backspace on the first line of the block and carries that change
 *           through the remaining lines.
 *          If subsequent lines start before the first line, those characters
 *           will be removed. eg:
 *
 *                 statement1;
 *                 statement2;
 *              }
 *
 *           Undenting these three lines will end up deleting the brace.
 *          If there's no room for the indentation, simply ignore that line.
 *          Place lines that have something deleted in the undo buffer.
 *
 * jmh 981205: Corrected BlockUndent and tab bugs.
 */
int  block_indent( TDE_WIN *window )
{
TDE_WIN *sw, s_w;
file_infos *file;
int  type;
int  indent;
char answer[MAX_COLS+2];
line_list_ptr p;
line_list_ptr temp_ll;
text_ptr t;
char *s;
int  col;
static int level = 2;           /* remember previous indentation level */
int  add = 0;
int  rem;
int  bc;
long er;
long li;
int  rc;

   entab_linebuff( );
   rc = un_copy_line( window->ll, window, TRUE );
   check_block( );
   if (g_status.marked == FALSE || rc == ERROR)
      return( ERROR );

   file = g_status.marked_file;
   type = file->block_type;
   if (type == STREAM) {
      /*
       * cannot indent stream blocks
       */
      error( WARNING, window->bottom_line, block30 );
      return( ERROR );
   }

   p  = file->block_start;
   li = file->block_br;
   er = file->block_er;
   bc = (type == BOX) ? file->block_bc : 0;
   indent = (g_status.command == BlockIndentN ||
             g_status.command == BlockIndent);

   /*
    * Skip any initial blank lines.
    */
   if (type == LINE)
      for (; li <= er && is_line_blank( p->line, p->len ); p = p->next, ++li) ;
   else {
      col = bc;
      for (; li <= er; p = p->next, ++li) {
         if (mode.inflate_tabs)
            col = entab_adjust_rcol( p->line, p->len, bc );
         if (col < p->len && !is_line_blank( p->line + col, p->len - col ))
            break;
      }
   }
   if (li > er)
      return( ERROR );

   copy_line( p );
   detab_linebuff( );
   g_status.copied = FALSE;     /* in case of error */

   /*
    * Verify something to undent.
    */
   if (!indent && g_status.line_buff[bc] != ' ')
      return( ERROR );

   /*
    * find a window that points to the file with the marked block.
    */
   sw = g_status.window_list;
   for (; ptoul( sw->file_info ) != ptoul( file );)
      sw = sw->next;
   if (mode.do_backups == TRUE) {
      file->modified = TRUE;
      rc = backup_file( sw );
      if (rc == ERROR)
         return( ERROR );
   }

   /*
    * use a local window structure to do the dirty work.
    * set s_w to invisible so the un_copy_line function will
    *   not display the lines while doing block stuff.
    */
   dup_window_info( &s_w, sw );
   s_w.visible = FALSE;

   /*
    * Determine the level of indentation.
    */
   for (col = bc; g_status.line_buff[col] == ' '; ++col) ;

   if (g_status.command == BlockIndent || g_status.command == BlockUndent) {
      s_w.rcol = col;
      s_w.ll   = p;
      if (g_status.command == BlockIndent)
         add = (mode.smart_tab) ? next_smart_tab( &s_w ) :
                                  mode.ltab_size - (bc % mode.ltab_size);
      else {
         if (mode.indent) {
            for (temp_ll = file->block_start->prev; temp_ll != NULL;
                 temp_ll = temp_ll->prev) {
               t = temp_ll->line;
               rem = temp_ll->len;
               if (mode.inflate_tabs)
                  t = detab_a_line( t, &rem );
               if (bc < rem) {
                  add = first_non_blank( t + bc, rem - bc );
                  if (add < col - bc  &&  add != rem - bc) {
                     add = col - bc - add;
                     break;
                  }
               }
            }
            if (add == 0)
               add = col - bc;
         } else {
            add = (mode.smart_tab) ? prev_smart_tab( &s_w ) :
                                     bc % mode.ltab_size;
            if (add == 0)
               add = mode.ltab_size;
         }
      }
   } else {
      /*
       * indentation level:
       */
      my_ltoa( level, answer, 10 );
      rc = get_name( block31, window->bottom_line, answer );
      if (rc != ERROR)
         add = atoi( answer );
      if (rc == ERROR || add <= 0)
         return( ERROR );
      level = add;
   }
   if (!indent) {
      if (add > col - bc)
         add = col - bc;
   }

   /*
    * for each line in the marked block, do the necessary indentation.
    */
   g_status.copied = TRUE;      /* already know first line is copied */
   for (; rc == OK  &&  li <= er  &&  !g_status.control_break; li++) {
      copy_line( p );
      detab_linebuff( );

      if (bc < g_status.line_buff_len) {
         s   = g_status.line_buff + bc;
         rem = g_status.line_buff_len - bc;
         if (indent) {
            if (g_status.line_buff_len + add < MAX_LINE_LENGTH) {
               memmove( s + add, s, rem );
               memset( s, ' ', add );
               g_status.line_buff_len += add;
            }
         } else {
            for (col = bc-1 + ((add > rem) ? rem : add); col >= bc; --col)
               if (g_status.line_buff[col] != ' ')
                  break;
            if (col >= bc)
               load_undo_buffer( file, p->line, p->len );

            rem -= add;
            if (rem > 0) {
               memmove( s, s + add, rem );
               g_status.line_buff_len -= add;
            } else
               g_status.line_buff_len = bc;
         }
         if (mode.inflate_tabs)
            entab_linebuff( );
         rc = un_copy_line( p, &s_w, TRUE );
      } else
         g_status.copied = FALSE;

      p = p->next;
   }

   /*
    * IMPORTANT:  we need to reset the copied flag because the cursor may
    * not necessarily be on the last line of the block.
    */
   g_status.copied = FALSE;
   file->dirty = GLOBAL;
   show_avail_mem( );

   return( OK );
}
